% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dominance_analysis.R
\name{dominance_analysis}
\alias{dominance_analysis}
\title{Dominance Analysis}
\usage{
dominance_analysis(
  model,
  sets = NULL,
  all = NULL,
  conditional = TRUE,
  complete = TRUE,
  quote_args = NULL,
  ...
)
}
\arguments{
\item{model}{A model object supported by \code{performance::r2()}. See 'Details'.}

\item{sets}{A (named) list of formula objects with no left hand
side/response.  If the list has names, the name provided each element
will be used as the label for the set.  Unnamed list elements will be
provided a set number name based on its position among the sets as entered.

Predictors in each formula are bound together as a set in the dominance
analysis and dominance statistics and designations are computed for
the predictors together.  Predictors in \code{sets} must be present in the model
submitted to the \code{model} argument and cannot be in the \code{all} argument.}

\item{all}{A formula with no left hand side/response.

Predictors in the formula are included in each subset in the dominance
analysis and the R2 value associated with them is subtracted from the
overall value.  Predictors in \code{all} must be present in the model
submitted to the \code{model} argument and cannot be in the \code{sets} argument.}

\item{conditional}{Logical.  If \code{FALSE} then conditional dominance matrix is not computed.

If conditional dominance is not desired as an importance criterion, avoiding computing the conditional dominance matrix can save computation time.}

\item{complete}{Logical.  If \code{FALSE} then complete dominance matrix is not computed.

If complete dominance is not desired as an importance criterion, avoiding computing complete dominance designations can save computation time.}

\item{quote_args}{A character vector of arguments in the model submitted to
\code{model} to \code{quote()} prior to submitting to the dominance analysis.  This
is necessary for data masked arguments (e.g., \code{weights}) to prevent them
from being evaluated before being applied to the model and causing an error.}

\item{...}{Not used at current.}
}
\value{
Object of class \code{"parameters_da"}.

An object of class \code{"parameters_da"} is a list of \code{data.frame}s composed
of the following elements:
\describe{
\item{\code{General}}{A \code{data.frame} which associates dominance statistics with
model parameters. The variables in this \code{data.frame} include:
\describe{
\item{\code{Parameter}}{Parameter names.}
\item{\code{General_Dominance}}{Vector of general dominance statistics.
The R2 ascribed to variables in the \code{all} argument are also reported
here though they are not general dominance statistics.}
\item{\code{Percent}}{Vector of general dominance statistics normalized
to sum to 1.}
\item{\code{Ranks}}{Vector of ranks applied to the general dominance
statistics.}
\item{\code{Subset}}{Names of the subset to which the parameter belongs in
the dominance analysis.  Each other \code{data.frame} returned will refer
to these subset names.}}}
\item{\code{Conditional}}{A \code{data.frame} of conditional dominance
statistics.  Each observation represents a subset and each variable
represents an the average increment to R2 with a specific number of
subsets in the model.  \code{NULL} if \code{conditional} argument is \code{FALSE}.}
\item{\code{Complete}}{A \code{data.frame} of complete dominance
designations. The subsets in the observations are compared to the
subsets referenced in each variable. Whether the subset
in each variable dominates the subset in each observation is
represented in the  logical value. \code{NULL} if \code{complete}
argument is \code{FALSE}..}
}
}
\description{
Computes Dominance Analysis Statistics and Designations
}
\details{
Computes two decompositions of the model's R2 and returns
a matrix of designations from which predictor relative importance
determinations can be obtained.

Note in the output that the "constant" subset is associated with a
component of the model that does not directly contribute to the R2 such
as an intercept. The "all" subset is apportioned a component of the fit
statistic but is not considered a part of the dominance analysis and
therefore does not receive a rank, conditional dominance statistics, or
complete dominance designations.

The input model is parsed using \code{insight::find_predictors()}, does not
yet support interactions, transformations, or offsets applied in the
R formula, and will fail with an error if any such terms are detected.

The model submitted must accept an formula object as a \code{formula}
argument.  In addition, the model object must accept the data on which
the model is estimated as a \code{data} argument.  Formulas submitted
using object references (i.e., \code{lm(mtcars$mpg ~ mtcars$vs)}) and
functions that accept data as a non-\code{data} argument
(e.g., \code{survey::svyglm()} uses \code{design}) will fail with an error.

Models that return \code{TRUE} for the \code{insight::model_info()}
function's values "is_bayesian", "is_mixed", "is_gam",
is_multivariate", "is_zero_inflated",
or "is_hurdle" are not supported at current.

When \code{performance::r2()} returns multiple values, only the first is used
by default.

The underlying \code{domir::domin()} function that implements the dominance
statistic and designation computations has only been tested to R version
3.5 and will fail with an error if called in R versions < 3.5.
}
\examples{
if (getRversion() >= "3.5.0" && require("domir") &&
  require("performance")) {
  data(mtcars)

  # Dominance Analysis with Logit Regression
  model <- glm(vs ~ cyl + carb + mpg, data = mtcars, family = binomial())

  performance::r2(model)
  dominance_analysis(model)

  # Dominance Analysis with Weighted Logit Regression
  model_wt <- glm(vs ~ cyl + carb + mpg,
    data = mtcars,
    weights = wt, family = quasibinomial()
  )

  dominance_analysis(model_wt, quote_args = "weights")
}
}
\references{
\itemize{
\item Azen, R., & Budescu, D. V. (2003). The dominance analysis approach
for comparing predictors in multiple regression. Psychological Methods,
8(2), 129-148. doi:10.1037/1082-989X.8.2.129
\item Budescu, D. V. (1993). Dominance analysis: A new approach to the
problem of relative importance of predictors in multiple regression.
Psychological Bulletin, 114(3), 542-551. doi:10.1037/0033-2909.114.3.542
\item Groemping, U. (2007). Estimators of relative importance in linear
regression based on variance decomposition. The American Statistician,
61(2), 139-147. doi:10.1198/000313007X188252
}
}
\seealso{
\code{\link[domir:domin]{domir::domin()}}
}
\author{
Joseph Luchman
}
