# within-subjects design - NAs --------------------------------------------------

testthat::test_that(
  desc = "`pairwise_comparisons()` works for within-subjects design - NAs",
  code = {

    # student's t test
    set.seed(123)
    df1 <-
      pairwiseComparisons::pairwise_comparisons(
        data = pairwiseComparisons::bugs_long,
        x = "condition",
        y = desire,
        type = "p",
        k = 3,
        paired = TRUE,
        p.adjust.method = "bonferroni"
      )

    # Durbin-Conover test
    set.seed(123)
    df2 <-
      pairwiseComparisons::pairwise_comparisons(
        data = pairwiseComparisons::bugs_long,
        x = condition,
        y = "desire",
        type = "np",
        k = 3,
        paired = TRUE,
        p.adjust.method = "BY"
      )

    # robust t test
    set.seed(123)
    df3 <-
      pairwiseComparisons::pairwise_comparisons(
        data = pairwiseComparisons::bugs_long,
        x = condition,
        y = desire,
        type = "r",
        k = 3,
        paired = TRUE,
        p.adjust.method = "hommel"
      )

    # bf
    df4 <-
      pairwiseComparisons::pairwise_comparisons(
        data = bugs_long,
        x = condition,
        y = desire,
        type = "bf",
        k = 4,
        paired = TRUE
      )

    # test details
    testthat::expect_identical(unique(df1$test.details), "Student's t-test")
    testthat::expect_identical(unique(df2$test.details), "Durbin-Conover test")
    testthat::expect_identical(unique(df3$test.details), "Yuen's trimmed means test")
    testthat::expect_identical(unique(df4$test.details), "Student's t-test")

    # adjustment method
    testthat::expect_identical(unique(df1$p.value.adjustment), "Bonferroni")
    testthat::expect_identical(unique(df2$p.value.adjustment), "BY")
    testthat::expect_identical(unique(df3$p.value.adjustment), "Hommel")

    # checking exact values
    testthat::expect_identical(
      df1$label,
      c(
        "list(~italic(p)[Bonferroni-corrected]==0.003)",
        "list(~italic(p)[Bonferroni-corrected]==0.421)",
        "list(~italic(p)[Bonferroni-corrected]==3.95e-12)",
        "list(~italic(p)[Bonferroni-corrected]==0.337)",
        "list(~italic(p)[Bonferroni-corrected]==0.008)",
        "list(~italic(p)[Bonferroni-corrected]==1.33e-08)"
      )
    )

    testthat::expect_equal(
      df2$statistic,
      c(
        4.78004208516409,
        2.44393129166284,
        8.01465703001196,
        2.33611079350124,
        3.23461494484788,
        5.57072573834912
      ),
      tolerance = 0.001
    )

    testthat::expect_identical(
      df2$label,
      c(
        "list(~italic(p)[BY-corrected]==1.44e-05)",
        "list(~italic(p)[BY-corrected]==0.045)",
        "list(~italic(p)[BY-corrected]==5.45e-13)",
        "list(~italic(p)[BY-corrected]==0.050)",
        "list(~italic(p)[BY-corrected]==0.005)",
        "list(~italic(p)[BY-corrected]==4.64e-07)"
      )
    )

    testthat::expect_equal(
      df3$estimate,
      c(
        1.15972222222222,
        0.5,
        2.09722222222222,
        -0.701388888888889,
        0.9375,
        1.54166666666667
      ),
      tolerance = 0.001
    )

    testthat::expect_identical(
      df3$label,
      c(
        "list(~italic(p)[Hommel-corrected]==0.001)",
        "list(~italic(p)[Hommel-corrected]==0.062)",
        "list(~italic(p)[Hommel-corrected]==1.79e-10)",
        "list(~italic(p)[Hommel-corrected]==0.062)",
        "list(~italic(p)[Hommel-corrected]==0.014)",
        "list(~italic(p)[Hommel-corrected]==1.16e-06)"
      )
    )

    testthat::expect_identical(
      df4$label,
      c(
        "list(~log[e](BF['01'])==-3.7273)",
        "list(~log[e](BF['01'])==0.5394)",
        "list(~log[e](BF['01'])==-23.2071)",
        "list(~log[e](BF['01'])==0.3589)",
        "list(~log[e](BF['01'])==-2.8966)",
        "list(~log[e](BF['01'])==-15.3854)"
      )
    )

    # checking if it is a tibble
    testthat::expect_s3_class(df1, "tbl_df")
    testthat::expect_s3_class(df2, "tbl_df")
    testthat::expect_s3_class(df3, "tbl_df")
    testthat::expect_s3_class(df4, "tbl_df")

    # columns should be same no matter the test
    testthat::expect_identical(df1$group1, df2$group1)
    testthat::expect_identical(df1$group1, df3$group1)
    testthat::expect_identical(df1$group1, df4$group1)
    testthat::expect_identical(df1$group2, df2$group2)
    testthat::expect_identical(df1$group2, df3$group2)
    testthat::expect_identical(df1$group2, df4$group2)

    # column names
    testthat::expect_identical(names(df1)[1:2], c("group1", "group2"))
    testthat::expect_identical(names(df2)[1:2], c("group1", "group2"))
    testthat::expect_identical(names(df3)[1:2], c("group1", "group2"))
    testthat::expect_identical(names(df4)[1:2], c("group1", "group2"))
  }
)



# within-subjects design - no NAs ---------------------------------------------

testthat::test_that(
  desc = "`pairwise_comparisons()` works for within-subjects design - NAs",
  code = {

    # student's t test
    set.seed(123)
    df1 <-
      pairwiseComparisons::pairwise_comparisons(
        data = WRS2::WineTasting,
        x = "Wine",
        y = Taste,
        type = "p",
        k = 3,
        paired = TRUE,
        p.adjust.method = "none"
      )

    # Durbin-Conover test
    set.seed(123)
    df2 <-
      pairwiseComparisons::pairwise_comparisons(
        data = WRS2::WineTasting,
        x = Wine,
        y = "Taste",
        type = "np",
        k = 3,
        paired = TRUE,
        p.adjust.method = "none"
      )

    # robust t test
    set.seed(123)
    df3 <-
      pairwiseComparisons::pairwise_comparisons(
        data = WRS2::WineTasting,
        x = Wine,
        y = Taste,
        type = "r",
        k = 3,
        paired = TRUE,
        p.adjust.method = "none"
      )

    # bf
    df4 <-
      pairwiseComparisons::pairwise_comparisons(
        data = WRS2::WineTasting,
        x = Wine,
        y = Taste,
        type = "bf",
        k = 4,
        paired = TRUE
      )

    # test details
    testthat::expect_identical(unique(df1$test.details), "Student's t-test")
    testthat::expect_identical(unique(df2$test.details), "Durbin-Conover test")
    testthat::expect_identical(unique(df3$test.details), "Yuen's trimmed means test")
    testthat::expect_identical(unique(df4$test.details), "Student's t-test")

    # adjustment method
    testthat::expect_identical(unique(df1$p.value.adjustment), "None")
    testthat::expect_identical(unique(df2$p.value.adjustment), "None")
    testthat::expect_identical(unique(df3$p.value.adjustment), "None")

    # checking exact values
    testthat::expect_identical(
      df1$label,
      c(
        "list(~italic(p)[uncorrected]==0.732)",
        "list(~italic(p)[uncorrected]==0.014)",
        "list(~italic(p)[uncorrected]==0.001)"
      )
    )

    testthat::expect_equal(
      df2$statistic,
      c(1.04673405118638, 3.66356917915232, 2.61683512796594),
      tolerance = 0.001
    )

    testthat::expect_identical(
      df2$label,
      c(
        "list(~italic(p)[uncorrected]==0.301)",
        "list(~italic(p)[uncorrected]==0.001)",
        "list(~italic(p)[uncorrected]==0.012)"
      )
    )

    testthat::expect_equal(
      df3$estimate,
      c(0.0166666666666668, 0.1, 0.0777777777777778),
      tolerance = 0.001
    )

    testthat::expect_identical(
      df3$label,
      c(
        "list(~italic(p)[uncorrected]==0.380)",
        "list(~italic(p)[uncorrected]==0.011)",
        "list(~italic(p)[uncorrected]==0.003)"
      )
    )

    testthat::expect_identical(
      df4$label,
      c(
        "list(~log[e](BF['01'])==1.4462)",
        "list(~log[e](BF['01'])==-1.3122)",
        "list(~log[e](BF['01'])==-3.9214)"
      )
    )

    # columns should be same no matter the test
    testthat::expect_identical(df1$group1, df2$group1)
    testthat::expect_identical(df1$group1, df3$group1)
    testthat::expect_identical(df1$group1, df4$group1)
    testthat::expect_identical(df1$group2, df2$group2)
    testthat::expect_identical(df1$group2, df3$group2)
    testthat::expect_identical(df1$group2, df4$group2)
  }
)

# works with subject id ------------------------------------------------------

testthat::test_that(
  desc = "works with subject id",
  code = {
    testthat::skip_if(getRversion() < "3.6")

    set.seed(123)
    df1 <-
      dplyr::bind_rows(
        pairwiseComparisons::pairwise_comparisons(
          data = WRS2::WineTasting,
          x = Wine,
          y = "Taste",
          type = "p",
          k = 3,
          subject.id = "Taster",
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = WRS2::WineTasting,
          x = Wine,
          y = "Taste",
          type = "np",
          k = 3,
          subject.id = "Taster",
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = WRS2::WineTasting,
          x = Wine,
          y = "Taste",
          type = "r",
          k = 3,
          subject.id = "Taster",
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = WRS2::WineTasting,
          x = Wine,
          y = "Taste",
          type = "bf",
          k = 3,
          subject.id = "Taster",
          paired = TRUE
        )
      )

    set.seed(123)
    df2 <-
      dplyr::bind_rows(
        pairwiseComparisons::pairwise_comparisons(
          data = dplyr::arrange(WRS2::WineTasting, Taster),
          x = Wine,
          y = "Taste",
          type = "p",
          k = 3,
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = dplyr::arrange(WRS2::WineTasting, Taster),
          x = Wine,
          y = "Taste",
          type = "np",
          k = 3,
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = dplyr::arrange(WRS2::WineTasting, Taster),
          x = Wine,
          y = "Taste",
          type = "r",
          k = 3,
          paired = TRUE
        ),
        pairwiseComparisons::pairwise_comparisons(
          data = dplyr::arrange(WRS2::WineTasting, Taster),
          x = Wine,
          y = "Taste",
          type = "bf",
          k = 3,
          paired = TRUE
        )
      )

    # columns should be same no matter the test
    testthat::expect_equal(as.data.frame(df1), as.data.frame(df2))
  }
)
