% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_regression.R
\name{fit_regression}
\alias{fit_regression}
\title{Fits a 3D Spherical Regression.}
\usage{
fit_regression(
  evaluation_points,
  explanatory_points,
  response_points,
  concentration,
  weights_generator = weight_explanatory_points,
  number_of_expansion_terms = 1,
  number_of_iterations = 1,
  allow_reflections = FALSE
)
}
\arguments{
\item{evaluation_points}{An \emph{n}-by-3 matrix whose rows contain
the Cartesian coordinates of the points at which the regression
will be estimated.}

\item{explanatory_points}{An \emph{m}-by-3 matrix whose rows contain
the Cartesian coordinates of the explanatory points used to
calculate the regression estimators.}

\item{response_points}{An \emph{m}-by-\emph{3} matrix whose rows contain
the Cartesian coordinates of the response points corresponding
to the explanatory points.}

\item{concentration}{A non negative scalar whose reciprocal value
is proportional to the bandwidth applied while estimating
a spherical regression model.}

\item{weights_generator}{A function that, given a matrix of \emph{n}
evaluation points, returns an \emph{m}-by-\emph{n} matrix whose
\emph{j}-th column contains
the weights assigned to the explanatory points while analyzing the
\emph{j}-th evaluation point. Defaults to \code{\link{weight_explanatory_points}}.}

\item{number_of_expansion_terms}{The number of terms to be included
in the expansion of the matrix exponential applied while
approximating a local rotation matrix. Must be \code{1} or \code{2}.
Defaults to \code{1}.}

\item{number_of_iterations}{The number of
rotation fitting steps to be executed.
At each step, the points estimated during the previous step
are exploited as the current explanatory points. Defaults to \code{1}.}

\item{allow_reflections}{A logical scalar value. If set to \code{TRUE} signals
that reflections are allowed. Defaults to \code{FALSE}. It is ignored if
\code{number_of_expansion_terms} is \code{2}.}
}
\value{
A \code{number_of_iterations}-length vector of lists, with the \code{s}-th
list having two components,
\code{fitted_response_points}, an \emph{n}-by-\emph{3} matrix whose rows contain
the Cartesian coordinates of the fitted points at iteration \code{s}, and
\code{explanatory_points}, an \emph{m}-by-\emph{3} matrix whose rows contain
the Cartesian coordinates of the points exploited as explanatory at
iteration \code{s}.
}
\description{
Returns 3D spherical points obtained by locally rotating
the specified evaluation
points, given an approximated model for local rotations
and a weighting scheme for the observed data set.
This function implements the method for sphere-sphere regression
proposed by Di Marzio et al. (2018).
}
\details{
Function \code{weights_generator} must be prototyped as having the
following three arguments:
\describe{
 \item{\code{evaluation_points}}{a matrix whose \emph{n} rows are the Cartesian coordinates of
given evaluation points.}
 \item{\code{explanatory_points}}{a matrix whose \emph{m} rows are the Cartesian coordinates of
given explanatory points.}
 \item{\code{concentration}}{A non negative scalar whose reciprocal value
is proportional to the bandwidth applied while estimating
a spherical regression model.}
}
It is also expected that \code{weights_generator} will return
a non \code{NULL} numerical \emph{m}-by-\emph{n} matrix whose \emph{j}-th column contains
the weights assigned to the explanatory points while analyzing the
\emph{j}-th evaluation point.
}
\examples{
library(nprotreg)

# Create 100 equally spaced design points on the sphere.

number_of_explanatory_points <- 100

explanatory_points <- get_equally_spaced_points(
  number_of_explanatory_points
)

# Define the regression model, where the rotation for a given "point"
# is obtained from the exponential of a skew-symmetric matrix with the
# following components.

local_rotation_composer <- function(point) {
  independent_components <- (1 / 8) *
    c(exp(2.0 * point[3]), - exp(2.0 * point[2]), exp(2.0 * point[1]))
}

# Define an error term given by a small rotation, similarly defined
# from a skew-symmetric matrix with random entries.

local_error_sampler <- function(point) {
  rnorm(3, sd = .01)
}

# Generate the matrix of responses, using the regression model
# and the error model.

response_points <- simulate_regression(
  explanatory_points,
  local_rotation_composer,
  local_error_sampler
)

# Create some "test data" for which the response will be predicted.

evaluation_points <- rbind(
  cbind(.5, 0, .8660254),
  cbind(-.5, 0, .8660254),
  cbind(1, 0, 0),
  cbind(0, 1, 0),
  cbind(-1, 0, 0),
  cbind(0, -1, 0),
  cbind(.5, 0, -.8660254),
  cbind(-.5, 0, -.8660254)
)

# Define a weight function for nonparametric fit.

weights_generator <- weight_explanatory_points

# Set the concentration parameter.

concentration <- 5

# Or obtain this by cross-validation: see
# the `cross_validate_concentration` function.

# Fit regression.

fitted_model <- fit_regression(
  evaluation_points,
  explanatory_points,
  response_points,
  concentration,
  weights_generator,
  number_of_expansion_terms = 1,
  number_of_iterations = 2
)

# Extract the point corresponding to the
# second evaluation point fitted at
# the first iteration.

cat("Point fitted at iteration 1 corresponding to the second evaluation point: \n")
cat(fitted_model[[1]]$fitted_response_points[2, ], "\n")

\dontrun{
# Create some plots to view the results.

# 3D plot.

library(rgl)

plot3d(
  explanatory_points,
  type = "n",
  xlab = "x",
  ylab = "y",
  zlab = "z",
  box = TRUE,
  axes = TRUE
)
spheres3d(0, 0, 0, radius = 1, lit = FALSE, color = "white")
spheres3d(0, 0, 0, radius = 1.01, lit = FALSE, color = "black", front = "culled")
text3d(c(0, 0, 1), text = "N", adj = 0)

ll <- 10
vv1 <- (ll - (0:(ll))) / ll
vv2 <- 1 - vv1
plot3d(explanatory_points, add = TRUE, col = 2)
for (i in 1:dim(explanatory_points)[1]) {
  m <- outer(vv1, explanatory_points[i,], "*") +
    outer(vv2, response_points[i,], "*")
  m <- m / sqrt(apply(m ^ 2, 1, sum))
  lines3d(m, col = 3)
}

plot3d(evaluation_points, add = TRUE, col = 4)

for (i in 1:dim(evaluation_points)[1]) {
  m <- outer(vv1, evaluation_points[i,], "*") +
    outer(vv2, fitted_model[[1]]$fitted_response_points[i,], "*")
  m <- m / sqrt(apply(m ^ 2, 1, sum))
  lines3d(m, col = 1)
}

# 2D plot.

explanatory_spherical_coords <- convert_cartesian_to_spherical(explanatory_points)
response_spherical_coords <- convert_cartesian_to_spherical(response_points)

plot(
  x = explanatory_spherical_coords[, 1],
  y = explanatory_spherical_coords[, 2],
  pch = 20,
  cex = .7,
  col = 2,
  xlab = "longitude",
  ylab = "latitude"
)

for (i in 1:dim(explanatory_spherical_coords)[1]) {
  column <- 1
  if ((explanatory_spherical_coords[i, 1] - response_spherical_coords[i, 1]) ^ 2 +
      (explanatory_spherical_coords[i, 2] - response_spherical_coords[i, 2]) ^ 2 > 4)
        column <- "grey"
  lines(
    c(explanatory_spherical_coords[i, 1], response_spherical_coords[i, 1]),
    c(explanatory_spherical_coords[i, 2], response_spherical_coords[i, 2]),
    col = column
  )
}

evaluation_spherical_coords <- convert_cartesian_to_spherical(
  evaluation_points
)

fitted_response_spherical_coords <- convert_cartesian_to_spherical(
  fitted_model[[1]]$fitted_response_points
)

points(
  x = evaluation_spherical_coords[, 1],
  y = evaluation_spherical_coords[, 2],
  pch = 20,
  cex = .7,
  col = 4
)

for (i in 1:dim(evaluation_spherical_coords)[1]) {
  column <- 3
  if ((evaluation_spherical_coords[i, 1] - fitted_response_spherical_coords[i, 1]) ^ 2 +
      (evaluation_spherical_coords[i, 2] - fitted_response_spherical_coords[i, 2]) ^ 2 > 4)
        column <- "grey"
  lines(
    c(evaluation_spherical_coords[i, 1], fitted_response_spherical_coords[i, 1]),
    c(evaluation_spherical_coords[i, 2], fitted_response_spherical_coords[i, 2]),
    col = column
  )
}

} 
}
\references{
Marco Di Marzio, Agnese Panzera & Charles C. Taylor (2018)
Nonparametric rotations for sphere-sphere regression,
Journal of the American Statistical Association,
DOI: \href{https://doi.org/10.1080/01621459.2017.1421542}{10.1080/01621459.2017.1421542}
}
\seealso{
Other Regression functions: 
\code{\link{cross_validate_concentration}()},
\code{\link{get_equally_spaced_points}()},
\code{\link{get_skew_symmetric_matrix}()},
\code{\link{simulate_regression}()},
\code{\link{simulate_rigid_regression}()},
\code{\link{weight_explanatory_points}()}
}
\concept{Regression functions}
