% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prevplot.R
\name{prevplot}
\alias{prevplot}
\title{Plot observed and expected prevalences for a multi-state model}
\usage{
prevplot(x, prev.obj, exacttimes = TRUE, M = FALSE, ci = FALSE)
}
\arguments{
\item{x}{A \code{msm} object.}

\item{prev.obj}{A list computed by \code{\link[msm]{prevalence.msm}}.
It can be with or without confidence intervals. \code{prevplot} will behaves
accordingly.}

\item{exacttimes}{If \code{TRUE} (default) then transition times are known
and exact. This is inherited from \code{msm} and should be set the same way.}

\item{M}{If \code{TRUE}, then a rough indicator of deviance from the model is
computed (see 'Details'). Default is \code{FALSE}.}

\item{ci}{If \code{TRUE}, then confidence intervals, if they exist, are plotted.
Default is \code{FALSE}.}
}
\description{
Provides a graphical indication of goodness of fit of a multi-state model
computed by \code{\link[msm]{msm}} using observed and expected prevalences.
It also computes a rough indicator of where the data depart from the estimated
Markov model.
}
\details{
When \code{M = TRUE}, a rough indicator of the deviance from the
Markov model is computed according to Titman and Sharples (2008).
A comparison at a given time \eqn{t_i} of a patient \emph{k} in the state
\emph{s} between observed counts \eqn{O_{is}} with expected ones \eqn{E_{is}}
is build as follows:
\deqn{M_{is} = \frac{(O_{is} - E_{is})^2}{E_{is}}}{ (O_{is} - E_{is})^2 / E_{is}}

The plot of the deviance M is returned together with the standard prevalence plot in the second
row. This is not editable by the user.
}
\examples{
\dontrun{
data( hosp )

# augmenting the data
hosp_augmented = augment( data = hosp, data_key = subj, n_events = adm_number,
                          pattern = label_3, t_start = dateIN, t_end = dateOUT,
                          t_cens = dateCENS )

# let's define the initial transition matrix for our model
Qmat = matrix( data = 0, nrow = 3, ncol = 3, byrow = TRUE )
Qmat[ 1, 1:3 ] = 1
Qmat[ 2, 1:3 ] = 1
colnames( Qmat ) = c( 'IN', 'OUT', 'DEAD' )
rownames( Qmat ) = c( 'IN', 'OUT', 'DEAD' )

# attaching the msm package and running the model using
# gender and age as covariates
library( msm )
msm_model = msm( status_num ~ augmented_int, subject = subj,
                 data = hosp_augmented, covariates = ~ gender + age,
                 exacttimes = TRUE, gen.inits = TRUE, qmatrix = Qmat,
                 method = 'BFGS', control = list( fnscale = 6e+05, trace = 0,
                 REPORT = 1, maxit = 10000 ) )

# defining the times at which compute the prevalences
t_min = min( hosp_augmented$augmented_int )
t_max = max( hosp_augmented$augmented_int )
steps = 100L

# computing prevalences
prev = prevalence.msm( msm_model, covariates = 'mean', ci = 'normal',
                       times = seq( t_min, t_max, steps ) )

# and plotting them using prevplot()
gof = prevplot( x = msm_model, prev.obj = prev, ci = TRUE, M = TRUE )
}

}
\references{
Titman, A. and Sharples, L.D. (2010). Model diagnostics for
multi-state models, \emph{Statistical Methods in Medical Research}, 19,
621-651.\cr

Titman, A. and Sharples, L.D. (2008). A general goodness-of-fit test for
Markov and hidden Markov models, \emph{Statistics in Medicine}, 27,
2177-2195. \cr

Gentleman RC, Lawless JF, Lindsey JC, Yan P. (1994). Multi-state Markov
models for analysing incomplete disease data with illustrations for HIV
disease. \emph{Statistics in Medicine}, 13:805-821. \cr

Jackson, C.H. (2011). Multi-State Models for Panel Data:\cr
The \emph{msm} Package for R. Journal of Statistical Software, 38(8), 1-29.\cr
URL \url{https://www.jstatsoft.org/v38/i08/}.
}
\seealso{
\code{\link[msm]{plot.prevalence.msm}} \code{\link[msm]{msm}}
\code{\link[msm]{prevalence.msm}}
}
\author{
Francesco Grossetti \email{francesco.grossetti@unibocconi.it}.
}
