% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nested-modeltime_data_prep.R
\name{prep_nested}
\alias{prep_nested}
\alias{extend_timeseries}
\alias{nest_timeseries}
\alias{split_nested_timeseries}
\alias{extract_nested_train_split}
\alias{extract_nested_test_split}
\title{Prepared Nested Modeltime Data}
\usage{
extend_timeseries(.data, .id_var, .date_var, .length_future, ...)

nest_timeseries(.data, .id_var, .length_future, .length_actual = NULL)

split_nested_timeseries(.data, .length_test, .length_train = NULL, ...)

extract_nested_train_split(.data, .row_id = 1)

extract_nested_test_split(.data, .row_id = 1)
}
\arguments{
\item{.data}{A data frame or tibble containing time series data. The data should have:
\itemize{
\item identifier (.id_var): Identifying one or more time series groups
\item date variable (.date_var): A date or date time column
\item target variable (.value): A column containing numeric values that is to be forecasted
}}

\item{.id_var}{An id column}

\item{.date_var}{A date or datetime column}

\item{.length_future}{Varies based on the function:
\itemize{
\item \code{extend_timeseries()}: Defines how far into the future to extend the
time series by each time series group.
\item \code{nest_timeseries()}: Defines which observations should be split into the \code{.future_data}.
}}

\item{...}{Additional arguments passed to the helper function. See details.}

\item{.length_actual}{Can be used to slice the \code{.actual_data} to a most recent number of observations.}

\item{.length_test}{Defines the length of the test split for evaluation.}

\item{.length_train}{Defines the length of the training split for evaluation.}

\item{.row_id}{The row number to extract from the nested data.}
}
\description{
A set of functions to simplify preparation of nested data for
iterative (nested) forecasting with Nested Modeltime Tables.
}
\details{
Preparation of nested time series follows a 3-Step Process:
\subsection{Step 1: Extend the Time Series}{

\code{extend_timeseries()}: A wrapper for \code{\link[timetk:future_frame]{timetk::future_frame()}} that extends a time series
group-wise into the future.
\itemize{
\item The group column is specified by \code{.id_var}.
\item The date column is specified by \code{.date_var}.
\item The length into the future is specified with \code{.length_future}.
\item The \code{...} are additional parameters that can be passed to \code{\link[timetk:future_frame]{timetk::future_frame()}}
}
}

\subsection{Step 2: Nest the Time Series}{

\code{nest_timeseries()}: A helper for nesting your data into \code{.actual_data} and \code{.future_data}.
\itemize{
\item The group column is specified by \code{.id_var}
\item The \code{.length_future} defines the length of the \code{.future_data}.
\item The remaining data is converted to the \code{.actual_data}.
\item The \code{.length_actual} can be used to slice the \code{.actual_data} to a most recent number of observations.
}

The result is a "nested data frame".
}

\subsection{Step 3: Split the Actual Data into Train/Test Splits}{

\code{split_nested_timeseries()}: A wrapper for \code{\link[timetk:time_series_split]{timetk::time_series_split()}} that generates
training/testing splits from the \code{.actual_data} column.
\itemize{
\item The \code{.length_test} is the primary argument that identifies the size of the
testing sample. This is typically the same size as the \code{.future_data}.
\item The \code{.length_train} is an optional size of the training data.
\item The \code{...} (dots) are additional arguments that can be passed to \code{\link[timetk:time_series_split]{timetk::time_series_split()}}.
}
}

\subsection{Helpers}{

\code{extract_nested_train_split()} and \code{extract_nested_test_split()} are used to simplify extracting
the training and testing data from the actual data. This can be helpful when making
preprocessing recipes using the \code{recipes} package.
}
}
\examples{

library(tidyverse)
library(timetk)
library(modeltime)


nested_data_tbl <- walmart_sales_weekly \%>\%
    select(id, Date, Weekly_Sales) \%>\%
    set_names(c("id", "date", "value")) \%>\%

    # Step 1: Extends the time series by id
    extend_timeseries(
        .id_var     = id,
        .date_var   = date,
        .length_future = 52
    ) \%>\%

    # Step 2: Nests the time series into .actual_data and .future_data
    nest_timeseries(
        .id_var     = id,
        .length_future = 52
    ) \%>\%

    # Step 3: Adds a column .splits that contains training/testing indicies
    split_nested_timeseries(
        .length_test = 52
    )

nested_data_tbl

# Helpers: Getting the Train/Test Sets
extract_nested_train_split(nested_data_tbl, .row_id = 1)

}
