% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlme_curve.R
\name{nlmeCurve}
\alias{nlmeCurve}
\title{Plot individual MNREAD fitted curves as estimated by a nonlinear mixed-effect (NLME) modeling.}
\usage{
nlmeCurve(nlme.model, displayCPS = TRUE, CPScriterion = NULL)
}
\arguments{
\item{nlme.model}{The object returned by \code{\link{nlmeModel}}}

\item{displayCPS}{Optional argument to display the CPS on the individuals curves. Default is TRUE. If set to FALSE, the CPS won't be plotted.}

\item{CPScriterion}{Optional argument to specify a criterion for CPS estimation. The default criterion value is '90 of MRS'. This criterion can vary from 75 to 95 of MRS and should only be modified for specific purposes, as discussed in Cheung et al. 2008}
}
\value{
The function returns a plot of reading speed (in log words/min) as a function of print size (in logMAR). 
If displayCPS is not specified, the Critical Print Size will be marked as an inverted triangle.
}
\description{
This function uses the NLME model created from \code{\link{nlmeModel}} to plot individual curves and Critical Print Size (CPS).
}
\section{Notes}{

Print size shown on the plot(s) have been corrected for non-standard testing viewing distance.

For more details on the nlme fit, see:\\
Cheung SH, Kallie CS, Legge GE, Cheong AM. Nonlinear mixed-effects modeling of MNREAD data. 
Invest Ophthalmol Vis Sci. 2008;49:828–835. doi: 10.1167/iovs.07-0555.
}

\examples{
# inspect the structure of the dataframe
head(data_low_vision, 10)

#------

# restrict dataset to one MNREAD test per subject (regular polarity only)
data_regular <- data_low_vision \%>\%
    filter (polarity == "regular")

# run the NLME model for data grouped by subject
\dontrun{ nlme_model <- nlmeModel(data_regular, ps, vd, rt, err, subject) }

#------

# plot MNREAD curves and CPS with a default CPS criterion of '90 of MRS' 
\dontrun{ nlmeCurve(nlme_model) }

# plot MNREAD curves without the CPS for a default CPS criterion of '90 of MRS' 
\dontrun{ nlmeCurve(nlme_model, FALSE) }

# plot MNREAD curves and CPS with a specific CPS criterion of '80 of MRS'
\dontrun{ nlmeCurve(nlme_model, TRUE, 0.8) }


#------

# Once created, the NLME curve can be further customized using ggplot2 

# plot the NLME curve 
\dontrun{ my_plot <- nlmeCurve(nlme_model) }

# display my.plot
\dontrun{ print(my_plot) }

# modify my.plot
\dontrun{ my_new_plot <- my_plot + 
 # overwrites the raw data points
    geom_point(data = nlme_model[[1]], aes(x=correct_ps, y = rs), size = 4) + 
 # changes the colors of the curve and raw data (effective only for nested designs)
    scale_color_brewer(palette="Set1") + 
 # changes the colors of the CPS diamond (effective only for nested designs)
    scale_fill_brewer(palette="Set1")  + 
 # modifies the aspect of the x-axis
    scale_x_continuous(breaks = seq (-0.5,2.5,0.4))  }

# display my.new.plot                                                                        
\dontrun{ print(my_new_plot) }


#------

# For very large datasets, it can be usefull to plot only selected facets to inspect individual fit
# To do so, one needs to restrict the dataframe called in each of the three layers of the plot

# list of subject names to keep
subjects_to_keep <- paste ("s", 1:4, sep = "")

# first filter the original data points (data called in the first layer)
\dontrun{ my_plot$data <- my_plot$data \%>\%
    filter(subject \%in\% subjects_to_keep) \%>\%
    droplevels() }
    
# then filter the fitted data points (data called in the second layer)
\dontrun{ my_plot$layers[[2]]$data <- my_plot$layers[[2]]$data \%>\% 
    filter(subject \%in\% subjects_to_keep) \%>\% 
    droplevels() }
  
# and finally, if 'displayCPS' was set to TRUE, filter the data used to display the CPS
\dontrun{ my_plot$layers[[4]]$data <- my_plot$layers[[4]]$data \%>\% 
    filter(subject \%in\% subjects_to_keep) \%>\%
    droplevels() }
    
# plot the restricted my.plot
\dontrun{ my_plot }



}
\seealso{
\code{\link{nlmeModel}} to fit MNREAD data using a nonlinear mixed-effect (NLME) modeling

\code{\link{nlmeParam}} to estimate Maximum Reading Speed (MRS) and Critical Print Size (CPS) from the NLME model
 
\code{\link{mnreadCurve}} for standard MNREAD curve
}
