% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blasting.R
\name{blastpAllAll}
\alias{blastpAllAll}
\title{Making BLAST search all against all genomes}
\usage{
blastpAllAll(
  prot.files,
  out.folder,
  e.value = 1,
  job = 1,
  threads = 1,
  verbose = TRUE
)
}
\arguments{
\item{prot.files}{A vector with the names of the FASTA files where the protein sequences of
each genome is found.}

\item{out.folder}{The name of the folder where the result files should end up.}

\item{e.value}{The chosen E-value threshold in BLAST. Default is \samp{e.value = 1}, a smaller value
will speed up the search at the cost of less sensitivity.}

\item{job}{An integer to separate multiple jobs. You may want to run several jobs in parallell,
and each job should have different number here to avoid confusion on databases.}

\item{threads}{The number of CPU's to use.}

\item{verbose}{Logical, if \code{TRUE} some text output is produced to monitor the progress.}
}
\value{
The function produces a result file for each pair of files listed in \samp{prot.files}.
These result files are located in \code{out.folder}. Existing files are never overwritten by
\code{\link{blastpAllAll}}, if you want to re-compute something, delete the corresponding result files first.
}
\description{
Runs a reciprocal all-against-all BLAST search to look for similarity of proteins
within and across genomes.
}
\details{
A basic step in pangenomics and many other comparative studies is to cluster proteins into
groups or families. One commonly used approach is based on BLASTing. This function uses the
\samp{blast+} software available for free from NCBI (Camacho et al, 2009). 

A vector listing FASTA files of protein sequences is given as input in \samp{prot.files}. These files
must have the genome_id in the first token of every header, and in their filenames as well, i.e. all input
files should first be prepared by \code{\link{panPrep}} to ensure this. Note that only protein sequences
are considered here. If your coding genes are stored as DNA, please translate them to protein prior to
using this function, see \code{\link[microseq]{translate}}.

In the first version of this package we used reciprocal BLASTing, i.e. we computed both genome A against
B and B against A. This may sometimes produce slightly different results, but in reality this is too
costly compared to its gain, and we now only make one of the above searches. This basically halves the
number of searches. This step is still very time consuming for larger number of genomes.

For every pair of genomes a result file is produced. If two genomes have genome_id's \samp{GID111},
and \samp{GID222} then the result file \samp{GID111_vs_GID222.txt} (or \samp{GID222_vs_GID111.txt}) will
be found in \samp{out.folder} after the completion of this search.

The \samp{out.folder} is scanned for already existing result files, and \code{\link{blastpAllAll}} never
overwrites an existing result file. If a file with the name \samp{GID111_vs_GID222.txt} already exists in
the \samp{out.folder}, this particular search is skipped. This makes it possible to run multiple jobs in
parallell, writing to the same \samp{out.folder}. It also makes it possible to add new genomes, and only
BLAST the new combinations without repeating previous comparisons. 

This search can be slow if the genomes contain many proteins and it scales quadratically in the number of
input files. It is best suited for the study of a smaller number of genomes. By
starting multiple R sessions, you can speed up the search by running \code{\link{blastpAllAll}} from each R
session, using the same \samp{out.folder} but different integers for the \code{job} option. If you are
using a computing cluster you can also increase the number of CPUs by increasing \code{threads}.

The result files are tab-separated text files, and can be read into R, but more
commonly they are used as input to \code{\link{bDist}} to compute distances between sequences for subsequent
clustering.
}
\note{
The \samp{blast+} software must be installed on the system for this function to work, i.e. the command
\samp{system("makeblastdb -help")} must be recognized as valid commands if you
run them in the Console window.
}
\examples{
\dontrun{
# This example requires the external BLAST+ software
# Using protein files in this package
pf <- file.path(path.package("micropan"), "extdata",
                str_c("xmpl_GID", 1:3, ".faa.xz"))

# We need to uncompress them first...
prot.files <- tempfile(fileext = c("_GID1.faa.xz","_GID2.faa.xz","_GID3.faa.xz"))
ok <- file.copy(from = pf, to = prot.files)
prot.files <- unlist(lapply(prot.files, xzuncompress))

# Blasting all versus all
out.dir <- "."
blastpAllAll(prot.files, out.folder = out.dir)

# Reading results, and computing blast.distances
blast.files <- list.files(out.dir, pattern = "GID[0-9]+_vs_GID[0-9]+.txt")
blast.distances <- bDist(file.path(out.dir, blast.files))

# ...and cleaning...
ok <- file.remove(prot.files)
ok <- file.remove(file.path(out.dir, blast.files))
}

}
\references{
Camacho, C., Coulouris, G., Avagyan, V., Ma, N., Papadopoulos, J., Bealer, K., Madden, T.L.
(2009). BLAST+: architecture and applications. BMC Bioinformatics, 10:421.
}
\seealso{
\code{\link{panPrep}}, \code{\link{bDist}}.
}
\author{
Lars Snipen and Kristian Hovde Liland.
}
