% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/riley.r
\name{riley}
\alias{riley}
\title{Fit the alternative model for bivariate random-effects meta-analysis}
\usage{
riley(X, slab, optimization = "Nelder-Mead", control = list(), pars,
  ...)
}
\arguments{
\item{X}{data frame containing integer variables \code{Y1}, \code{vars1}, \code{Y2} and \code{vars2},
where the columns \code{Y1} and \code{Y2} represent the effect sizes of outcome 1 and, respectively, outcome 2. The columns 
\code{vars1} and \code{vars2} represent the error variances of \code{Y1} and, respectively, \code{Y2}. Alternatively, 
when considering a meta-analysis of diagnostic test accuracy data, the columns \code{TP}, \code{FN}, \code{FP} and 
\code{TN} may be specified. Corresponding values then represent the number of true positives, the number of false negatives,
the number of false positives and, respectively, the number of true negatives.}

\item{slab}{Optional vector specifying the label for each study}

\item{optimization}{The optimization method that should be used for minimizing the negative (restricted) 
log-likelihood function. The default method is an implementation of that of Nelder and Mead (1965), 
that uses only function values and is robust but relatively slow. Other methods are described in \link[stats]{optim}.}

\item{control}{A list of control parameters to pass to \link[stats]{optim}.}

\item{pars}{List with additional arguments. The width of confidence, credibility and prediction intervals is 
defined by \code{level} (defaults to 0.95).}

\item{\dots}{Arguments to be passed on to other functions. See "Details" for more information.}
}
\value{
An object of the class \code{riley} for which many standard methods are available. A warning message is 
casted when the Hessian matrix contains negative eigenvalues, which implies that the identified solution is a 
saddle point and thus not optimal.
}
\description{
This function fits the alternative model for bivariate random-effects meta-analysis when the within-study correlations 
are unknown. This bivariate model was proposed by Riley et al. (2008) and is similar to the general bivariate 
random-effects model (van Houwelingen et al. 2002), but includes an overall correlation parameter rather than 
separating the (usually unknown) within- and between-study correlation. As a consequence, the alternative model 
is not fully hierarchical, and estimates of additional variation beyond sampling error (\code{psi}) are not 
directly equivalent to the between-study variation (\code{tau}) from the general model. This model is particularly 
useful when there is large within-study variability, few primary studies are available or the general model 
estimates the between-study correlation as 1 or -1.
}
\details{
Parameters are estimated by iteratively maximizing the restriced log-likelihood using the Newton-Raphson procedure. 
The results from a univariate random-effects meta-analysis with a method-of-moments estimator are used as starting 
values for \code{beta1}, \code{beta2}, \code{psi1} and \code{psi2} in the \code{optim} command. Standard errors for all parameters are obtained 
from the inverse Hessian matrix.

\subsection{Meta-analysis of effect sizes}{
The following parameters are estimated by iteratively maximizing the restriced log-likelihood using the Newton-Raphson 
procedure: pooled effect size for outcome 1 (\code{beta1}), pooled effect size for outcome 2 (\code{beta2}), 
additional variation of \code{beta1} beyond sampling error (\code{psi1}), additional variation of \code{beta2} 
beyond sampling error (\code{psi2}) and the correlation \code{rho} between \code{psi1} and \code{psi2}. 

}

\subsection{Meta-analysis of diagnostic test accuracy}{
Although the model can also be used for diagnostic test accuracy data when substantial within-study correlations 
are expected, assuming zero within-study correlations (i.e. applying Reitsma's approach) is usually justified 
(Reitsma et al. 2005, Daniels and Hughes 1997, Korn et al. 2005, Thompson et al. 2005, Van Houwelingen et al. 2002).

A logit transformation is applied to the sensitivities ans false positive rates of each study, in order to meet the normality 
assumptions. When zero cell counts occur, continuity corrections may be required. The correction value can be specified using
\code{correction} (defaults to 0.5). Further, when the argument \code{correction.control} is set to \code{"all"} 
(the default) the continuity correction is added to the whole data if only one cell in one study is zero. 
If \code{correction.control="single"} the correction is only applied to rows of the data which have a zero.

The following parameters are estimated: logit of sensitivity (\code{beta1}), logit of false positive rate (\code{beta2}), 
additional variation of \code{beta1} beyond sampling error (\code{psi1}), additional variation of \code{beta2} beyond 
sampling error (\code{psi2}) and the correlation (\code{rho}) between \code{psi1} and \code{psi2}. 
}
}
\note{
The overall correlation parameter \code{rho} is transformed during estimation to ensure that corresponding values
remain between -1 and 1. The transformed correlation \code{rhoT} is given as \code{logit((rho+1)/2)}. During optimization,
the starting value for \code{rhoT} is set to 0.  The standard error of \code{rho} is derived from \code{rhoT} using 
the Delta method. Similarly, the Delta methods is used to derive the standard error of the sensitivity and false 
positive rate from \code{beta1} and, respectively, \code{beta2}.

Algorithms for dealing with missing data are currently not implemented, but Bayesian approaches will become 
available in later versions.
}
\examples{
data(Scheidler)
data(Daniels)
data(Kertai)

# Meta-analysis of potential surrogate markers data
# The results obtained by Riley (2008) were as follows:
# beta1 = -0.042 (SE = 0.063),
# beta2 = 14.072 (SE = 4.871)
# rho   = -0.759
\dontrun{
fit1 <- riley(Daniels) #maxit reached, try again with more iterations
}
fit1 <- riley(Daniels, control=list(maxit=10000))
summary(fit1)

# Meta-analysis of prognostic test studies
fit2 <- riley(Kertai)
fit2

# Meta-analysis of computed tomography data 
ds <- Scheidler[which(Scheidler$modality==1),]
fit3 <- riley(ds)
fit3

}
\references{
\itemize{
\item Korn EL, Albert PS, McShane LM. Assessing surrogates as trial endpoints using mixed models. 
\emph{Statistics in Medicine} 2005; \bold{24}: 163--182.
\item Nelder JA, Mead R. A simplex algorithm for function minimization. \emph{Computer Journal} (1965); \bold{7}: 308--313.
\item Reitsma J, Glas A, Rutjes A, Scholten R, Bossuyt P, Zwinderman A. Bivariate analysis of sensitivity and 
specificity produces informative summary measures in diagnostic reviews. \emph{Journal of Clinical Epidemiology} 2005; 
\bold{58}: 982--990.
\item Riley RD, Thompson JR, Abrams KR. An alternative model for bivariate random-effects meta-analysis when 
the within-study correlations are unknown. \emph{Biostatistics} 2008; \bold{9}: 172--186.
\item Thompson JR, Minelli C, Abrams KR, Tobin MD, Riley RD. Meta-analysis of genetic studies using mendelian 
randomization--a multivariate approach. \emph{Statistics in Medicine} 2005; \bold{24}: 2241--2254.
\item van Houwelingen HC, Arends LR, Stijnen T. Advanced methods in meta-analysis: multivariate approach and 
meta-regression. \emph{Statistics in Medicine} 2002; \bold{21}: 589--624.
}
}
\author{
Thomas Debray <thomas.debray@gmail.com>
}
\keyword{bivariate}
\keyword{meta-analysis}
\keyword{multivariate}
\keyword{regression}
\keyword{riley}
