% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcrpure.R
\name{mcrpure}
\alias{mcrpure}
\title{Multivariate curve resolution based on pure variables}
\usage{
mcrpure(
  x,
  ncomp,
  purevars = NULL,
  offset = 0.05,
  use.deriv = 0,
  savgol = NULL,
  exclrows = NULL,
  exclcols = NULL,
  info = ""
)
}
\arguments{
\item{x}{spectra of mixtures (matrix or data frame).}

\item{ncomp}{maximum number of components to calculate.}

\item{purevars}{vector with indices for pure variables (optional, if you want to provide the variables directly).}

\item{offset}{offset for computing purity spectra (should be value within [0, 1)).}

\item{use.deriv}{a number which tells how to use derivative (0 - do not use, 1 - only for estimation of  pure
variables, 2 - both for pure variables estimation and for unmixing).}

\item{savgol}{list with parameters for Savitzky-Golay preprocessing (if \code{use.deriv} is not 0).}

\item{exclrows}{rows to be excluded from calculations (numbers, names or vector with logical values).}

\item{exclcols}{columns to be excluded from calculations (numbers, names or vector with logical values).}

\item{info}{a short text with description of the case (optional).}
}
\value{
Returns an object of \code{\link{mcrpure}} class with the following fields:
\item{resspec}{matrix with resolved spectra.}
\item{rescont}{matrix with resolved contributions.}
\item{purevars}{indices of the selected pure variables.}
\item{purevals}{purity values for the selected pure variables.}
\item{purityspec}{purity spectra (matrix with purity values for each variable and component).}
\item{expvar }{vector with explained variance for each component (in percent).}
\item{cumexpvar }{vector with cumulative explained variance for each component (in percent).}
\item{offset}{offset value used to compute the purity}
\item{ncomp}{number of resolved components}
\item{use.deriv}{value for corresponding argument.}
\item{savgol}{value for corresponding argument.}
\item{info }{information about the model, provided by user when build the model.}


More details and examples can be found in the Bookdown tutorial.
}
\description{
\code{mcrpure} allows to resolve spectroscopic data to linear combination of individual spectra
and contributions using the pure variables approach.
}
\details{
The method estimates purity of each variable and then uses the purest ones to decompose the
spectral data into spectra (`resspec`) and contributions (`rescont`) of individual
chemical components by ordinary least squares.

The purity of of variables for the first component is computed as:

   `purity = sigma / (mu + max(mu) * offset)``

where `sigma` is standard deviation of the original spectra, `mu` is mean and `offset` is a
parameter, provided by user. The default value for the `offset` is 0.05, but usually values
between 0.01 and 0.10 can be a good choice.

Sometimes, especially for NIR and UV/Vis data, using derivatives can help to get better results.
In this case, you need to specify a value for \code{use.deriv} - 1 if you want to use derivative
only for detection of pure variables or 2 - if you want to use derivative both for estimatiion
of the pure variables and for the unmixing/resolving. The derivative is computed using
Savitzky-Golay transformation, so you need to provide the parameters as additional argument
\code{savgol}. The values should be provided as a list using names from \code{\link{prep.savgol}}
, for example: \code{savgol = list(width = 3, porder = 1, dorder = 1)}.

More details about the method can be found in [1].
}
\examples{
library(mdatools)

# resolve mixture of carbonhydrates Raman spectra

data(carbs)
m = mcrpure(carbs$D, ncomp = 3)

# examples for purity spectra plot (you can select which components to show)
par(mfrow = c(2, 1))
plotPuritySpectra(m)
plotPuritySpectra(m, comp = 2:3)

# you can do it manually and combine e.g. with original spectra
par(mfrow = c(1, 1))
mdaplotg(
   list(
      "spectra" = prep.norm(carbs$D, "area"),
      "purity" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
   ), col = c("gray", "red"), type = "l"
)

# show the maximum purity for each component
par(mfrow = c(1, 1))
plotPurity(m)

# plot cumulative and individual explained variance
par(mfrow = c(1, 2))
plotVariance(m)
plotCumVariance(m)

# plot resolved spectra (all of them or individually)
par(mfrow = c(2, 1))
plotSpectra(m)
plotSpectra(m, comp = 2:3)

# plot resolved contributions (all of them or individually)
par(mfrow = c(2, 1))
plotContributions(m)
plotContributions(m, comp = 2:3)

# of course you can do this manually as well, e.g. show original
# and resolved spectra
par(mfrow = c(1, 1))
mdaplotg(
   list(
      "original" = prep.norm(carbs$D, "area"),
      "resolved" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
   ), col = c("gray", "red"), type = "l"
)

# in case if you have reference spectra of components you can compare them with
# the resolved ones:
par(mfrow = c(3, 1))
for (i in 1:3) {
   mdaplotg(
      list(
         "pure" = prep.norm(mda.subset(mda.t(carbs$S), 1), "area"),
         "resolved" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
      ), col = c("gray", "red"), type = "l", lwd = c(3, 1)
   )
}

# See bookdown tutorial for more details.

}
\references{
1. S. Kucheryavskiy, A. Bogomolov, W. Windih  (2016). Spectral unmixing using the concept of pure
variables. I C. Ruckebusch (red.), Resolving Spectral Mixtures: With Applications from Ultrafast
Time-Resolved Spectroscopy to Super-Resolution Imaging (1 udg., s. 53-100). Elsevier. Data
Handling in Science and Technology, Bind. 30
}
\seealso{
Methods for \code{mcrpure} objects:
\tabular{ll}{
   \code{summary.mcrpure} \tab shows some statistics for the case.\cr
   \code{\link{unmix.mcrpure}} \tab makes unmixing of new set of spectra.\cr
   \code{\link{predict.mcrpure}} \tab computes contributions by projection of new spectra to
   the resolved ones.\cr
}

Plotting methods for \code{mcrpure} objects:
\tabular{ll}{
   \code{\link{plotPurity.mcrpure}} \tab shows plot with maximum purity of each component.\cr
   \code{\link{plotPuritySpectra.mcrpure}} \tab shows plot with purity spectra.\cr
   \code{\link{plotSpectra.mcr}} \tab shows plot with resolved spectra.\cr
   \code{\link{plotContributions.mcr}} \tab shows plot with resolved contributions.\cr
   \code{\link{plotVariance.mcr}} \tab shows plot with explained variance.\cr
   \code{\link{plotCumVariance.mcr}} \tab shows plot with cumulative explained variance.\cr
}
}
\author{
Sergey Kucheryavskiy (svkucheryavski@gmail.com)
}
