% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mc_bart.R
\name{brt}
\alias{brt}
\title{Create a model component object for a BART (Bayesian Additive Regression Trees)
component in the linear predictor}
\usage{
brt(
  formula,
  X = NULL,
  n.trees = 75L,
  name = "",
  debug = FALSE,
  keepTrees = FALSE,
  ...
)
}
\arguments{
\item{formula}{a formula specifying the predictors to be used in the BART
model component. Variable names are looked up in the data frame
passed as \code{data} argument to \code{\link{create_sampler}} or
\code{\link{generate_data}}, or in \code{environment(formula)}.}

\item{X}{a design matrix can be specified directly, as an alternative
to the creation of one based on \code{formula}. If \code{X} is
specified \code{formula} is ignored.}

\item{n.trees}{number of trees used in the BART ensemble.}

\item{name}{the name of the model component. This name is used in the output of the
MCMC simulation function \code{\link{MCMCsim}}. By default the name will be 'bart'
with the number of the model term attached.}

\item{debug}{if \code{TRUE} a breakpoint is set at the beginning of the posterior
draw function associated with this model component. Mainly intended for developers.}

\item{keepTrees}{whether to store the trees ensemble for each Monte Carlo draw. This
is required for prediction based on new data. The default is \code{FALSE} to save
memory.}

\item{...}{parameters passed to \code{\link[dbarts]{dbarts}}.}
}
\value{
An object with precomputed quantities and functions for sampling from
 prior or conditional posterior distributions for this model component. Intended
 for internal use by other package functions.
}
\description{
This function is intended to be used on the right hand side of the
\code{formula} argument to \code{\link{create_sampler}} or
\code{\link{generate_data}}. It creates a BART term in the
model's linear predictor. To use this model component one needs
to have R package \pkg{dbarts} installed.
}
\examples{
# generate data, based on an example in Friedman (1991)
gendat <- function(n=200L, p=10L, sigma=1) {
  x <- matrix(runif(n * p), n, p)
  mu <- 10*sin(pi*x[, 1] * x[, 2]) + 20*(x[, 3] - 0.5)^2 + 10*x[, 4] + 5*x[, 5]
  y <- mu + sigma * rnorm(n)
  data.frame(x=x, mu=mu, y=y)
}

train <- gendat()
test <- gendat(n=25)

# keep trees for later prediction based on new data
sampler <- create_sampler(
  y ~ brt(~ . - y, name="bart", keepTrees=TRUE),
  sigma.mod=pr_invchisq(df=3,  scale=var(train$y)),
  data = train
)
sim <- MCMCsim(sampler, n.chain=2, n.iter=700, thin=2,
  store.all=TRUE, verbose=FALSE)
(summ <- summary(sim))
plot(train$mu, summ$bart[, "Mean"]); abline(0, 1)
# NB prediction is currently slow
\donttest{
pred <- predict(sim, newdata=test,
  iters=sample(seq_len(ndraws(sim)), 100),
  show.progress=FALSE
)
(summpred <- summary(pred))
plot(test$mu, summpred[, "Mean"]); abline(0, 1)
}

}
\references{
H.A. Chipman, E.I. Georgea and R.E. McCulloch (2010).
   BART: Bayesian additive regression trees.
   The Annals of Applied Statistics 4(1), 266-298.
   
 J.H. Friedman (1991).
   Multivariate adaptive regression splines.
   The Annals of Statistics 19, 1-67.
}
