% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_base.R
\name{fit_base.mcgf}
\alias{fit_base.mcgf}
\title{Parameter estimation for symmetric correlation functions for an \code{mcgf}
object.}
\usage{
\method{fit_base}{mcgf}(
  x,
  lag,
  horizon = 1,
  model = c("spatial", "temporal", "sep", "fs", "none"),
  method = c("wls", "mle"),
  optim_fn = c("nlminb", "optim", "other"),
  par_fixed = NULL,
  par_init,
  lower = NULL,
  upper = NULL,
  other_optim_fn = NULL,
  dists_base = NULL,
  scale_time = 1,
  ...
)
}
\arguments{
\item{x}{An \code{mcgf} object containing attributes \code{dists}, \code{acfs}, \code{ccfs}, and
\code{sds}.}

\item{lag}{Integer time lag.}

\item{horizon}{Integer forecast horizon.}

\item{model}{Base model, one of \code{spatial}, \code{temporal}, \code{sep}, \code{fs}, \code{none}.
Only \code{sep} and \code{fs} are supported when \code{method = mle}. If \code{none}, NULLs are
returned.}

\item{method}{Parameter estimation methods, weighted least square (\code{wls}) or
maximum likelihood estimation (\code{mle}).}

\item{optim_fn}{Optimization functions, one of \code{nlminb}, \code{optim}, \code{other}.
When \code{optim_fn = other}, supply \code{other_optim_fn}.}

\item{par_fixed}{Fixed parameters.}

\item{par_init}{Initial values for parameters to be optimized.}

\item{lower}{Optional; lower bounds of parameters.}

\item{upper}{Optional: upper bounds of parameters.}

\item{other_optim_fn}{Optional, other optimization functions. The first two
arguments must be initial values for the parameters and a function to be
minimized respectively (same as that of \code{optim} and \code{nlminb}).}

\item{dists_base}{List of distance matrices. If NULL, \code{dists(x)} is used.
Must be a matrix or an array of distance matrices.}

\item{scale_time}{Scale of time unit, default is 1. \code{lag} is divided by
\code{scale_time} for parameter estimation.}

\item{...}{Additional arguments passed to \code{optim_fn}.}
}
\value{
A list containing outputs from optimization functions of \code{optim_fn}.
}
\description{
Parameter estimation for symmetric correlation functions for an \code{mcgf}
object.
}
\details{
This function fits the separable and fully symmetric models using weighted
least squares and maximum likelihood estimation. Optimization functions such
as \code{nlminb} and \code{optim} are supported. Other functions are also supported by
setting \code{optim_fn = "other"} and supplying \code{other_optim_fn}. \code{lower} and
\code{upper} are lower and upper bounds of parameters in \code{par_init} and default
bounds are used if they are not specified.

Note that both \code{wls} and \code{mle} are heuristic approaches when \code{x} contains
observations from a subset of the discrete spatial domain, though estimation
results are close to that using the full spatial domain for large sample
sizes.
}
\examples{
data(sim1)
sim1_mcgf <- mcgf(sim1$data, dists = sim1$dists)
sim1_mcgf <- add_acfs(sim1_mcgf, lag_max = 5)
sim1_mcgf <- add_ccfs(sim1_mcgf, lag_max = 5)

# Fit a pure spatial model
fit_spatial <- fit_base(
    sim1_mcgf,
    model = "spatial",
    lag = 5,
    par_init = c(c = 0.001, gamma = 0.5),
    par_fixed = c(nugget = 0)
)
fit_spatial$fit

# Fit a pure temporal model
fit_temporal <- fit_base(
    sim1_mcgf,
    model = "temporal",
    lag = 5,
    par_init = c(a = 0.3, alpha = 0.5)
)
fit_temporal$fit

# Fit a separable model
fit_sep <- fit_base(
    sim1_mcgf,
    model = "sep",
    lag = 5,
    par_init = c(
        c = 0.001,
        gamma = 0.5,
        a = 0.3,
        alpha = 0.5
    ),
    par_fixed = c(nugget = 0)
)
fit_sep$fit
}
\seealso{
Other functions on fitting an mcgf: 
\code{\link{add_base.mcgf}()},
\code{\link{add_lagr.mcgf}()},
\code{\link{fit_lagr.mcgf}()},
\code{\link{krige.mcgf}()}
}
\concept{functions on fitting an mcgf}
