% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mp_directions.R
\name{mp_directions}
\alias{mp_directions}
\title{Get directions from the Google Maps Directions API}
\usage{
mp_directions(
  origin,
  waypoints = NULL,
  destination,
  mode = c("driving", "transit", "walking", "bicycling"),
  arrival_time = NULL,
  departure_time = NULL,
  alternatives = FALSE,
  avoid = c(NA, "tolls", "highways", "ferries", "indoor"),
  region = NULL,
  traffic_model = c("best_guess", "pessimistic", "optimistic"),
  transit_mode = c("bus", "subway", "train", "tram"),
  transit_routing_preference = c(NA, "less_walking", "fewer_transfers"),
  language = NULL,
  key,
  quiet = FALSE
)
}
\arguments{
\item{origin}{Origin, as \itemize{
\item{\code{character} vector of length one with address to be geocoded}
\item{\code{numeric} vector of length two (lon, lat)}
\item{\code{matrix} with one row and two columns (lon, lat)}
\item{\code{sf} or \code{sfc} point layer with one feature}
}}

\item{waypoints}{Waypoints, in one of the same formats as for \code{origins} but possibly with more than one location, i.e. \itemize{
\item{\code{character} vector with addresses to be geocoded}
\item{\code{numeric} vector of length two (lon, lat)}
\item{\code{matrix} with two columns (lon, lat)}
\item{\code{sf} or \code{sfc} point layer}
}}

\item{destination}{Destination, in one of the same formats as for \code{origins}}

\item{mode}{Travel mode, one of: \code{"driving"} (default), \code{"transit"}, \code{"walking"}, \code{"bicycling"}}

\item{arrival_time}{The desired time of arrival for transit directions, as \code{POSIXct}}

\item{departure_time}{The desired time of departure, as \code{POSIXct}}

\item{alternatives}{Whether to return more than one alternative (\code{logical}, default is \code{FALSE})}

\item{avoid}{\code{NA} (default, means avoid nothing) or one of: \code{"tolls"}, \code{"highways"}, \code{"ferries"} or \code{"indoor"}}

\item{region}{The region code, specified as a ccTLD ("top-level domain") two-character value (e.g. \code{"es"} for Spain) (optional)}

\item{traffic_model}{The traffic model, one of: \code{"best_guess"} (the default), \code{"pessimistic"}, \code{"optimistic"}. The \code{traffic_model} parameter is only taken into account when \code{departure_time} is specified!}

\item{transit_mode}{Transit preferred mode, one or more of: \code{"bus"}, \code{"subway"}, \code{"train"} or \code{"tram"}}

\item{transit_routing_preference}{Transit route preference. \code{NA} (default, means no preference) or one of: \code{"less_walking"} or \code{"fewer_transfers"}}

\item{language}{The language in which to return directions. See \url{https://developers.google.com/maps/faq#languagesupport} for list of language codes.}

\item{key}{Google APIs key}

\item{quiet}{Logical; suppress printing URL for Google Maps API call (e.g. to hide API key)}
}
\value{
XML document with Google Maps Directions API response
}
\description{
Get directions from the Google Maps Directions API
}
\note{
\itemize{
\item Use function \code{\link{mp_get_routes}} to extract \code{sf} line layer where each feature is a \strong{route}
\item Use function \code{\link{mp_get_segments}} to extract \code{sf} line layer where each feature is a \strong{route segment}
}
}
\examples{
# Built-in reponse example
library(xml2)
doc = as_xml_document(response_directions_driving)
r = mp_get_routes(doc)
seg = mp_get_segments(doc)

\dontrun{

# Text file with API key
key = readLines("~/key")

# Using 'numeric' input
doc = mp_directions(
  origin = c(34.81127, 31.89277),
  destination = c(34.781107, 32.085003),
  alternatives = TRUE,
  key = key
)

# Using 'character' and 'sf' input
library(sf)
doc = mp_directions(
  origin = "Beer-Sheva",
  destination = c(34.781107, 32.085003) |> st_point() |> st_sfc(crs = 4326),
  alternatives = TRUE,
  key = key
)

# Comparing traffic models
doc = mp_directions(
  origin = "Beer-Sheva",
  destination = "Tel Aviv",
  departure_time = Sys.time() + as.difftime(1, units = "hours"),
  traffic_model = "best_guess",
  key = key
)
mp_get_routes(doc)$duration_in_traffic_text
doc = mp_directions(
  origin = "Beer-Sheva",
  destination = "Tel Aviv",
  departure_time = Sys.time() + as.difftime(1, units = "hours"),
  traffic_model = "optimistic",
  key = key
)
mp_get_routes(doc)$duration_in_traffic_text
doc = mp_directions(
  origin = "Beer-Sheva",
  destination = "Tel Aviv",
  departure_time = Sys.time() + as.difftime(1, units = "hours"),
  traffic_model = "pessimistic",
  key = key
)
mp_get_routes(doc)$duration_in_traffic_text
}
}
\references{
\url{https://developers.google.com/maps/documentation/directions/overview}
}
