% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_diffusion.R
\name{measure_diffusion_node}
\alias{measure_diffusion_node}
\alias{node_adoption_time}
\alias{node_thresholds}
\alias{node_recovery}
\alias{node_exposure}
\title{Measures of nodes in a diffusion}
\usage{
node_adoption_time(diff_model)

node_thresholds(diff_model, normalized = TRUE, lag = 1)

node_recovery(diff_model)

node_exposure(.data, mark, time = 0)
}
\arguments{
\item{diff_model}{A valid network diffusion model,
as created by \code{as_diffusion()} or \code{play_diffusion()}.}

\item{normalized}{Logical scalar, whether the centrality scores are normalized.
Different denominators are used depending on whether the object is one-mode or two-mode,
the type of centrality, and other arguments.}

\item{lag}{The number of time steps back upon which the thresholds are
inferred.}

\item{.data}{An object of a manynet-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{mark}{A valid 'node_mark' object or
logical vector (TRUE/FALSE) of length equal to
the number of nodes in the network.}

\item{time}{A time point until which infections/adoptions should be
identified. By default \code{time = 0}.}
}
\description{
These functions allow measurement of various features of
a diffusion process:
\itemize{
\item \code{node_adoption_time()}: Measures the number of time steps until
nodes adopt/become infected
\item \code{node_thresholds()}: Measures nodes' thresholds from the amount
of exposure they had when they became infected
\item \code{node_infection_length()}: Measures the average length nodes that become
infected remain infected in a compartmental model with recovery
\item \code{node_exposure()}: Measures how many exposures nodes have to
a given mark
}
}
\section{Adoption time}{

\code{node_adoption_time()} measures the time units it took
until each node became infected.
Note that an adoption time of 0 indicates that this was a seed node.
}

\section{Thresholds}{

\code{node_thresholds()} infers nodes' thresholds based on how much
exposure they had when they were infected.
This inference is of course imperfect,
especially where there is a sudden increase in exposure,
but it can be used heuristically.
In a threshold model,
nodes activate when \eqn{\sum_{j:\text{active}} w_{ji} \geq \theta_i},
where \eqn{w} is some (potentially weighted) matrix,
\eqn{j} are some already activated nodes,
and \eqn{theta} is some pre-defined threshold value.
Where a fractional threshold is used, the equation is
\eqn{\frac{\sum_{j:\text{active}} w_{ji}}{\sum_{j} w_{ji}} \geq \theta_i}.
That is, \eqn{theta} is now a proportion,
and works regardless of whether \eqn{w} is weighted or not.
}

\section{Infection length}{

\code{node_infection_length()} measures the average length of time that nodes
that become infected remain infected in a compartmental model with recovery.
Infections that are not concluded by the end of the study period are
calculated as infinite.
}

\section{Exposure}{

\code{node_exposure()} calculates the number of infected/adopting nodes
to which each susceptible node is exposed.
It usually expects network data and
an index or mark (TRUE/FALSE) vector of those nodes which are currently infected,
but if a diff_model is supplied instead it will return
nodes exposure at \eqn{t = 0}.
}

\examples{
  smeg <- generate_smallworld(15, 0.025)
  smeg_diff <- play_diffusion(smeg, recovery = 0.2)
  plot(smeg_diff)
  # To measure when nodes adopted a diffusion/were infected
  (times <- node_adoption_time(smeg_diff))
  # To infer nodes' thresholds
  node_thresholds(smeg_diff)
  # To measure how long each node remains infected for
  node_recovery(smeg_diff)
  # To measure how much exposure nodes have to a given mark
  node_exposure(smeg, mark = c(1,3))
  node_exposure(smeg_diff)
}
\references{
\subsection{On diffusion measures}{

Valente, Tom W. 1995. \emph{Network models of the diffusion of innovations}
(2nd ed.). Cresskill N.J.: Hampton Press.
}
}
\seealso{
Other measures: 
\code{\link{measure_attributes}},
\code{\link{measure_central_between}},
\code{\link{measure_central_close}},
\code{\link{measure_central_degree}},
\code{\link{measure_central_eigen}},
\code{\link{measure_closure}},
\code{\link{measure_cohesion}},
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_net}},
\code{\link{measure_features}},
\code{\link{measure_heterogeneity}},
\code{\link{measure_hierarchy}},
\code{\link{measure_holes}},
\code{\link{measure_periods}},
\code{\link{measure_properties}},
\code{\link{member_diffusion}}

Other diffusion: 
\code{\link{make_play}},
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_net}},
\code{\link{member_diffusion}}
}
\concept{diffusion}
\concept{measures}
