% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliBioCliIdx.R
\name{cliBioCliIdxPoints}
\alias{cliBioCliIdxPoints}
\title{Calculator for Bioclimatic Indices}
\usage{
cliBioCliIdxPoints(
  temp,
  prec,
  bsdf = NULL,
  lat = NULL,
  elv = NULL,
  year = 2000,
  MSMC = 150,
  aprchTEMP = c("hip", "tsi", "const"),
  aprchPREC = c("tsi", "hip", "const"),
  aprchBSDF = c("hip", "const"),
  dvTEMP = rep(0.7, 12),
  dvPREC = rep(0.7, 12),
  bciOpVar = c("abt", "tap", "per", "fai"),
  argCkd = FALSE
)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{bsdf}{'numeric' R object with one-year time series of monthly mean relative sunshine duration (dimensionless)}

\item{lat}{'numeric' vector with the latitude coordinates (in decimal degrees)}

\item{elv}{'numeric' vector with the elevation values (in meters above sea level)}

\item{year}{'numeric' vector with values of the year (using astronomical year numbering)}

\item{MSMC}{'numeric' vector with values of the maximum soil moisture capacity (aka 'bucket size') (in mm)}

\item{aprchTEMP}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily mean air temperature for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean air temperature in order to generate daily values; \cr
(b) \code{'tsi'} -
this scheme uses an iterative interpolation technique (Lüdeke et al. 1994) to time series of the monthly mean
air temperature, in order to generate a synthetic time series of the selected meteorological variable at a
temporal resolution that is higher than the daily scale; finally, this synthetic time series is upscaled to a
daily resolution; \cr
(c) \code{'const'} -
this scheme is assumed that values of the daily mean air temperature are constant within each month.}

\item{aprchPREC}{'character' vector of length 1 that indicates the scheme to generate daily values of the
daily precipitation sum. Valid values are as follows: \cr
(a) \code{'tsi'} -
this scheme uses an iterative interpolation technique (Lüdeke et al. 1994) to time series of the monthly mean
precipitation intensity, in order to generate a synthetic time series of the selected meteorological variable
at a temporal resolution that is higher than the daily scale; finally, this synthetic time series is upscaled
to a daily resolution; \cr
(b) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean precipitation intensity in order to generate daily values; \cr
(c) \code{'const'} -
this scheme is assumed that values of the daily precipitation sum are constant within each month (the monthly
precipitation sum is divided equally across each day of the month).}

\item{aprchBSDF}{'character' vector of length 1 that indicates the scheme used to generate daily values of the
daily fractional sunshine duration for a specific year. Valid values are as follows: \cr
(a) \code{'hip'} -
this scheme applies the mean-preserving 'harmonic' interpolation method of Epstein (1991) to the values of
monthly mean relative sunshine duration in order to generate daily values; \cr
(b) \code{'const'} -
this scheme is assumed that values of the daily relative sunshine duration are constant within each month.}

\item{dvTEMP}{'numeric' vector of length 12 with monthly values of the damping variable for the air temperature
data.}

\item{dvPREC}{'numeric' vector of length 12 with monthly values of the damping variable for the precipitation data.}

\item{bciOpVar}{'character' vector of at least one length that indicates which of the bioclimatic indices is/are
to be computed. Valid values are as follows: \cr
(a) \code{'abt'} - Mean Annual Biotemperature (in °C); \cr
(b) \code{'tap'} - Total Annual Precipitation (in mm); \cr
(c) \code{'per'} - Potential Evapotranspiration Ratio (dimensionless); \cr
(d) \code{'fai'} - Forestry Aridity Index (dimensionless); \cr
(e) \code{'gdd0'} - Growing Degree-Days on 0°C base (in °C day); \cr
(f) \code{'gdd5'} - Growing Degree-Days on 5°C base (in °C day);  \cr
(g) \code{'bdi'} - Budyko's Dryness Index (dimensionless); \cr
(h) \code{'cci'} - Condrad's Continentality Index (in per cent);  \cr
(i) \code{'tc'} - Mean Temperature of the Coldest Month (in °C); \cr
(j) \code{'tw'} - Mean Temperature of the Warmest Month (in °C);  \cr
(k) \code{'ptc'} - Priestley–Taylor Coefficient (dimensionless).}

\item{argCkd}{'logical' scalar that indicates whether or not the checking and correction of arguments can be
omitted.}
}
\value{
A matrix with one or more columns where each column contain the values of a given bioclimatic index.
}
\description{
Calculates the values of selected bioclimatic indices, for a given geographical location (latitude
    and elevation) and year/epoch, by using the monthly time series of temperature, precipitation and relative
    sunshine duration.
}
\details{
Taking into account all implemented bioclimatic indices, the following five require only temperature data:

    \itemize{
      \item{\code{abt}: Mean Annual Biotemperature (Eq 1 in Szelepcsényi et al. (2014); in °C)}
      \item{\code{tc}: Mean Temperature of the Coldest Month (in °C)}
      \item{\code{tw}: Mean Temperature of the Warmest Month (in °C)}
      \item{\code{gdd5}: Growing Degree-Days on 5°C base (in °C day)}
      \item{\code{gdd0}: Growing Degree-Days on 0°C base (in °C day)}
    }

    Monthly data are sufficient to calculate values of the mean temperatures of the coldest and warmest months and
    the mean annual biotemperature, while daily values are needed to compute values of the growing degree-days. If
    only a set of these bioclimatic indices has to be calculated, the setting \code{prec = NULL} must be used. \cr
    An important bioclimatic index for the Holdridge life zone system is the total annual precipitation, for the
    calculation of which requires only monthly precipitation data. If only this bioclimatic index has to be
    computed, the setting \code{temp = NULL} must be used. \cr
    In addition to monthly temperature data, latitude coordinates are also required to calculate the Condrad's
    Continentality Index (\code{'cci'}: Eq 4 in Conrad (1946); in per cent). \cr
    For calculating values of the Potential Evapotranspiration Ratio used in the Holdridge life zone system
    (\code{'per'}: Eq 4 in Szelepcsényi et al. (2014); dimensionless) and the Forestry Aridity Index introduced by
    the forestry climate classification (\code{'fai'}: Eq 1 in Führer et al. (2011); dimensionless), both
    temperature and precipitation data at a monthly timescale are also required. \cr
    The computation of the Budyko's Dryness Index (\code{'bdi'}, dimensionless) and the Priestley–Taylor
    Coefficient (\code{'ptc'}, dimensionless) requires a simulation of evapotranspiration at daily time step via
    the implementation of the SPLASH algorithm (Davis et al. 2017) (see
    \code{\link[macroBiome]{dlyEngWtrFluxPoints}}). In addition to one-year time series of daily temperature and
    precipitation data, the application of the SPLASH algorithm requires values of the relative sunshine duration
    at a daily timescale, latitude coordinate, altitude, year/epoch, and the so-called 'bucket size'. The Dryness
    Index is a ratio of annual potential evapotranspiration to precipitation (see Monserud et al. 1993). The value
    of the Priestley–Taylor coefficient is calculated as the ratio of actual evapotranspiration to equilibrium
    evapotranspiration, which represents the fraction of plant-available surface moisture (see Prentice et al.
    1992, Davis et al. 2017). \cr
    The function applies only monthly time series to compute values of each bioclimatic index, considering the
    field of application of the package. However, as we can see, in some cases there is a need for daily time
    series that are here generated by using the function \code{\link[macroBiome]{dlyWeaGenPoints}}.
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
    follows: \code{'temp'} (one-year time series of monthly mean air temperature), \code{'prec'} (one-year time
    series of monthly precipitation sum), and \code{'bsdf'} (one-year time series of monthly mean relative sunshine
    duration). The objects \code{'temp'}, \code{'prec'} and \code{'bsdf'} must be either vectors of length 12 or
    12-column matrices. The first dimensions of these matrices have to be the same length. The function
    automatically converts vectors into single-row matrices during the error handling, and then uses these
    matrices. The first dimensions of these matrices determines the number of rows in the result matrix. In the
    case of arguments that do not affect the course of the calculation procedure or the structure of the return
    object, scalar values (i.e., 'numeric' vector of length 1) may also be allowed. In this case, they are as
    follows: \code{'lat'} (latitude coordinates in decimal degrees), \code{'elv'} (elevation in meters above sea
    level), \code{'year'} (year using astronomical year numbering), and \code{'MSMC'} ('bucket size' in mm). These
    scalars are converted to vectors by the function during the error handling, and these vectors are applied in
    the further calculations. If these data are stored in vectors of length at least 2, their length must be the
    same size of first dimension of the matrices containing the basic data.
}
\examples{
# Loading mandatory data for the Example 'Points'
data(inp_exPoints)

# Calculate values of all default bioclimatic indices with default settings,
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
with(inp_exPoints, {
bci1 <- cliBioCliIdxPoints(colMeans(temp), colMeans(prec))
bci1
})

\donttest{
# Calculate values of all selected bioclimatic indices with default settings,
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
with(inp_exPoints, {
year <- trunc(mean(seq(1981, 2010)))
bciOpVar <- c("gdd5", "bdi", "cci", "tc", "gdd0", "tw", "ptc")
bci2 <- cliBioCliIdxPoints(colMeans(temp), colMeans(prec), colMeans(bsdf), lat, elv,
   year = year, bciOpVar = bciOpVar)
bci2
})
}

}
\references{
\cite{Conrad V (1964) Usual formulas of continentality and their limits of validity. EOS, Trans Am Geophys Union
    27(5):663-664. \doi{10.1029/TR027i005p00663}}

\cite{Davis TW, Prentice IC, Stocker BD, Thomas RT, Whitley RJ, Wang H, Evans BJ, Gallego-Sala AV, Sykes MT,
    Cramer W (2017) Simple process-led algorithms for simulating habitats (SPLASH v.1.0): robust indices of
    radiation, evapotranspiration and plant-available moisture. Geosci Model Dev 10(2):689–708.
    \doi{10.5194/gmd-10-689-2017}}

\cite{Epstein ES (1991) On Obtaining Daily Climatological Values from Monthly Means. J Clim 4(3):365–368.
    \doi{10.1175/1520-0442(1991)004<0365:OODCVF>2.0.CO;2}}

\cite{Führer E, Horváth L, Jagodics A, Machon A, Szabados I (2011) Application of a new aridity index in Hungarian
    forestry practice. Időjárás 115(3):205–216}

\cite{Lüdeke MKB, Badeck FW, Otto RD, Häger C, Dönges S, Kindermann J, Würth G, Lang T, Jäkel U, Klaudius A,
    Ramge P, Habermehl S, Kohlmaier GH (1994) The Frankfurt Biosphere Model: A global process-oriented model of
    seasonal and long-term CO2 exchange between terrestrial ecosystems and the atmosphere. I. Model description
    and illustrative results for cold deciduous and boreal forests. Clim Res 4(2):143-166. \doi{10.3354/cr004143}}

\cite{Monserud RA, Denissenko OV, Tchebakova NM (1993) Comparison of Siberian paleovegetation to current and
    future vegetation under climate change. Clim Res 3(3):143–159. \doi{10.3354/cr003143}}

\cite{Prentice IC, Cramer W, Harrison SP, Leemans R, Monserud RA, Solomon AM (1992) A Global Biome Model Based on
    Plant Physiology and Dominance, Soil Properties and Climate. J Biogeogr 19(2):117–134. \doi{10.2307/2845499}}

\cite{Szelepcsényi Z, Breuer H, Sümegi P (2014) The climate of Carpathian Region in the 20th century based on the
    original and modified Holdridge life zone system. Cent Eur J Geosci 6(3):293–307.
    \doi{10.2478/s13533-012-0189-5}}
}
