# ===============================================================================
#
# PROGRAMMERS:
#
# jean-romain.roussel.1@ulaval.ca  -  https://github.com/Jean-Romain/lidR
#
# COPYRIGHT:
#
# Copyright 2016-2018 Jean-Romain Roussel
#
# This file is part of lidR R package.
#
# lidR is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>
#
# ===============================================================================

#' Summary and Print for \code{LAS*} objects
#'
#' @param object,x A \code{LAS*} object
#' @param ... Unused
#'
#' @return NULL, used for its side-effect of printing information
setGeneric("print", function(x, ...)
  standardGeneric("print"))

setGeneric("summary", function(object, ...)
  standardGeneric("summary"))

#' Plot a LAS* object
#'
#' Plot displays a 3D interactive windows-based on rgl for \link{LAS} objects
#'
#' @param x A \code{LAS*} object
#' @param y Unused (inherited from R base)
#' @param color characters. The attribute used to color the point cloud. Default is Z coordinates. RGB
#' is an allowed string even if it refers to three attributes simultaneously.
#' @param colorPalette characters. A vector of colors such as that generated by heat.colors,
#' topo.colors, terrain.colors or similar functions. Default is \code{height.colors(50)}
#' provided by the package \code{lidR}
#' @param bg The color for the background. Default is black.
#' @param trim numeric. Enables trimming of values when outliers break the color palette range.
#' Every point with a value higher than \code{trim} will be plotted with the highest color.
#' @param clear_artifacts logical. It is a known and documented issue that the 3D visualisation with
#' \code{rgl} displays artifacts. The points look aligned and/or regularly spaced in some view angles.
#' This is because \code{rgl} computes with single precision \code{float}. To fix that the point
#' cloud is shifted to (0,0) to reduce the number of digits needed to represent its coordinates.
#' The drawback is that the point cloud is not plotted at its actual coordinates.
#' @param nbits integer. If \code{color = RGB} it assumes that RGB colors are coded on 16 bits as described
#' in the LAS format specification. However, this is not always respected. If the colors are stored
#' on 8 bits set this parameter to 8.
#' @param backend character. Can be \code{"rgl"} or \code{"pcv"}. If \code{"rgl"} is chosen
#' the display relies on the \code{rgl} package. If \code{"pcv"} is chosen it relies on the
#' \code{PointCloudViewer} package, which is much more efficient and can handle million of points
#' using less memory. \code{PointCloudViewer} is not available on CRAN yet and should
#' be installed from github (see. \url{https://github.com/Jean-Romain/PointCloudViewer}).
#' @param ... Will be passed to \link[rgl:points3d]{points3d} (LAS) or \link[graphics:plot]{plot}
#' if \code{mapview = FALSE} or to \link[mapview:mapView]{mapview} if \code{mapview = TRUE} (LAScatalog).
#' @examples
#' LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
#' las <- readLAS(LASfile)
#'
#' plot(las, clear_artifact = TRUE)
#'
#' # Outliers in intensity break the color range. Use the trim parameter.
#' col <- heat.colors(50)
#' plot(las, color = Intensity, colorPalette = col)
#' plot(las, color = Intensity, colorPalette = col, trim = 70)
#' @export
#' @method plot LAS
setGeneric("plot", function(x, y, ...)
  standardGeneric("plot"))

#' Test if a \code{LAS} object is empty
#'
#' An empty \code{LAS} object is a point cloud with 0 points
#'
#' @param object A \code{LAS} object
#' @param ... Unused
#'
#' @return TRUE or FALSE
#'
#' @examples
#' LASfile <- system.file("extdata", "example.laz", package="rlas")
#' las = readLAS(LASfile)
#' is.empty(las)
#'
#' las = new("LAS")
#' is.empty(las)
#' @export
setGeneric("is.empty", function(object, ...)
  standardGeneric("is.empty"))

#' Surface covered by a LAS* object.
#'
#' Surface covered by a \code{LAS*} object. For \code{LAS} point clouds it is computed based on the
#' convex hull of the points. For a \code{LAScatalog} it is computed as the sum of the bounding boxes
#' of the files. For overlapping tiles the value may be larger than the total covered area because
#' some regions are sampled twice.
#'
#' @param x An object of the class \code{LAS*}
#' @param ... unused
#'
#' @return numeric. The area of the object computed in the same units as the coordinate reference system
#' @export
#' @importMethodsFrom raster area
#' @rdname area
setGeneric("area", function(x, ...)
  standardGeneric("area"))

#' Get or set epsg code of a LAS* object
#'
#' Currently, at least in R, the only way to store the CRS in a las file is to store the epsg code of
#' the CRS. On the other hand, all spatial R packages use the proj4string to store the CRS. This is why
#' the CRS is duplicated in a LAS object. The proj4string is accessible with the functions \code{projection}
#' from \code{raster} or \code{proj4string} from \code{sp}. These functions do nothing special except
#' they return or update the proj4string, which is not recognized in a las file. The function \code{epsg},
#' in turn, updates the header of the LAS object \strong{and} updates the proj4string and thus should
#' be preferred to ensure that the CRS will be written into a las file.
#'
#' @param object An object of class LAS or eventually LASheader (regular users don't need to manipulate
#' LASheader objects)
#' @param ... Unused
#' @param value integer. EPSG code.
#'
#' @export
#' @examples
#' LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
#' las = readLAS(LASfile)
#'
#' projection(las)
#' epsg(las)
#'
#' # Changing the CRS via projection updates the proj4string but not the header
#' # The proj4string will be propagated in all subsequent spatial objects but the las file
#' # itself will still have the epsg code 26917 if written on the disk
#' projection(las) = sp::CRS("+init=epsg:2567")
#' projection(las)
#' epsg(las)
#'
#' # Changing the CRS via epsg updates the proj4string and the header. This way the good CRS will
#' # written in the las file (if written).
#' epsg(las) = 2567
#' epsg(las)
#' projection(las)
setGeneric("epsg", function(object, ...)
  standardGeneric("epsg"))

#' @export
#' @rdname epsg
setGeneric("epsg<-", function(object, value)
  standardGeneric("epsg<-"))
