% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flab.R
\name{flab}
\alias{flab}
\title{Filter Rows Using Variable Value Labels}
\usage{
flab(data, condition)
}
\arguments{
\item{data}{the data.frame from which columns will be selected.}

\item{condition}{row-filtering conditions along the lines of base::subset()
and/or dplyr::filter(), which may involve a combination of value labels
(for value-labeled variables only) and actual values (for non-value-labeled
variables only).}
}
\value{
a labelr label attribute-preserving data.frame consisting of the
selected rows that meet the filtering condition(s).
}
\description{
\code{flab} ("filter using labels") allows one to filter-subset a data.frame
using value or numerical range labels.
}
\details{
\code{flab} accepts a labelr value-labeled data.frame, followed by condition-
based row-filtering instructions (in the fashion of base::subset or
dplyr::filter) expressed in terms of variable value labels that exist only
as meta-data (i.e., not visible using View(), head(), etc.), and returns the
filtered data.frame in terms of the values themselves.

For example, if value labels were added to the integer nominal variable
"raceth" of (notional) data.frame df (using \code{add_val_labs} or \code{add_val1}), one
could then use flab to filter down to only raceth==3 rows using a call like
flab(df, raceth=="African-American") (assuming here that the integer value
raceth==3 has previously been given the value label "African-American". As
another example, flab(mtcars, am=="automatic") would return (only) those rows
of mtcars where automatic==0 if the value label "automatic" has been uniquely
associated with the mtcars$am value of 0 (again, via a prior call to
\code{add_val_labs} or \code{add_val1}). This functionality may be useful for
interactively subsetting a data.frame, where character value labels may be
more intuitive and easily recalled than the underlying variable values
themselves (e.g., raceth=="White" & gender="F" may be more intuitive or readily
recalled than raceth==3 & gender==2).

Note that \code{flab} (and labelr more broadly) is intended for moderate-sized (or
smaller) data.frames, defined loosely as those with a few million or fewer
rows. With a conventional (c. 2023) laptop, labelr operations on modest-
sized (~100K rows) take seconds (or less); with larger (> a few million rows)
data.frames, labelr may take several minutes (or run out of memory and fail
altogether!), depending on specifics.

See also \code{slab}, \code{use_val_labs}, \code{add_val_labs}, \code{add_val1}, \code{add_quant_labs}, \code{add_quant1}, \cr
\code{get_val_labs}, \code{drop_val_labs}. For label-preserving subsetting tools that subset in
terms of raw values (not value labels), see \code{sfilter}, \code{sbrac}, \code{ssubset}, \code{sdrop}.
}
\examples{
# make toy demographic (gender, raceth, etc.) data set
set.seed(555)
df <- make_demo_data(n = 1000) # another labelr:: function
# let's add variable VALUE labels for variable "raceth"
df <- add_val_labs(df,
  vars = "raceth", vals = c(1:7),
  labs = c("White", "Black", "Hispanic", "Asian", "AIAN", "Multi", "Other"),
  max.unique.vals = 50
)

# let's add variable VALUE labels for variable "gender"
# note that, if we are labeling a single variable, we can use add_val1()
# distinction between add_val1() and add_val_labs() will become more meaningful
# when we get to our Likert example
df <- add_val1(
  data = df, gender, vals = c(0, 1, 2),
  labs = c("M", "F", "O"), max.unique.vals = 50
)

# see what we did
# get_val_labs(df)
get_val_labs(df, "gender")
get_val_labs(df, "raceth")

# use --labels-- to filter w/ flab() ("*F*ilter *lab*el")
dflab <- flab(df, raceth == "Asian" & gender == "F")
head(dflab, 4)

# equivalently, use --values--- to filter w/ sfilter() ("*S*afe filter")
dfsf <- sfilter(df, raceth == 3 & gender == 1)
head(dfsf, 4)
}
