% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layers-rnn.R
\name{layer_rnn}
\alias{layer_rnn}
\title{Base class for recurrent layers}
\usage{
layer_rnn(
  object,
  cell,
  return_sequences = FALSE,
  return_state = FALSE,
  go_backwards = FALSE,
  stateful = FALSE,
  unroll = FALSE,
  zero_output_for_mask = FALSE,
  ...
)
}
\arguments{
\item{object}{Object to compose the layer with. A tensor, array, or sequential model.}

\item{cell}{A RNN cell instance or a list of RNN cell instances.
A RNN cell is a class that has:
\itemize{
\item A \code{call(input_at_t, states_at_t)} method, returning
\verb{(output_at_t, states_at_t_plus_1)}. The call method of the
cell can also take the optional argument \code{constants}, see
section "Note on passing external constants" below.
\item A \code{state_size} attribute. This can be a single integer
(single state) in which case it is the size of the recurrent
state. This can also be a list of integers
(one size per state).
\item A \code{output_size} attribute, a single integer.
\item A \code{get_initial_state(batch_size=NULL)}
method that creates a tensor meant to be fed to \code{call()} as the
initial state, if the user didn't specify any initial state
via other means. The returned initial state should have
shape \verb{(batch_size, cell.state_size)}.
The cell might choose to create a tensor full of zeros,
or other values based on the cell's implementation.
\code{inputs} is the input tensor to the RNN layer, with shape
\verb{(batch_size, timesteps, features)}.
If this method is not implemented
by the cell, the RNN layer will create a zero filled tensor
with shape \verb{(batch_size, cell$state_size)}.
In the case that \code{cell} is a list of RNN cell instances, the cells
will be stacked on top of each other in the RNN, resulting in an
efficient stacked RNN.
}}

\item{return_sequences}{Boolean (default \code{FALSE}). Whether to return the last
output in the output sequence, or the full sequence.}

\item{return_state}{Boolean (default \code{FALSE}).
Whether to return the last state in addition to the output.}

\item{go_backwards}{Boolean (default \code{FALSE}).
If \code{TRUE}, process the input sequence backwards and return the
reversed sequence.}

\item{stateful}{Boolean (default \code{FALSE}). If TRUE, the last state
for each sample at index \code{i} in a batch will be used as initial
state for the sample of index \code{i} in the following batch.}

\item{unroll}{Boolean (default \code{FALSE}).
If TRUE, the network will be unrolled, else a symbolic loop will be
used. Unrolling can speed-up a RNN, although it tends to be more
memory-intensive. Unrolling is only suitable for short sequences.}

\item{zero_output_for_mask}{Boolean (default \code{FALSE}).
Whether the output should use zeros for the masked timesteps.
Note that this field is only used when \code{return_sequences}
is \code{TRUE} and \code{mask} is provided.
It can useful if you want to reuse the raw output sequence of
the RNN without interference from the masked timesteps, e.g.,
merging bidirectional RNNs.}

\item{...}{For forward/backward compatability.}
}
\value{
The return value depends on the value provided for the first argument.
If  \code{object} is:
\itemize{
\item a \code{keras_model_sequential()}, then the layer is added to the sequential model
(which is modified in place). To enable piping, the sequential model is also
returned, invisibly.
\item a \code{keras_input()}, then the output tensor from calling \code{layer(input)} is returned.
\item \code{NULL} or missing, then a \code{Layer} instance is returned.
}
}
\description{
Base class for recurrent layers
}
\section{Call Arguments}{
\itemize{
\item \code{inputs}: Input tensor.
\item \code{initial_state}: List of initial state tensors to be passed to the first
call of the cell.
\item \code{mask}: Binary tensor of shape \verb{[batch_size, timesteps]}
indicating whether a given timestep should be masked.
An individual \code{TRUE} entry indicates that the corresponding
timestep should be utilized, while a \code{FALSE} entry indicates
that the corresponding timestep should be ignored.
\item \code{training}: Python boolean indicating whether the layer should behave in
training mode or in inference mode. This argument is passed
to the cell when calling it.
This is for use with cells that use dropout.
}
}

\section{Input Shape}{
3-D tensor with shape \verb{(batch_size, timesteps, features)}.
}

\section{Output Shape}{
\itemize{
\item If \code{return_state}: a list of tensors. The first tensor is
the output. The remaining tensors are the last states,
each with shape \verb{(batch_size, state_size)}, where \code{state_size} could
be a high dimension tensor shape.
\item If \code{return_sequences}: 3D tensor with shape
\verb{(batch_size, timesteps, output_size)}.
}
}

\section{Masking:}{
This layer supports masking for input data with a variable number
of timesteps. To introduce masks to your data,
use a \code{\link[=layer_embedding]{layer_embedding()}} layer with the \code{mask_zero} parameter
set to \code{TRUE}.

Note on using statefulness in RNNs:

You can set RNN layers to be 'stateful', which means that the states
computed for the samples in one batch will be reused as initial states
for the samples in the next batch. This assumes a one-to-one mapping
between samples in different successive batches.

To enable statefulness:
\itemize{
\item Specify \code{stateful = TRUE} in the layer constructor.
\item Specify a fixed batch size for your model, by passing
\itemize{
\item If sequential model:
\code{input_batch_shape = c(...)} to the \code{keras_model_sequential()} call.
\item Else for functional model with 1 or more input layers:
\code{batch_shape = c(...)} to the \code{layer_input()} call(s).
}

This is the expected shape of your inputs
\emph{including the batch size}.
It should be a list of integers, e.g. \code{c(32, 10, 100)}.
\item Specify \code{shuffle = FALSE} when calling \code{fit()}.
}

To reset the states of your model, call \code{\link[=reset_state]{reset_state()}} on either
a specific layer, or on your entire model.

Note on specifying the initial state of RNNs:

You can specify the initial state of RNN layers symbolically by
calling them with the keyword argument \code{initial_state}. The value of
\code{initial_state} should be a tensor or list of tensors representing
the initial state of the RNN layer.
}

\section{Examples}{
First, let's define a RNN Cell, as a layer subclass.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rnn_cell_minimal <- Layer(
  "MinimalRNNCell",

  initialize = function(units, ...) \{
    super$initialize(...)
    self$units <- as.integer(units)
    self$state_size <- as.integer(units)
  \},

  build = function(input_shape) \{
    self$kernel <- self$add_weight(
      shape = shape(tail(input_shape, 1), self$units),
      initializer = 'uniform',
      name = 'kernel'
    )
    self$recurrent_kernel <- self$add_weight(
      shape = shape(self$units, self$units),
      initializer = 'uniform',
      name = 'recurrent_kernel'
    )
    self$built <- TRUE
  \},

  call = function(inputs, states) \{
    prev_output <- states[[1]]
    h <- op_matmul(inputs, self$kernel)
    output <- h + op_matmul(prev_output, self$recurrent_kernel)
    list(output, list(output))
  \}
)
}\if{html}{\out{</div>}}

Let's use this cell in a RNN layer:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{cell <- rnn_cell_minimal(units = 32)
x <- layer_input(shape = shape(NULL, 5))
layer <- layer_rnn(cell = cell)
y <- layer(x)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{cells <- list(rnn_cell_minimal(units = 32), rnn_cell_minimal(units = 4))
x <- layer_input(shape = shape(NULL, 5))
layer <- layer_rnn(cell = cells)
y <- layer(x)
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/layers/recurrent_layers/rnn#rnn-class}
}

Other rnn cells: \cr
\code{\link{rnn_cell_gru}()} \cr
\code{\link{rnn_cell_lstm}()} \cr
\code{\link{rnn_cell_simple}()} \cr

Other rnn layers: \cr
\code{\link{layer_bidirectional}()} \cr
\code{\link{layer_conv_lstm_1d}()} \cr
\code{\link{layer_conv_lstm_2d}()} \cr
\code{\link{layer_conv_lstm_3d}()} \cr
\code{\link{layer_gru}()} \cr
\code{\link{layer_lstm}()} \cr
\code{\link{layer_simple_rnn}()} \cr
\code{\link{layer_time_distributed}()} \cr
\code{\link{rnn_cell_gru}()} \cr
\code{\link{rnn_cell_lstm}()} \cr
\code{\link{rnn_cell_simple}()} \cr
\code{\link{rnn_cells_stack}()} \cr

Other layers: \cr
\code{\link{Layer}()} \cr
\code{\link{layer_activation}()} \cr
\code{\link{layer_activation_elu}()} \cr
\code{\link{layer_activation_leaky_relu}()} \cr
\code{\link{layer_activation_parametric_relu}()} \cr
\code{\link{layer_activation_relu}()} \cr
\code{\link{layer_activation_softmax}()} \cr
\code{\link{layer_activity_regularization}()} \cr
\code{\link{layer_add}()} \cr
\code{\link{layer_additive_attention}()} \cr
\code{\link{layer_alpha_dropout}()} \cr
\code{\link{layer_attention}()} \cr
\code{\link{layer_average}()} \cr
\code{\link{layer_average_pooling_1d}()} \cr
\code{\link{layer_average_pooling_2d}()} \cr
\code{\link{layer_average_pooling_3d}()} \cr
\code{\link{layer_batch_normalization}()} \cr
\code{\link{layer_bidirectional}()} \cr
\code{\link{layer_category_encoding}()} \cr
\code{\link{layer_center_crop}()} \cr
\code{\link{layer_concatenate}()} \cr
\code{\link{layer_conv_1d}()} \cr
\code{\link{layer_conv_1d_transpose}()} \cr
\code{\link{layer_conv_2d}()} \cr
\code{\link{layer_conv_2d_transpose}()} \cr
\code{\link{layer_conv_3d}()} \cr
\code{\link{layer_conv_3d_transpose}()} \cr
\code{\link{layer_conv_lstm_1d}()} \cr
\code{\link{layer_conv_lstm_2d}()} \cr
\code{\link{layer_conv_lstm_3d}()} \cr
\code{\link{layer_cropping_1d}()} \cr
\code{\link{layer_cropping_2d}()} \cr
\code{\link{layer_cropping_3d}()} \cr
\code{\link{layer_dense}()} \cr
\code{\link{layer_depthwise_conv_1d}()} \cr
\code{\link{layer_depthwise_conv_2d}()} \cr
\code{\link{layer_discretization}()} \cr
\code{\link{layer_dot}()} \cr
\code{\link{layer_dropout}()} \cr
\code{\link{layer_einsum_dense}()} \cr
\code{\link{layer_embedding}()} \cr
\code{\link{layer_feature_space}()} \cr
\code{\link{layer_flatten}()} \cr
\code{\link{layer_flax_module_wrapper}()} \cr
\code{\link{layer_gaussian_dropout}()} \cr
\code{\link{layer_gaussian_noise}()} \cr
\code{\link{layer_global_average_pooling_1d}()} \cr
\code{\link{layer_global_average_pooling_2d}()} \cr
\code{\link{layer_global_average_pooling_3d}()} \cr
\code{\link{layer_global_max_pooling_1d}()} \cr
\code{\link{layer_global_max_pooling_2d}()} \cr
\code{\link{layer_global_max_pooling_3d}()} \cr
\code{\link{layer_group_normalization}()} \cr
\code{\link{layer_group_query_attention}()} \cr
\code{\link{layer_gru}()} \cr
\code{\link{layer_hashed_crossing}()} \cr
\code{\link{layer_hashing}()} \cr
\code{\link{layer_identity}()} \cr
\code{\link{layer_integer_lookup}()} \cr
\code{\link{layer_jax_model_wrapper}()} \cr
\code{\link{layer_lambda}()} \cr
\code{\link{layer_layer_normalization}()} \cr
\code{\link{layer_lstm}()} \cr
\code{\link{layer_masking}()} \cr
\code{\link{layer_max_pooling_1d}()} \cr
\code{\link{layer_max_pooling_2d}()} \cr
\code{\link{layer_max_pooling_3d}()} \cr
\code{\link{layer_maximum}()} \cr
\code{\link{layer_mel_spectrogram}()} \cr
\code{\link{layer_minimum}()} \cr
\code{\link{layer_multi_head_attention}()} \cr
\code{\link{layer_multiply}()} \cr
\code{\link{layer_normalization}()} \cr
\code{\link{layer_permute}()} \cr
\code{\link{layer_random_brightness}()} \cr
\code{\link{layer_random_contrast}()} \cr
\code{\link{layer_random_crop}()} \cr
\code{\link{layer_random_flip}()} \cr
\code{\link{layer_random_rotation}()} \cr
\code{\link{layer_random_translation}()} \cr
\code{\link{layer_random_zoom}()} \cr
\code{\link{layer_repeat_vector}()} \cr
\code{\link{layer_rescaling}()} \cr
\code{\link{layer_reshape}()} \cr
\code{\link{layer_resizing}()} \cr
\code{\link{layer_separable_conv_1d}()} \cr
\code{\link{layer_separable_conv_2d}()} \cr
\code{\link{layer_simple_rnn}()} \cr
\code{\link{layer_spatial_dropout_1d}()} \cr
\code{\link{layer_spatial_dropout_2d}()} \cr
\code{\link{layer_spatial_dropout_3d}()} \cr
\code{\link{layer_spectral_normalization}()} \cr
\code{\link{layer_string_lookup}()} \cr
\code{\link{layer_subtract}()} \cr
\code{\link{layer_text_vectorization}()} \cr
\code{\link{layer_tfsm}()} \cr
\code{\link{layer_time_distributed}()} \cr
\code{\link{layer_torch_module_wrapper}()} \cr
\code{\link{layer_unit_normalization}()} \cr
\code{\link{layer_upsampling_1d}()} \cr
\code{\link{layer_upsampling_2d}()} \cr
\code{\link{layer_upsampling_3d}()} \cr
\code{\link{layer_zero_padding_1d}()} \cr
\code{\link{layer_zero_padding_2d}()} \cr
\code{\link{layer_zero_padding_3d}()} \cr
\code{\link{rnn_cell_gru}()} \cr
\code{\link{rnn_cell_lstm}()} \cr
\code{\link{rnn_cell_simple}()} \cr
\code{\link{rnn_cells_stack}()} \cr
}
\concept{layers}
\concept{rnn cells}
\concept{rnn layers}
