# This file contains functions named in English which are just wrappers to the original functions which are named in Spanish. Here I also include the documentation in English. Only external functions that can be used by users have this bilingual feature.

#' Create Karel's world
#'
#' This function takes a "world" (i.e. a list with data about its size, walls,
#' beepers and Karel's position and direction), plots it and prepares everything
#' so that Karel can start performing actions in it. It must be run always
#' before Karel starts working on her goals, especially if we have made a
#' mistake, we must start all over again by first running this function.
#'
#' After running \code{generate_mundo()}, we can run Karel's actions and finally
#' visualize it all with the function \code{run_actions()}.
#'
#' @param world Character vector of length 1 with the name of one of the
#'   provided worlds in the package or a list provided by the user with all the
#'   components that a world needs (see more below in details).
#'
#' @return Plots the initial state of Karel's world and prepares everything to
#'   start recording her actions.
#'
#' @export
#'
#' @examples
#' generate_world("mundo001")
#'
#' @seealso \code{\link{actions}} \code{\link{run_actions}}
#'
#' @details Argument \code{world} can be create by the user. In this case, it
#'   must be a list with the following components:
#'
#'   \enumerate{
#'     \item \code{nx}: TODO
#'     \item \code{ny}:
#'     \item \code{hor_walls}:
#'     \item \code{ver_walls}:
#'     \item \code{karel_x}:
#'     \item \code{karel_y}:
#'     \item \code{karel_dir}:
#'     \item \code{beepers_x}:
#'     \item \code{beepers_y}:
#'     \item \code{beepers_n}:
#'     \item \code{beepers_bag}:
#'   }
#'
#' @importFrom ggplot2 ggplot geom_segment geom_point aes scale_x_continuous
#'   scale_y_continuous theme element_blank element_text geom_tile geom_text
#'   geom_rect coord_fixed
#' @importFrom dplyr tibble add_row slice mutate bind_rows n
#' @importFrom magrittr %>%
#'
generate_world <- function(world) generar_mundo(world)

#' Run actions
#'
#' This function produces the animation that shows all actions performed by
#' Karel since its world was generated by \code{generate_world}.
#'
#' @param loop A logical value TRUE or FALSE indicating if the animation should
#' repeat itself after finished or not (defaults to TRUE).
#'
#' @return Produces an animation with \code{gganimate}.
#'
#' @examples
#' generate_world("mundo001")
#' move()
#' pick_beeper()
#' turn_left()
#' put_beeper()
#' run_actions()
#'
#' @seealso \code{\link{generate_world}}
#'
#' @export
run_actions <- function(loop = TRUE) ejecutar_acciones(repetir = loop)

#' Available actions for Karel
#'
#' \code{move()}, \code{turn_left()}, \code{pick_beeper()} y \code{put_beeper()}
#' are the four basic activities that Karel can perform. If you turn on Karel's
#' superpowers with \code{load_super_karel()}, then she can also
#' \code{turn_right()} y \code{turn_around()}.
#'
#' @return These functions don't return anything, but make changes in Karel's
#'   world that are visible when all the actions are run through
#'   \code{run_actions()}.
#'
#' @examples
#' generate_world("mundo001")
#' move()
#' pick_beeper()
#' turn_left()
#' put_beeper()
#' run_actions()
#'
#' @seealso \code{\link{load_super_karel}} \code{\link{generate_world}}
#'   \code{\link{run_actions}}
#'
#' @name actions
NULL
#> NULL

#' @rdname actions
#' @export
move <- function() avanzar()

#' @rdname actions
#' @export
turn_left <- function() girar_izquierda()

#' @rdname actions
#' @export
put_beeper <- function() poner_coso()


#' @rdname actions
#' @export
pick_beeper <- function() juntar_coso()

#' Turn on Karel's superpowers
#'
#' After running \code{load_super_karel()}, Karel can also turn right and turn
#' around with \code{turn_right()} and \code{turn_around()}. If these
#' superpowers aren't loaded, then these functions won't be available and Karel
#' can't use them.
#'
#' @return It doesn't return anything but attaches to the global environment the
#'   functions \code{turn_right()} and \code{turn_around()}.
#'
#' @examples
#' generate_world("mundo001")
#' load_super_karel()
#' turn_around()
#' turn_right()
#' run_actions()
#'
#' @seealso \code{\link{actions}} \code{\link{generate_world}}
#'   \code{\link{run_actions}}
#' @export
load_super_karel <- function() {
  # This hack solves the note
  global_env_set_hack("turn_right", turn_right, 1L)
  global_env_set_hack("turn_around", turn_around, 1L)
}
#
#' @rdname actions
#' @export
turn_right <- function() girar_derecha()

#' @rdname actions
#' @export
turn_around <- function() darse_vuelta()

#' Conditions that Karel can test
#'
#' These group of functions return a logical value \code{TRUE} o \code{FALSE} according to Karel's evaluation of her world.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @details The functions \code{front_is_clear()}, \code{front_is_blocked()}, \code{left_is_clear()}, \code{left_is_blocked()}, \code{right_is_clear()} y \code{right_is_blocked()} test if there is a wall in front of Karel, to her left or to her right, respectively. The functions \code{beepers_present()} y \code{no_beepers_present()} test if there are or there are not \code{beepers} at Karel's current position. The functions \code{karel_has_beepers()} y \code{karel_has_no_beepers()} test if Karel has or hasn't got  \code{beepers} in her bag (not visible in the plot). The functions \code{facing_east()}, \code{facing_west()}, \code{facing_north()} y \code{facing_south()} test the direction to which Karel is facing right now.
#'
#' @examples
#' generate_world("mundo001")
#' front_is_clear()
#' front_is_blocked()
#' left_is_clear()
#' left_is_blocked()
#' right_is_clear()
#' right_is_blocked()
#' beepers_present()
#' no_beepers_present()
#' karel_has_beepers()
#' karel_has_no_beepers()
#' facing_east()
#' facing_west()
#' facing_north()
#' facing_south()
#'
#' @seealso \code{\link{generate_world}}
#'
#' @name conditions
NULL
#> NULL

#' @rdname conditions
#' @export
front_is_clear <- function() frente_abierto()

#' @rdname conditions
#' @export
front_is_blocked <- function() frente_cerrado()

#' @rdname conditions
#' @export
left_is_clear <- function() izquierda_abierto()

#' @rdname conditions
#' @export
left_is_blocked <- function() izquierda_cerrado()

#' @rdname conditions
#' @export
right_is_clear <- function() derecha_abierto()

#' @rdname conditions
#' @export
right_is_blocked <- function() derecha_cerrado()

#' @rdname conditions
#' @export
beepers_present <- function() hay_cosos()

#' @rdname conditions
#' @export
no_beepers_present <- function() no_hay_cosos()

#' @rdname conditions
#' @export
karel_has_beepers <- function() karel_tiene_cosos()

#' @rdname conditions
#' @export
karel_has_no_beepers <- function() karel_no_tiene_cosos()

#' @rdname conditions
#' @export
facing_east <- function() mira_al_este()

#' @rdname conditions
#' @export
facing_west <- function() mira_al_oeste()

#' @rdname conditions
#' @export
facing_north <- function() mira_al_norte()

#' @rdname conditions
#' @export
facing_south <- function() mira_al_sur()
