% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/long2wide_omv.R
\name{long2wide_omv}
\alias{long2wide_omv}
\title{Converts .omv-files for the statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org}) from long to wide format}
\usage{
long2wide_omv(
  dtaInp = NULL,
  fleOut = "",
  varID = "ID",
  varTme = c(),
  varExc = c(),
  varTgt = c(),
  varSep = "_",
  varOrd = c("times", "vars"),
  varAgg = c("mean", "first"),
  varSrt = c(),
  usePkg = c("foreign", "haven"),
  selSet = "",
  ...
)
}
\arguments{
\item{dtaInp}{Either a data frame or the name of a data file to be read (including the path, if required; "FILENAME.ext"; default: NULL); files can be of any supported file type, see Details below}

\item{fleOut}{Name of the data file to be written (including the path, if required; "FILE_OUT.omv"; default: ""); if empty, the resulting data frame is returned instead}

\item{varID}{Names of one or more variables that identify the same group / individual (default: c())}

\item{varTme}{Name of the variable(s) that differentiates multiple records from the same group / individual (default: c())}

\item{varExc}{Name of the variable(s) should be excluded from the transformation, typically this will be between-subject-variable(s) (default: c())}

\item{varTgt}{Names of one or more variables to be transformed / reshaped (other variables are excluded, if empty(c()) all variables except varTme, varID and varExc are included; default: c())}

\item{varSep}{Separator character when concatenating the fixed and time-varying part of the variable name ("VAR1_1", "VAR1_2"; default: "_")}

\item{varOrd}{How variables / columns are organized: for "times" (default) the steps of the time varying variable are adjacent, for "vars" the steps of the original columns in the long dataset}

\item{varAgg}{How multiple occurrences of particular combinations of time varying variables are aggregated: either "mean" (calculate the mean over occurrences), or "first" (take the first occurrence)}

\item{varSrt}{Variable(s) that are used to sort the data frame (see Details; if empty, the order returned from reshape is kept; default: c())}

\item{usePkg}{Name of the package: "foreign" or "haven" that shall be used to read SPSS, Stata and SAS files; "foreign" is the default (it comes with base R), but "haven" is newer and more comprehensive}

\item{selSet}{Name of the data set that is to be selected from the workspace (only applies when reading .RData-files)}

\item{...}{Additional arguments passed on to methods; see Details below}
}
\value{
a data frame (only returned if fleOut is empty) where the input data set is converted from long to wide format
}
\description{
Converts .omv-files for the statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org}) from long to wide format
}
\details{
The ellipsis-parameter (...) can be used to submit arguments / parameters to the functions that are used for transforming or reading the data. The transformation uses \code{reshape}. When reading the
data, the functions are: \code{read_omv} (for jamovi-files), \code{read.table} (for CSV / TSV files; using similar defaults as \code{read.csv} for CSV and \code{read.delim} for TSV which both are based upon
\code{read.table} but with adjusted defaults for the respective file types), \code{readRDS} (for rds-files), \code{read_sav} (needs R-package "haven") or \code{read.spss} (needs R-package "foreign") for SPSS-files,
\code{read_dta} ("haven") / \code{read.dta} ("foreign") for Stata-files, \code{read_sas} ("haven") for SAS-data-files, and \code{read_xpt} ("haven") / \code{read.xport} ("foreign") for SAS-transport-files. If you would
like to use "haven", it may be needed to install it manually (i.e., \code{install.packages("haven", dep = TRUE)}).
}
\examples{
\dontrun{
library(jmvReadWrite)
# generate a test dataframe with 100 (imaginary) participants / units of
#  observation (ID), 8 measurement (measure) of one variable (X)
dtaInp <- data.frame(ID = rep(as.character(seq(1, 100)), each = 8),
                     measure = rep(seq(1, 8), times = 100),
                     X = runif(800, -10, 10))
cat(str(dtaInp))
# the output should look like this
# 'data.frame': 800 obs. of  3 variables:
#  $ ID     : chr  "1" "1" "1" "1" ...
#  $ measure: int  1 2 3 4 5 6 7 8 1 2 ...
#  $ X      : num  ...
# this data set is stored as (temporary) RDS-file and later processed by long2wide
nmeInp <- paste0(tempfile(), ".rds")
nmeOut <- paste0(tempfile(), ".omv")
saveRDS(dtaInp, nmeInp)
long2wide_omv(dtaInp = nmeInp, fleOut = nmeOut, varID = "ID", varTme = "measure", varTgt = "X")
# it is required to give at least the arguments dtaInp, varID and varTme
# check whether the file was created and its size
cat(list.files(dirname(nmeOut), basename(nmeOut)))
# -> "file[...].omv" ([...] contains a random combination of numbers / characters
cat(file.info(nmeOut)$size)
# -> 6851 (approximate size; size may differ in every run [in dependence of
#          how well the generated random data can be compressed])
cat(str(read_omv(nmeOut, sveAtt = FALSE)))
# the data set is now transformed into wide (and each the measurements is now
# indicated as a suffix to X; X_1, X_2, ...)
# 'data.frame':	100 obs. of  9 variables:
#  $ ID : chr  "1" "2" "3" "4" "5" "6" "7" "8" "9" "10" ...
#   ..- attr(*, "jmv-id")= logi TRUE
#   ..- attr(*, "missingValues")= list()
#  $ X_1: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_2: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_3: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_4: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_5: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_6: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_7: num  ...
#   ..- attr(*, "missingValues")= list()
#  $ X_8: num  ...
#   ..- attr(*, "missingValues")= list()

unlink(nmeInp)
unlink(nmeOut)
}

}
