\name{ssaOptim}
\alias{ssaOptim}
\title{ Spatial simulated annealing (SSA) for optimization of sampling designs using a geostatistical measure of spatial prediction error }
\description{
Spatial simulated annealing uses slight perturbations of previous sampling designs and a random search technique to solve spatial optimization problems. 
Candidate measurement locations are iteratively moved around and optimized by minimizing the mean universal kriging variance (\code{\link{calculateMukv}}). The approach relies on a known, pre-specified
model for underlying spatial variation (\code{variogramModel}). }
\usage{
ssaOptim(observations, predGrid, candidates, action, nDiff, model,
         nr_iterations, plotOptim = TRUE, formulaString = NULL, 
         coolingFactor = nr_iterations/10, covariates = "over", ...)

}
\arguments{
\item{observations}{ object of class \code{\link[sp:Spatial-class]{Spatial}} with coordinates and possible covariates }
\item{predGrid}{ object of class \code{\link[sp:Spatial-class]{Spatial}}*
               used when \code{method = "ssa"}. \code{predGrid} should contain the coordinates 
               of prediction locations for optimization. Usually predGrid is a 
               \code{\link[sp:SpatialGrid]{SpatialGrid}} / \code{\link[sp:SpatialGrid]{SpatialPixels}} 
               or a \code{\link[sp:SpatialGrid]{SpatialGridDataFrame}} /  
               \code{\link[sp:SpatialGrid]{SpatialPixelsDataFrame}} when 
               independent covariate predictor variables are used }
\item{candidates}{ candidates is the study area of class 
               \code{\link[sp:SpatialPolygons]{SpatialPolygonsDataFrame}} }
\item{action}{ character string indicating which type of action to perform: 
         \code{"add"} to add new measurement stations to the existing network or 
         \code{"del"} to delete existing stations }
\item{nDiff}{ number of stations to add or delete }
\item{model}{ variogram model:object of class \code{variogramModel},
         as generated by \code{\link[gstat:vgm]{vgm}}}
\item{nr_iterations}{ number of iterations to process before stopping. The default
         \code{coolingFactor} is also a function of number of iterations. }
\item{plotOptim}{ logical; if TRUE, creates a plot of the result as optimization 
         progresses; TRUE by default }
\item{formulaString}{ formula that defines the dependent variable as a linear model 
      of independent variables; suppose the dependent variable has name \code{z}, 
      for ordinary and simple kriging use the formula \code{z~1}; 
      for universal kriging, suppose \code{z} is linearly dependent on 
      \code{x} and \code{y}, use the formula \code{z~x+y}. The formulaString defaults
      to \code{"value~1"} if \code{value} is a part of the data set. 
      If not, the first column of the data set is used. }
\item{coolingFactor}{ variable defining how fast the algorithm will cool down. With SSA, 
      worsening designs are accepted with a decreasing probability (generally set to
      \code{p} $<$ 0.2 to avoid selection of local minima). The \code{coolingFactor}
      dictates the rate at which \code{p} decreases to zero. Commonly \code{p} is set to 
      exponentially decrease or cool as a function of number of iterations to ensure convergence 
      (Brus & Heuvelink 2007, Melles et al. 2011). Smaller numbers give quicker cooling; 
      higher numbers give slower cooling. }
\item{covariates}{ character string defining whether possible covariates should be found 
         by "over" or "krige", see also details below }
\item{...}{ other arguments to be passed on to lower level functions }
}

\details{
This function applies spatial simulated annealing for the special case of 
optimization with the MUKV criterion (\code{\link{calculateMukv}}). 
With covariates, the function takes a universal kriging model into account. 
When optimizing a sampling design using SSA and \code{criterion = "mukv"}, 
measurement values at new sampling locations are not required in order to 
calculate prediction error variance (\code{criterion = "mukv"}). 
This attractive property allows one to estimate the kriging prediction error 
variance prior to collecting the data (i.e., the dependent variable 
is unknown at new candidate locations), and it is this property that is used in 
the SSA optimization procedure after (Brus & Heuvelink 2007, Melles et al. 2011).

A stopping criterion \code{countMax} is implemented in lower level functions to 
end the optimization procedure after 200 search steps 
have occurred without an improvement in the design. If this stopping criterion 
is reached before \code{nr_iterations}, SSA will terminate. 

\code{method = "ssa"} with \code{criterion = "mukv"} makes it possible to assume 
a linear relationship between independent variables in predGrid 
and dependent variables at observation locations using universal kriging 
(\code{\link[gstat:krige]{krige}}). However, a correct estimate of mean universal 
kriging variance requires that the \code{independent} covariate variables be 
known at candidate locations. Thus it is necessary to have complete spatial 
coverage for all covariate predictors in the linear model. Covariate information 
must be available at both new candidate measurement locations and prediction locations. 
This is not possible (except for the measurement locations themselves). 
Instead, these are estimated from the prediction locations.

There are two possible methods to attain information on covariates at the 
candidate locations, and the method can be set using the argument \code{covariates}: 
\code{\link[sp]{over}} and \code{\link[gstat]{krige}}. \code{\link[sp]{over}} finds the value of 
covariates at new locations by overlaying candidate locations on the prediction grid 
and taking the value of the nearest neighbour. The second method uses kriging to 
estimate covariate values at new locations from predGrid. The first method is 
generally faster, the second method is most likely more exact, particularly if 
the resolution of predGrid is low in relation to the spatial correlation lengths of the covariates. 
Both methods are approximations that may influence the criterion used for 
optimization with increasing numbers of points added. }

\value{
SpatialPointsDataFrame with optimized locations
}

\references{

[1] \url{http://www.intamap.org/}

[2] D. J. Brus, G. B. M. Heuvelink (2007). Optimization of sample patterns for universal 
kriging of environmental variables, Geoderma, 138: 86-95 (2007). 

[3] S. J. Melles, G. B. M. Heuvelink, C. J. W. Twenhofel, U. Stohlker (2011).	Optimizing 
the spatial pattern of networks for monitoring radioactive releases, Computers and Geosciences, 
37: 280-288 (2011).

}
\author{ O. Baume, S.J. Melles, J. Skoien }
\examples{
\dontrun{
# load data:
data(meuse)
coordinates(meuse) = ~x+y
data(meuse.grid)
coordinates(meuse.grid) = ~x+y
gridded(meuse.grid) = TRUE
predGrid = meuse.grid

# estimate variograms (OK/UK):
vfitOK = fit.variogram(variogram(zinc~1, meuse), vgm(1, "Exp", 300, 1))
vfitUK = fit.variogram(variogram(zinc~x+y, meuse), vgm(1, "Exp", 300, 1))
vfitRK = fit.variogram(variogram(zinc~dist+ffreq+soil, meuse), vgm(1, "Exp", 300, 1))

# study area of interest:
bb = bbox(predGrid)
boun = SpatialPoints(data.frame(x=c(bb[1,1],bb[1,2],bb[1,2],bb[1,1],bb[1,1]),
                                y=c(bb[2,1],bb[2,1],bb[2,2],bb[2,2],bb[2,1])))
Srl = Polygons(list(Polygon(boun)),ID = as.character(1))
candidates = SpatialPolygonsDataFrame(SpatialPolygons(list(Srl)),
                                      data = data.frame(ID=1))

# add 20 more points assuming OK model (SSA method):
optimOK <- ssaOptim(meuse, meuse.grid, candidates = candidates, covariates = "over",
            nDiff = 20, action = "add", model = vfitOK, nr_iterations = 10000, 
            formulaString = zinc~1, nmax = 40, countMax = 200)

# add 20 more points assuming UK model (SSA method):
optimUK <- ssaOptim(meuse, meuse.grid, candidates = candidates, covariates = "over",
            nDiff = 20, action = "add", model = vfitUK, nr_iterations = 10000, 
            formulaString = zinc~x+y, nmax = 40, countMax = 200)

# add 20 more points with auxiliary variables (SSA method):
optimRK <- ssaOptim(meuse, meuse.grid, candidates = candidates, covariates = "over",
            nDiff = 20, action = "add", model = vfitRK, nr_iterations = 10000, 
            formulaString = zinc~dist+ffreq+soil, nmax = 40, countMax = 200)
   }
   }

\keyword{spatial}
