% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R, R/fit_optim_split.R, R/print.R
\name{fit}
\alias{fit}
\alias{fit_optim_split}
\alias{print.incidence_fit}
\alias{print.incidence_fit_list}
\title{Fit exponential models to incidence data}
\usage{
fit(x, split = NULL, level = 0.95, quiet = FALSE)

fit_optim_split(x, window = x$timespan/4, plot = TRUE, quiet = TRUE,
  separate_split = TRUE)

\method{print}{incidence_fit}(x, ...)

\method{print}{incidence_fit_list}(x, ...)
}
\arguments{
\item{x}{An incidence object, generated by the function
\code{\link[=incidence]{incidence()}}. For the plotting function, an \code{incidence_fit}
object.}

\item{split}{An optional time point identifying the separation between the
two models. If NULL, a single model is fitted. If provided, two models would
be fitted on the time periods on either side of the split.}

\item{level}{The confidence interval to be used for predictions; defaults to
95\%.}

\item{quiet}{A logical indicating if warnings from \code{fit} should be
hidden; FALSE by default. Warnings typically indicate some zero incidence,
which are removed before performing the log-linear regression.}

\item{window}{The size, in days, of the time window either side of the
split.}

\item{plot}{A logical indicating whether a plot should be added to the
output (\code{TRUE}, default), showing the mean R2 for various splits.}

\item{separate_split}{If groups are present, should separate split dates be
determined for each group? Defaults to \code{TRUE}, in which separate split dates
and thus, separate models will be constructed for each group. When \code{FALSE},
the split date will be determined from the pooled data and modelled with the
groups as main effects and interactions with date.}

\item{...}{currently unused.}
}
\value{
For \code{fit()}, a list with the class \code{incidence_fit} (for a
single model), or a list containing two \code{incidence_fit} objects (when
fitting two models). \code{incidence_fit} objects contain:
\itemize{
\item \code{$model}: the fitted linear model
\item \code{$info}: a list containing various information extracted from the model
(detailed further)
\item \code{$origin}: the date corresponding to day '0'
}

The \code{$info} item is a list containing:
\itemize{
\item \code{r}: the growth rate
\item \code{r.conf}: the confidence interval of 'r'
\item \code{pred}: a \code{data.frame} containing predictions of the model,
including the true dates (\code{dates}), their numeric version used in the
model (\code{dates.x}), the predicted value (\code{fit}), and the lower
(\code{lwr}) and upper (\code{upr}) bounds of the associated confidence
interval.
\item \code{doubling}: the predicted doubling time in days; exists only if 'r' is
positive
\item \code{doubling.conf}: the confidence interval of the doubling time
\item \code{halving}: the predicted halving time in days; exists only if 'r' is
negative
\item \code{halving.conf}: the confidence interval of the halving time
}

For \code{fit_optim_split}, a list containing:
\itemize{
\item \code{df}: a \code{data.frame} of dates that were used in the optimization
procedure, and the corresponding average R2 of the resulting models.
\item \code{split}: the optimal splitting date
\item \code{fit}: an \code{incidence_fit_list} object containing the fit for each split.
If the \code{separate_split = TRUE}, then the \code{incidence_fit_list} object will
contain these splits nested within each group. All of the \code{incidence_fit}
objects can be retrieved with \code{\link[=get_fit]{get_fit()}}.
\item \code{plot}: a plot showing the content of \code{df} (ggplot2 object)
}
}
\description{
The function \code{fit} fits two exponential models to incidence data, of the
form: \eqn{log(y) = r * t + b} \cr where 'y' is the incidence, 't' is time
(in days), 'r' is the growth rate, and 'b' is the origin. The function \code{fit}
will fit one model by default, but will fit two models on either side of a
splitting date (typically the peak of the epidemic) if the argument \code{split}
is provided. When groups are present, these are included in the model as main
effects and interactions with dates. The function \code{fit_optim_split()} can be
used to find the optimal 'splitting' date, defined as the one for which the
best average R2 of the two models is obtained. Plotting can be done using
\code{plot}, or added to an existing incidence plot by the piping-friendly
function \code{add_incidence_fit()}.
}
\examples{

if (require(outbreaks)) { withAutoprint({
 dat <- ebola_sim$linelist$date_of_onset

 ## EXAMPLE WITH A SINGLE MODEL

 ## compute weekly incidence
 i.7 <- incidence(dat, interval=7)
 plot(i.7)
 plot(i.7[1:20])

 ## fit a model on the first 20 weeks
 f <- fit(i.7[1:20])
 f
 names(f)
 head(get_info(f, "pred"))

 ## plot model alone (not recommended)
 plot(f)

 ## plot data and model (recommended)
 plot(i.7, fit = f)
 plot(i.7[1:25], fit = f)

## piping versions
if (require(magrittr)) { withAutoprint({
  plot(i.7) \%>\% add_incidence_fit(f)


  ## EXAMPLE WITH 2 PHASES
  ## specifying the peak manually
  f2 <- fit(i.7, split = as.Date("2014-10-15"))
  f2
  plot(i.7) \%>\% add_incidence_fit(f2)

  ## finding the best 'peak' date
  f3 <- fit_optim_split(i.7)
  f3
  plot(i.7) \%>\% add_incidence_fit(f3$fit)
})}

})}

}
\seealso{
the \code{\link[=incidence]{incidence()}} function to generate the 'incidence'
objects. The \code{\link[=get_fit]{get_fit()}} function to flatten \code{incidence_fit_list} objects to
a list of \code{incidence_fit} objects.
}
\author{
Thibaut Jombart \email{thibautjombart@gmail.com}, Zhian N. Kamvar
\email{zkamvar@gmail.com}.
}
