/*
 * Copyright (c) 2010, Guoshen Yu <yu@cmap.polytechnique.fr>,
 *                     Guillermo Sapiro <guille@umn.edu>
 * All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*----------------------------------- DCT2D ---------------------------------*/
// This code implements "DCT image denoising: a simple and effective image 
// denoising algorithm".
// http://www.ipol.im/pub/algo/ys_dct_denoising
// Copyright, Guoshen Yu, Guillermo Sapiro, 2010.
// Please report bugs and/or send comments to Guoshen Yu 
// yu@cmap.polytechnique.fr
/*---------------------------------------------------------------------------*/

#include <Rcpp.h>

# define PATCHSIZE8 8


// Define a 8x8 1D DCT basis (each ROW is a vector of the forward
// transform basis).
const double DCTbasis8[PATCHSIZE8][PATCHSIZE8] = {
    {   0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,
        0.35355339059327373085750423342688009142875671386719,   },

    {   0.49039264020161521528962111915461719036102294921875,
      0.41573480615127261783570133957255166023969650268555,
      0.27778511650980114433551193542371038347482681274414,
      0.09754516100806412404189416065491968765854835510254,
      -0.09754516100806412404189416065491968765854835510254,
      -0.27778511650980114433551193542371038347482681274414,
      -0.41573480615127261783570133957255166023969650268555,
      -0.49039264020161521528962111915461719036102294921875,    },

    {   0.46193976625564336924156805253005586564540863037109,
      0.19134171618254489088961634024599334225058555603027,
      -0.19134171618254489088961634024599334225058555603027,
      -0.46193976625564336924156805253005586564540863037109,
      -0.46193976625564336924156805253005586564540863037109,
      -0.19134171618254489088961634024599334225058555603027,
      0.19134171618254489088961634024599334225058555603027,
      0.46193976625564336924156805253005586564540863037109,     },

    {   0.41573480615127261783570133957255166023969650268555,
      -0.09754516100806417955304539191274670884013175964355,
      -0.49039264020161521528962111915461719036102294921875,
      -0.27778511650980108882436070416588336229324340820312,
      0.27778511650980108882436070416588336229324340820312,
      0.49039264020161521528962111915461719036102294921875,
      0.09754516100806417955304539191274670884013175964355,
      -0.41573480615127261783570133957255166023969650268555,    },

    {   0.35355339059327378636865546468470711261034011840820,
      -0.35355339059327378636865546468470711261034011840820,
      -0.35355339059327378636865546468470711261034011840820,
      0.35355339059327378636865546468470711261034011840820,
      0.35355339059327378636865546468470711261034011840820,
      -0.35355339059327378636865546468470711261034011840820,
      -0.35355339059327378636865546468470711261034011840820,
      0.35355339059327378636865546468470711261034011840820,     },

    {   0.27778511650980114433551193542371038347482681274414,
      -0.49039264020161532631192358167027123272418975830078,
      0.09754516100806412404189416065491968765854835510254,
      0.41573480615127261783570133957255166023969650268555,
      -0.41573480615127261783570133957255166023969650268555,
      -0.09754516100806412404189416065491968765854835510254,
      0.49039264020161532631192358167027123272418975830078,
      -0.27778511650980114433551193542371038347482681274414,    },

    {   0.19134171618254491864519195587490685284137725830078,
      -0.46193976625564336924156805253005586564540863037109,
      0.46193976625564336924156805253005586564540863037109,
      -0.19134171618254491864519195587490685284137725830078,
      -0.19134171618254491864519195587490685284137725830078,
      0.46193976625564336924156805253005586564540863037109,
      -0.46193976625564336924156805253005586564540863037109,
      0.19134171618254491864519195587490685284137725830078,     },

    {   0.09754516100806416567525758409828995354473590850830,
      -0.27778511650980108882436070416588336229324340820312,
      0.41573480615127267334685257083037868142127990722656,
      -0.49039264020161521528962111915461719036102294921875,
      0.49039264020161521528962111915461719036102294921875,
      -0.41573480615127267334685257083037868142127990722656,
      0.27778511650980108882436070416588336229324340820312,
      -0.09754516100806416567525758409828995354473590850830     }
};


// 1D DCT transform of a signal of size 8x1.
// flag: 1/-1 forward/inverse transforms.
void DCT1D(std::vector< double >& in, std::vector< double >& out, int flag)
{
    // forward transform
    if (flag == 1) 
    {
        for (int j = 0; j < PATCHSIZE8; ++j) 
    {
            out[j] = 0;
            for (int i = 0; i < PATCHSIZE8; ++i) 
            {
                out[j] += in[i] * DCTbasis8[j][i];
            }
        }
    }
    // reverse transform
    else if (flag == -1) 
    {
        for (int j = 0; j < PATCHSIZE8; ++j) 
       {
            out[j] = 0;
            for (int i = 0; i < PATCHSIZE8; ++i) 
            {
                out[j] += in[i] * DCTbasis8[i][j];
            }
        }
    }
}


// 2D DCT of a 8x8 patches. The result is restored in-place.
// flag: 1/-1 forward/inverse transforms.
void DCT2D(std::vector< std::vector< double > >& patch1, int flag)
{
    std::vector< std::vector< double > > tmp1;
    tmp1.resize(PATCHSIZE8);
    std::vector< std::vector< double > > tmp2;
    tmp2.resize(PATCHSIZE8);
    for (int i = 0; i < PATCHSIZE8; ++i) 
    {
        tmp1[i].resize(PATCHSIZE8);
        tmp2[i].resize(PATCHSIZE8);
    }

    // transform row by row
    for (int j = 0; j < PATCHSIZE8; ++j) 
    {
        DCT1D(patch1[j], tmp1[j], flag);
    }

    // transform column by column
    // (by transposing the matrix,
    // transforming row by row, and
    // transposing again the matrix.)
    for (int j = 0; j < PATCHSIZE8; ++j) 
    {
        for (int i = 0; i < PATCHSIZE8; ++i)
        {
            tmp2[j][i] = tmp1[i][j];
        }
    }
    for (int j = 0; j < PATCHSIZE8; ++j) 
    {
        DCT1D(tmp2[j], tmp1[j], flag);
    }
    for (int j = 0; j < PATCHSIZE8; ++j) 
    {
        for (int i = 0; i < PATCHSIZE8; ++i)
        {
            patch1[j][i] = tmp1[i][j];
        }
    }
}


/*
 * Copyright (c) 2010, Guoshen Yu <yu@cmap.polytechnique.fr>,
 *                     Guillermo Sapiro <guille@umn.edu>
 * All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


/*----------------------------------- DCT2D ---------------------------------*/
// This code implements "DCT image denoising: a simple and effective image 
// denoising algorithm".
// http://www.ipol.im/pub/algo/ys_dct_denoising
// Copyright, Guoshen Yu, Guillermo Sapiro, 2010.
// Please report bugs and/or send comments to Guoshen Yu 
// yu@cmap.polytechnique.fr
/*---------------------------------------------------------------------------*/

# define PATCHSIZE16 16


// Define a 16x16 1D DCT basis (each ROW is a vector of the forward
// transform basis).
const double DCTbasis16[PATCHSIZE16][PATCHSIZE16] = {
    {   0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,
        0.25000000000000000000000000000000000000000000000000,   },

    {   0.35185093438159564804834644746733829379081726074219,
      0.33832950029358815902114088203234132379293441772461,
      0.31180625324666783049210039280296768993139266967773,
      0.27330046675043939696791994720115326344966888427734,
      0.22429189658565909515353098413470434024930000305176,
      0.16666391461943663721001485100714489817619323730469,
      0.10263113188058936575686175274313427507877349853516,
      0.03465429229977286945674208595846721436828374862671,
      -0.03465429229977286945674208595846721436828374862671,
      -0.10263113188058936575686175274313427507877349853516,
      -0.16666391461943663721001485100714489817619323730469,
      -0.22429189658565909515353098413470434024930000305176,
      -0.27330046675043939696791994720115326344966888427734,
      -0.31180625324666783049210039280296768993139266967773,
      -0.33832950029358815902114088203234132379293441772461,
      -0.35185093438159564804834644746733829379081726074219,},


    {   0.34675996133053688108560663749813102185726165771484,
      0.29396890060483971129912106334813870489597320556641,
      0.19642373959677555950165128706430550664663314819336,
      0.06897484482073575062788250988887739367783069610596,
      -0.06897484482073575062788250988887739367783069610596,
      -0.19642373959677555950165128706430550664663314819336,
      -0.29396890060483971129912106334813870489597320556641,
      -0.34675996133053688108560663749813102185726165771484,
      -0.34675996133053688108560663749813102185726165771484,
      -0.29396890060483971129912106334813870489597320556641,
      -0.19642373959677555950165128706430550664663314819336,
      -0.06897484482073575062788250988887739367783069610596,
      0.06897484482073575062788250988887739367783069610596,
      0.19642373959677555950165128706430550664663314819336,
      0.29396890060483971129912106334813870489597320556641,
      0.34675996133053688108560663749813102185726165771484,},

    {   0.33832950029358815902114088203234132379293441772461,
      0.22429189658565906739795536850579082965850830078125,
      0.03465429229977286251784818205123883672058582305908,
      -0.16666391461943663721001485100714489817619323730469,
      -0.31180625324666783049210039280296768993139266967773,
      -0.35185093438159559253719521620951127260923385620117,
      -0.27330046675043934145676871594332624226808547973633,
      -0.10263113188058933800128613711422076448798179626465,
      0.10263113188058933800128613711422076448798179626465,
      0.27330046675043934145676871594332624226808547973633,
      0.35185093438159559253719521620951127260923385620117,
      0.31180625324666783049210039280296768993139266967773,
      0.16666391461943663721001485100714489817619323730469,
      -0.03465429229977286251784818205123883672058582305908,
      -0.22429189658565906739795536850579082965850830078125,
      -0.33832950029358815902114088203234132379293441772461,},

    {   0.32664074121909414394338000420248135924339294433594,
      0.13529902503654925305376366395648801699280738830566,
      -0.13529902503654925305376366395648801699280738830566,
      -0.32664074121909414394338000420248135924339294433594,
      -0.32664074121909414394338000420248135924339294433594,
      -0.13529902503654925305376366395648801699280738830566,
      0.13529902503654925305376366395648801699280738830566,
      0.32664074121909414394338000420248135924339294433594,
      0.32664074121909414394338000420248135924339294433594,
      0.13529902503654925305376366395648801699280738830566,
      -0.13529902503654925305376366395648801699280738830566,
      -0.32664074121909414394338000420248135924339294433594,
      -0.32664074121909414394338000420248135924339294433594,
      -0.13529902503654925305376366395648801699280738830566,
      0.13529902503654925305376366395648801699280738830566,
      0.32664074121909414394338000420248135924339294433594,},

    {   0.31180625324666783049210039280296768993139266967773,
      0.03465429229977286251784818205123883672058582305908,
      -0.27330046675043939696791994720115326344966888427734,
      -0.33832950029358821453229211329016834497451782226562,
      -0.10263113188058936575686175274313427507877349853516,
      0.22429189658565906739795536850579082965850830078125,
      0.35185093438159564804834644746733829379081726074219,
      0.16666391461943663721001485100714489817619323730469,
      -0.16666391461943663721001485100714489817619323730469,
      -0.35185093438159564804834644746733829379081726074219,
      -0.22429189658565906739795536850579082965850830078125,
      0.10263113188058936575686175274313427507877349853516,
      0.33832950029358821453229211329016834497451782226562,
      0.27330046675043939696791994720115326344966888427734,
      -0.03465429229977286251784818205123883672058582305908,
      -0.31180625324666783049210039280296768993139266967773,},

    {   0.29396890060483971129912106334813870489597320556641,
      -0.06897484482073579226124593333224765956401824951172,
      -0.34675996133053688108560663749813102185726165771484,
      -0.19642373959677553174607567143539199605584144592285,
      0.19642373959677553174607567143539199605584144592285,
      0.34675996133053688108560663749813102185726165771484,
      0.06897484482073579226124593333224765956401824951172,
      -0.29396890060483971129912106334813870489597320556641,
      -0.29396890060483971129912106334813870489597320556641,
      0.06897484482073579226124593333224765956401824951172,
      0.34675996133053688108560663749813102185726165771484,
      0.19642373959677553174607567143539199605584144592285,
      -0.19642373959677553174607567143539199605584144592285,
      -0.34675996133053688108560663749813102185726165771484,
      -0.06897484482073579226124593333224765956401824951172,
      0.29396890060483971129912106334813870489597320556641,},

    {   0.27330046675043939696791994720115326344966888427734,
      -0.16666391461943663721001485100714489817619323730469,
      -0.33832950029358821453229211329016834497451782226562,
      0.03465429229977287639563598986569559201598167419434,
      0.35185093438159564804834644746733829379081726074219,
      0.10263113188058935187907394492867751978337764739990,
      -0.31180625324666783049210039280296768993139266967773,
      -0.22429189658565909515353098413470434024930000305176,
      0.22429189658565909515353098413470434024930000305176,
      0.31180625324666783049210039280296768993139266967773,
      -0.10263113188058935187907394492867751978337764739990,
      -0.35185093438159564804834644746733829379081726074219,
      -0.03465429229977287639563598986569559201598167419434,
      0.33832950029358821453229211329016834497451782226562,
      0.16666391461943663721001485100714489817619323730469,
      -0.27330046675043939696791994720115326344966888427734,},

    {   0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      -0.25000000000000005551115123125782702118158340454102,
      0.25000000000000005551115123125782702118158340454102,},

    {   0.22429189658565909515353098413470434024930000305176,
      -0.31180625324666783049210039280296768993139266967773,
      -0.10263113188058937963464956055759103037416934967041,
      0.35185093438159564804834644746733829379081726074219,
      -0.03465429229977282088448475860786857083439826965332,
      -0.33832950029358815902114088203234132379293441772461,
      0.16666391461943658169886361974931787699460983276367,
      0.27330046675043934145676871594332624226808547973633,
      -0.27330046675043934145676871594332624226808547973633,
      -0.16666391461943658169886361974931787699460983276367,
      0.33832950029358815902114088203234132379293441772461,
      0.03465429229977282088448475860786857083439826965332,
      -0.35185093438159564804834644746733829379081726074219,
      0.10263113188058937963464956055759103037416934967041,
      0.31180625324666783049210039280296768993139266967773,
      -0.22429189658565909515353098413470434024930000305176,},

    {   0.19642373959677558725722690269321901723742485046387,
      -0.34675996133053693659675786875595804303884506225586,
      0.06897484482073573675009470207442063838243484497070,
      0.29396890060483971129912106334813870489597320556641,
      -0.29396890060483971129912106334813870489597320556641,
      -0.06897484482073573675009470207442063838243484497070,
      0.34675996133053693659675786875595804303884506225586,
      -0.19642373959677558725722690269321901723742485046387,
      -0.19642373959677558725722690269321901723742485046387,
      0.34675996133053693659675786875595804303884506225586,
      -0.06897484482073573675009470207442063838243484497070,
      -0.29396890060483971129912106334813870489597320556641,
      0.29396890060483971129912106334813870489597320556641,
      0.06897484482073573675009470207442063838243484497070,
      -0.34675996133053693659675786875595804303884506225586,
      0.19642373959677558725722690269321901723742485046387,},

    {   0.16666391461943669272116608226497191935777664184570,
      -0.35185093438159559253719521620951127260923385620117,
      0.22429189658565901188680413724796380847692489624023,
      0.10263113188058929636792271367085049860179424285889,
      -0.33832950029358821453229211329016834497451782226562,
      0.27330046675043939696791994720115326344966888427734,
      0.03465429229977293190678722112352261319756507873535,
      -0.31180625324666777498094916154514066874980926513672,
      0.31180625324666777498094916154514066874980926513672,
      -0.03465429229977293190678722112352261319756507873535,
      -0.27330046675043939696791994720115326344966888427734,
      0.33832950029358821453229211329016834497451782226562,
      -0.10263113188058929636792271367085049860179424285889,
      -0.22429189658565901188680413724796380847692489624023,
      0.35185093438159559253719521620951127260923385620117,
      -0.16666391461943669272116608226497191935777664184570,},

    {   0.13529902503654928080933927958540152758359909057617,
      -0.32664074121909414394338000420248135924339294433594,
      0.32664074121909414394338000420248135924339294433594,
      -0.13529902503654928080933927958540152758359909057617,
      -0.13529902503654928080933927958540152758359909057617,
      0.32664074121909414394338000420248135924339294433594,
      -0.32664074121909414394338000420248135924339294433594,
      0.13529902503654928080933927958540152758359909057617,
      0.13529902503654928080933927958540152758359909057617,
      -0.32664074121909414394338000420248135924339294433594,
      0.32664074121909414394338000420248135924339294433594,
      -0.13529902503654928080933927958540152758359909057617,
      -0.13529902503654928080933927958540152758359909057617,
      0.32664074121909414394338000420248135924339294433594,
      -0.32664074121909414394338000420248135924339294433594,
      0.13529902503654928080933927958540152758359909057617,},

    {   0.10263113188058933800128613711422076448798179626465,
      -0.27330046675043939696791994720115326344966888427734,
      0.35185093438159564804834644746733829379081726074219,
      -0.31180625324666788600325162406079471111297607421875,
      0.16666391461943666496559046663605840876698493957520,
      0.03465429229977289027342379768015234731137752532959,
      -0.22429189658565906739795536850579082965850830078125,
      0.33832950029358821453229211329016834497451782226562,
      -0.33832950029358821453229211329016834497451782226562,
      0.22429189658565906739795536850579082965850830078125,
      -0.03465429229977289027342379768015234731137752532959,
      -0.16666391461943666496559046663605840876698493957520,
      0.31180625324666788600325162406079471111297607421875,
      -0.35185093438159564804834644746733829379081726074219,
      0.27330046675043939696791994720115326344966888427734,
      -0.10263113188058933800128613711422076448798179626465,},

    {   0.06897484482073577838345812551779090426862239837646,
      -0.19642373959677555950165128706430550664663314819336,
      0.29396890060483971129912106334813870489597320556641,
      -0.34675996133053688108560663749813102185726165771484,
      0.34675996133053688108560663749813102185726165771484,
      -0.29396890060483971129912106334813870489597320556641,
      0.19642373959677555950165128706430550664663314819336,
      -0.06897484482073577838345812551779090426862239837646,
      -0.06897484482073577838345812551779090426862239837646,
      0.19642373959677555950165128706430550664663314819336,
      -0.29396890060483971129912106334813870489597320556641,
      0.34675996133053688108560663749813102185726165771484,
      -0.34675996133053688108560663749813102185726165771484,
      0.29396890060483971129912106334813870489597320556641,
      -0.19642373959677555950165128706430550664663314819336,
      0.06897484482073577838345812551779090426862239837646,},

    {   0.03465429229977292496789331721629423554986715316772,
      -0.10263113188058928249013490585639374330639839172363,
      0.16666391461943666496559046663605840876698493957520,
      -0.22429189658565903964237975287687731906771659851074,
      0.27330046675043939696791994720115326344966888427734,
      -0.31180625324666771946979793028731364756822586059570,
      0.33832950029358815902114088203234132379293441772461,
      -0.35185093438159559253719521620951127260923385620117,
      0.35185093438159559253719521620951127260923385620117,
      -0.33832950029358815902114088203234132379293441772461,
      0.31180625324666771946979793028731364756822586059570,
      -0.27330046675043939696791994720115326344966888427734,
      0.22429189658565903964237975287687731906771659851074,
      -0.16666391461943666496559046663605840876698493957520,
      0.10263113188058928249013490585639374330639839172363,
      -0.03465429229977292496789331721629423554986715316772}
};


// 1D DCT transform of a signal of size 8x1.
// flag: 1/-1 forward/inverse transforms.
void DCT1D16(std::vector< double >& in, std::vector< double >& out, int flag)
{
    // forward transform
    if (flag == 1) 
    {
        for (int j = 0; j < PATCHSIZE16; ++j) 
        {
            out[j] = 0;
            for (int i = 0; i < PATCHSIZE16; ++i) 
            {
                out[j] += in[i] * DCTbasis16[j][i];
            }
        }
    }
    // reverse transform
    else if (flag == -1) 
    {
        for (int j = 0; j < PATCHSIZE16; ++j) 
        {
            out[j] = 0;
            for (int i = 0; i < PATCHSIZE16; ++i) 
            {
                out[j] += in[i] * DCTbasis16[i][j];
            }
        }
    }
}


// 2D DCT of a 16x16 patches. The result is restored in-place.
// flag: 1/-1 forward/inverse transforms.
void DCT2D16x16(std::vector< std::vector< double > >& patch1, int flag)
{
    std::vector< std::vector< double > > tmp1;
    tmp1.resize(PATCHSIZE16);
    std::vector< std::vector< double > > tmp2;
    tmp2.resize(PATCHSIZE16);
    for (int i = 0; i < PATCHSIZE16; ++i) 
    {
        tmp1[i].resize(PATCHSIZE16);
        tmp2[i].resize(PATCHSIZE16);
    }

    // transform row by row
    for (int j = 0; j < PATCHSIZE16; ++j) 
    {
        DCT1D16(patch1[j], tmp1[j], flag);
    }

    // transform column by column
    // (by transposing the matrix,
    // transforming row by row, and
    // transposing again the matrix.)
    for (int j = 0; j < PATCHSIZE16; ++j) 
    {
        for (int i = 0; i < PATCHSIZE16; ++i)
        {
            tmp2[j][i] = tmp1[i][j];
        }
    }
    for (int j = 0; j < PATCHSIZE16; ++j) 
    {
        DCT1D16(tmp2[j], tmp1[j], flag);
    }
    for (int j = 0; j < PATCHSIZE16; ++j) 
    {
        for (int i = 0; i < PATCHSIZE16; ++i) 
        {
            patch1[j][i] = tmp1[i][j];
        }
    }
}

/*
 * Copyright (c) 2010, Guoshen Yu <yu@cmap.polytechnique.fr>,
 *                     Guillermo Sapiro <guille@umn.edu>
 * All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


/*--------------------------- DCTdenoising  -------------------------*/
// This code implements "DCT image denoising: a simple and effective image 
// denoising algorithm".
// http://www.ipol.im/pub/algo/ys_dct_denoising
// Copyright, Guoshen Yu, Guillermo Sapiro, 2010.
// Please report bugs and/or send comments to Guoshen Yu 
// yu@cmap.polytechnique.fr
/*---------------------------------------------------------------------------*/

#define ABS(x)    (((x) > 0) ? (x) : (-(x)))

// Define a 3x3 1D DCT basis (each ROW is a vector of the forward
// transform basis).
/*
const double DCTbasis3x3[3][3] = {
    {   0.5773502588272094726562500000000000000000,
        0.5773502588272094726562500000000000000000,
        0.5773502588272094726562500000000000000000,     },

    {   0.7071067690849304199218750000000000000000,
      0.0000000000000000000000000000000000000000,
      -0.7071067690849304199218750000000000000000, },

    {
        0.4082483053207397460937500000000000000000,
        -0.8164966106414794921875000000000000000000,
        0.4082483053207397460937500000000000000000      }
};
*/

void Image2Patches(std::vector<double>&, 
     std::vector< std::vector< std::vector< std::vector< double > > > >&, int, int, int, int, int);
void Patches2Image(std::vector<double>&, 
     std::vector< std::vector< std::vector< std::vector< double > > > >&, int, int, int, int, int);

// Denoise an image with sliding DCT thresholding.
// ipixels, opixels: noisy and denoised images.
// width, height, channel: image width, height and number of channels.
// sigma: standard deviation of Gaussian white noise in ipixels.
// [[Rcpp::export]]
Rcpp::NumericMatrix DCTdenoising(Rcpp::NumericMatrix ipixelsR, int width, int height, double sigma, int flag_dct16x16)
{
    //Convert ipixelsR to std::vector<double>
    int size_ipixels = width * height;
    std::vector<double> ipixels;
    ipixels.resize(size_ipixels);
    for(int i = 0; i < height; ++i)
    {
        for (int j = 0; j < width; ++j)
        {
            ipixels[i * width + j] = ipixelsR(i, j);
        }
    }

    // Threshold
    double Th = 3 * sigma;

    // DCT window size
    int width_p, height_p;
    if (flag_dct16x16 == 0) {
        width_p = 16;
        height_p = 16;
    } else {
        width_p = 8;
        height_p = 8;
    }

    int num_patches = (width - width_p + 1) * (height - height_p + 1);
    int channel = 1;

    std::vector< std::vector< std::vector< std::vector< double > > > > patches;
    patches.resize(num_patches);
    for (int p = 0; p < num_patches; p ++) {
        patches[p].resize(channel);
        for (int k = 0; k < channel; k ++) {
            patches[p][k].resize(height_p);
            for (int j = 0; j < height_p; j ++)
                patches[p][k][j].resize(width_p);
        }
    }
    Image2Patches(ipixels, patches, width, height, channel, width_p, height_p);

    // 2D DCT forward
    for (int p = 0; p < num_patches; p ++) {
        for (int k = 0; k < channel; k ++) {
            if (flag_dct16x16 == 0)
                DCT2D16x16(patches[p][k], 1);
            else
                DCT2D(patches[p][k], 1);
        }
    }

    // Thresholding
    for (int p = 0; p < num_patches; p ++)
        for (int k = 0; k < channel; k ++)
            for (int j = 0; j < height_p; j ++)
                for (int i = 0; i < width_p; i ++) {
                    if ( ABS(patches[p][k][j][i]) < Th )
                        patches[p][k][j][i] = 0;
                }

    // 2D DCT inverse
    for (int p = 0; p < num_patches; p ++) {
        for (int k = 0; k < channel; k ++) {
            if (flag_dct16x16 == 0)
            {
                DCT2D16x16(patches[p][k], -1);
            } else {
                DCT2D(patches[p][k], -1);
            }
        }
    }

    Patches2Image(ipixels, patches, width, height, channel, width_p, height_p);

    Rcpp::NumericMatrix res(height, width);
    for(int i = 0; i < height; ++i)
    {
        for (int j = 0; j < width; ++j)
        {
            res(i ,j) = ipixels[i * width + j];
        }
    }
    return res;
}

// Transfer an image im of size width x height x channel to sliding patches of 
// size width_p x height_p xchannel.
// The patches are stored in patches, where each ROW is a patch after being 
// reshaped to a vector.
void Image2Patches(std::vector<double>& im, std::vector< std::vector< std::vector< std::vector< double > > > >& patches, int width, int height, int channel, int width_p, int height_p)
{
    int size1 = width * height;
    int counter_patch = 0;

    // Loop over the patch positions
    for (int j = 0; j < height - height_p + 1; ++j)
    {
        for (int i = 0; i < width - width_p + 1; ++i) 
        {
            int counter_pixel = 0;
            // loop over the pixels in the patch
            for (int kp = 0; kp < channel; ++kp)
            {
                for (int jp = 0; jp < height_p; ++jp)
                {
                    for (int ip = 0; ip < width_p; ++ip) 
                    {
                        patches[counter_patch][kp][jp][ip] = 
                                         im[kp*size1 + (j+jp)*width + i + ip];
                        ++counter_pixel;
                    }
                }
            }
            ++counter_patch;
        }
    }
}

// Transfer sliding patches of size width_p x height_p xchannel to an image im 
// of size width x height x channel.
// The patches are stored in patches, where each ROW is a patch after being 
// reshaped to a vector.
void Patches2Image(std::vector<double>& im, std::vector< std::vector< std::vector< std::vector< double > > > >& patches, int width, int height, int channel, int width_p, int height_p)
{
    int size1 = width * height;
    int size = size1 * channel;

    // clean the image
    for (int i = 0; i < size; ++i)
    {
        im[i] = 0;
    }

    // Store the weight
    std::vector<double> im_weight;
    im_weight.resize(size);
    for (int i = 0; i < size; ++i)
    {
        im_weight[i] = 0;
      }

    int counter_patch = 0;

    // Loop over the patch positions
    for (int j = 0; j < height - height_p + 1; ++j)
    {
        for (int i = 0; i < width - width_p + 1; ++i) 
        {
            int counter_pixel = 0;
            // loop over the pixels in the patch
            for (int kp = 0; kp < channel; ++kp)
            {
                for (int jp = 0; jp < height_p; ++jp)
                {
                    for (int ip = 0; ip < width_p; ++ip) 
                    {
                        im[kp*size1 + (j+jp)*width + i + ip] += 
                                       patches[counter_patch][kp][jp][ip];
                        ++im_weight[kp*size1 + (j+jp)*width + i + ip];
                        ++counter_pixel;
                    }
                }
            }
            ++counter_patch;
        }
    }

    // Normalize by the weight
    for (int i = 0; i < size; ++i)
    {
        im[i] = im[i] / im_weight[i];
    }
}
