% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iterators.R
\name{igraph-vs-indexing}
\alias{igraph-vs-indexing}
\alias{[.igraph.vs}
\title{Indexing vertex sequences}
\usage{
\method{[}{igraph.vs}(x, ..., na_ok = FALSE)
}
\arguments{
\item{x}{A vertex sequence.}

\item{...}{Indices, see details below.}

\item{na_ok}{Whether it is OK to have \code{NA}s in the vertex
sequence.}
}
\value{
Another vertex sequence, referring to the same graph.
}
\description{
Vertex sequences can be indexed very much like a plain numeric R vector,
with some extras.
}
\details{
Vertex sequences can be indexed using both the single bracket and
the double bracket operators, and they both work the same way.
The only difference between them is that the double bracket operator
marks the result for printing vertex attributes.
}
\section{Multiple indices}{

When using multiple indices within the bracket, all of them
are evaluated independently, and then the results are concatenated
using the \code{c()} function (except for the \code{na_ok} argument,
which is special an must be named. E.g. \code{V(g)[1, 2, .nei(1)]}
is equivalent to \code{c(V(g)[1], V(g)[2], V(g)[.nei(1)])}.
}

\section{Index types}{

Vertex sequences can be indexed with positive numeric vectors,
negative numeric vectors, logical vectors, character vectors:
\itemize{
  \item When indexed with positive numeric vectors, the vertices at the
    given positions in the sequence are selected. This is the same as
    indexing a regular R atomic vector with positive numeric vectors.
  \item When indexed with negative numeric vectors, the vertices at the
    given positions in the sequence are omitted. Again, this is the same
    as indexing a regular R atomic vector.
  \item When indexed with a logical vector, the lengths of the vertex
    sequence and the index must match, and the vertices for which the
    index is \code{TRUE} are selected.
  \item Named graphs can be indexed with character vectors,
    to select vertices with the given names.
}
}

\section{Vertex attributes}{

When indexing vertex sequences, vertex attributes can be refered
to simply by using their names. E.g. if a graph has a \code{name} vertex
attribute, then \code{V(g)[name == "foo"]} is equivalent to
\code{V(g)[V(g)$name == "foo"]}. See examples below.
}

\section{Special functions}{

There are some special igraph functions that can be used only
in expressions indexing vertex sequences: \describe{
  \item{\code{.nei}}{takes a vertex sequence as its argument
    and selects neighbors of these vertices. An optional \code{mode}
    argument can be used to select successors (\code{mode="out"}), or
    precedessors (\code{mode="in"}) in directed graphs.}
  \item{\code{.inc}}{Takes an edge sequence as an argument, and
    selects vertices that have at least one incident edge in this
    edge sequence.}
  \item{\code{.from}}{Similar to \code{.inc}, but only considers the
    tails of the edges.}
  \item{\code{.to}}{Similar to \code{.inc}, but only considers the
    heads of the edges.}
  \item{\code{.innei}, \code{.outnei}}{\code{.innei(v)} is a shorthand for
    \code{.nei(v, mode = "in")}, and \code{.outnei(v)} is a shorthand for
    \code{.nei(v, mode = "out")}.
  }
}
Note that multiple special functions can be used together, or with
regular indices, and then their results are concatenated. See more
examples below.
}

\examples{
# -----------------------------------------------------------------
# Setting attributes for subsets of vertices
largest_comp <- function(graph) {
  cl <- components(graph)
  V(graph)[which.max(cl$csize) == cl$membership]
}
g <- sample_(gnp(100, 2/100),
  with_vertex_(size = 3, label = ""),
  with_graph_(layout = layout_with_fr)
)
giant_v <- largest_comp(g)
V(g)$color <- "green"
V(g)[giant_v]$color <- "red"
plot(g)

# -----------------------------------------------------------------
# nei() special function
g <- graph( c(1,2, 2,3, 2,4, 4,2) )
V(g)[ .nei( c(2,4) ) ]
V(g)[ .nei( c(2,4), "in") ]
V(g)[ .nei( c(2,4), "out") ]

# -----------------------------------------------------------------
# The same with vertex names
g <- graph(~ A -+ B, B -+ C:D, D -+ B)
V(g)[ .nei( c('B', 'D') ) ]
V(g)[ .nei( c('B', 'D'), "in" ) ]
V(g)[ .nei( c('B', 'D'), "out" ) ]
}
\seealso{
Other vertex and edge sequences: \code{\link{E}},
  \code{\link{V}}, \code{\link{igraph-es-attributes}},
  \code{\link{igraph-es-indexing2}},
  \code{\link{igraph-es-indexing}},
  \code{\link{igraph-vs-attributes}},
  \code{\link{igraph-vs-indexing2}},
  \code{\link{print.igraph.es}},
  \code{\link{print.igraph.vs}}

Other vertex and edge sequence operations: \code{\link{c.igraph.es}},
  \code{\link{c.igraph.vs}},
  \code{\link{difference.igraph.es}},
  \code{\link{difference.igraph.vs}},
  \code{\link{igraph-es-indexing2}},
  \code{\link{igraph-es-indexing}},
  \code{\link{igraph-vs-indexing2}},
  \code{\link{intersection.igraph.es}},
  \code{\link{intersection.igraph.vs}},
  \code{\link{rev.igraph.es}}, \code{\link{rev.igraph.vs}},
  \code{\link{union.igraph.es}},
  \code{\link{union.igraph.vs}},
  \code{\link{unique.igraph.es}},
  \code{\link{unique.igraph.vs}}
}
