% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dstm.R
\name{dstm_eof}
\alias{dstm_eof}
\title{Dynamic spatio-temporal model with EOFs}
\usage{
dstm_eof(
  Y,
  proc_model = "Dense",
  P = 4L,
  proc_error = "IW",
  n_samples = 1L,
  sample_sigma2 = TRUE,
  verbose = FALSE,
  params = NULL
)
}
\arguments{
\item{Y}{(numeric matrix) S by T data matrix containing the response variable
at S spatial locations and T time points.
The t-th column (NOT row) corresponds to the t-th observation vector.}

\item{proc_model}{(character string) Process model: one of "RW" (identity process matrix),
"AR" (diagonal process matrix), or "Dense" (dense process matrix).}

\item{P}{(integer) Number of EOFs or, in other words, the state space size.}

\item{proc_error}{(character string) Process error: 
"IW" (inverse-Wishart) or "Discount" (discount factor).}

\item{n_samples}{(numeric scalar) Number of samples to draw}

\item{sample_sigma2}{(logical) Whether to sample the variance of the iid observation error.}

\item{verbose}{(logical) Whether to print additional information;
e.g., iteration in sampling algorithm.}

\item{params}{(list) List of hyperparameter values; see details.}
}
\description{
Fits a dynamic spatio-temporal model using empirical orthogonal functions
(EOFs).
The model does not require the spatial locations because the process model
is based on the principal components of the data matrix.
Three broad model types are supported:

1. RW: A random walk model for which the process matrix is the identity.

2. AR: An auto-regressive model for which the process matrix is diagonal
   and its elements are estimated.
   
3. Dense: A model in which the process matrix is a dense, estimated
   matrix.
   
For each broad model type,
users can specify a variety of options including
the size of the state space,
the form of the process error,
and whether to sample the observation error.
Users can specify prior distributions for all sampled quantities using the
`params` argument.

Each model type mentioned above is a dynamic linear model (DLM),
and the state vectors can be estimated using the forward filtering
backward sampling algorithm.
The other parameters are estimated with conditionally conjugate updates.
}
\details{
This section explains how to specify custom hyperparameters using the `params` argument.
For each distribution referenced below,
we use the scale parameterization found on the distribution's Wikipedia page.
You may specify the following as named elements of the `params` list:

m_0: (numeric vector) The prior mean of the state vector at time zero
 (\eqn{\theta_0}).

C_0: (numeric matrix) The prior variance-covariance matrix of the state
vector at time zero (\eqn{\theta_0}).

alpha_sigma2, beta_sigma2: (numeric scalars) The inverse-Gamma parameters 
(scale parameterization) of the prior distribution on the observation error 
(\eqn{\sigma^2}).

sigma2: (numeric scalar) The value to use for the observation error 
(\eqn{\sigma^2}) if `sample_sigma2` = FALSE.

mu_G: (numeric matrix) The prior mean for the process matrix G.
If `proc_model` = "AR", then `mu_G` must be a diagonal matrix.
If `proc_model` = "Dense", then `mu_G` has no constraints.

Sigma_G: (numeric matrix) The prior variance-covariance matrix for the
process matrix. If proc_model = "AR", then Sigma_G should be P by 
P and is the variance-covariance matrix for diag(G).
If proc_model = "Dense", then Sigma_G should be P^2 by P^2 and is the 
variance-covariance matrix for vec(G).

alpha_lambda, beta_lambda: (numeric scalars) The inverse-Gamma parameters 
(scale parameterization) of the prior distribution on 
\eqn{\lambda = (1 - \delta) / \delta},
where \eqn{\delta} is the discount factor.

scale_W: (numeric matrix) The scale matrix for the inverse-Wishart prior
distribution on the variance-covariance matrix of the process error (`W`).

df_W: (numeric scalar) The degees of freedom for the inverse-Wishart prior
distribution on the variance-covariance matrix of the process error (`W`).
}
\examples{
# Load example data
data("ide_standard")

# Illustrate methods
rw_model <- dstm_eof(ide_standard, proc_model="RW", verbose=TRUE)
summary(rw_model)
predict(rw_model) 

# Other model types
dstm_eof(ide_standard, proc_model="AR") # Diagonal process matrix
dstm_eof(ide_standard, proc_model="Dense") # Dense process matrix

# Specify hyperparameters
P <- 4
dstm_eof(ide_standard, P=P,
         params=list(m_0=rep(1, P), C_0=diag(0.01, P),
                     scale_W=diag(P), df_W=100))
}
\references{
Cressie, N., and Wikle, C. K. (2011), Statistics for spatio-temporal data,
John Wiley and Sons, New York, ISBN:978-0471692744.

Fruhwirth-Schnatter, S. (1994), 
\dQuote{Data Augmentation and Dynamic Linear Models,}
Journal of Time Series Analysis, 15, 183–202,
<doi:10.1111/j.1467-9892.1994.tb00184.x>.

Petris, G., Petrone, S., and Campagnoli, P. (2009), 
Dynamic Linear Models with R, useR!, Springer-Verlag, New York,
ISBN:978-0387772370, <doi:10.1007/b135794>.
}
\seealso{
[dstm_ide]
}
