% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/req-promise.R
\name{req_perform_promise}
\alias{req_perform_promise}
\title{Perform request asynchronously using the promises package}
\usage{
req_perform_promise(req, path = NULL, pool = NULL, verbosity = NULL)
}
\arguments{
\item{req}{A httr2 \link{request} object.}

\item{path}{Optionally, path to save body of the response. This is useful
for large responses since it avoids storing the response in memory.}

\item{pool}{Optionally, a curl pool made by \code{\link[curl:multi]{curl::new_pool()}}. Supply
this if you want to override the defaults for total concurrent connections
(100) or concurrent connections per host (6).}

\item{verbosity}{How much information to print? This is a wrapper
around \code{\link[=req_verbose]{req_verbose()}} that uses an integer to control verbosity:
\itemize{
\item \code{0}: no output
\item \code{1}: show headers
\item \code{2}: show headers and bodies
\item \code{3}: show headers, bodies, and curl status messages.
}

Use \code{\link[=with_verbosity]{with_verbosity()}} to control the verbosity of requests that
you can't affect directly.}
}
\value{
a \code{\link[promises:promise]{promises::promise()}} object which resolves to a \link{response} if
successful or rejects on the same errors thrown by \code{\link[=req_perform]{req_perform()}}.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

This variation on \code{\link[=req_perform]{req_perform()}} returns a \code{\link[promises:promise]{promises::promise()}} object immediately
and then performs the request in the background, returning program control before the request
is finished. See the
\href{https://rstudio.github.io/promises/articles/promises_01_motivation.html}{promises package documentation}
for more details on how to work with the resulting promise object.

If using together with \code{\link[later:create_loop]{later::with_temp_loop()}} or other private event loops,
a new curl pool made by \code{\link[curl:multi]{curl::new_pool()}} should be created for requests made
within the loop to ensure that only these requests are being polled by the loop.

Like with \code{\link[=req_perform_parallel]{req_perform_parallel()}}, exercise caution when using this function;
it's easy to pummel a server with many simultaneous requests. Also, not all servers
can handle more than 1 request at a time, so the responses may still return
sequentially.

\code{req_perform_promise()} also has similar limitations to the
\code{\link[=req_perform_parallel]{req_perform_parallel()}} function, it:
\itemize{
\item Will not retrieve a new OAuth token if it expires after the promised request
is created but before it is actually requested.
\item Does not perform throttling with \code{\link[=req_throttle]{req_throttle()}}.
\item Does not attempt retries as described by \code{\link[=req_retry]{req_retry()}}.
\item Only consults the cache set by \code{\link[=req_cache]{req_cache()}} when the request is promised.
}
}
\examples{
\dontrun{
library(promises)
request_base <- request(example_url()) |> req_url_path_append("delay")

p <- request_base |> req_url_path_append(2) |> req_perform_promise()

# A promise object, not particularly useful on its own
p

# Use promise chaining functions to access results
p \%...>\%
  resp_body_json() \%...>\%
  print()


# Can run two requests at the same time
p1 <- request_base |> req_url_path_append(2) |> req_perform_promise()
p2 <- request_base |> req_url_path_append(1) |> req_perform_promise()

p1 \%...>\%
  resp_url_path \%...>\%
  paste0(., " finished") \%...>\%
  print()

p2 \%...>\%
  resp_url_path \%...>\%
  paste0(., " finished") \%...>\%
  print()

# See the [promises package documentation](https://rstudio.github.io/promises/)
# for more information on working with promises
}
}
