% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/realizedMeasures.R
\name{rBACov}
\alias{rBACov}
\title{rBACov}
\usage{
rBACov(
  pData,
  shares,
  outstanding,
  nonEquity,
  ETFNAME = "ETF",
  unrestricted = TRUE,
  targetBeta = c("HY", "VAB", "expert"),
  expertBeta = NULL,
  preEstimator = "rCov",
  noiseRobustEstimator = rTSCov,
  noiseCorrection = FALSE,
  returnL = FALSE,
  ...
)
}
\arguments{
\item{pData}{a named list. Each list-item contains an \code{xts} or \code{data.table} object with the intraday price data of an ETF and it's component stocks. \code{xts} objects are turned into \code{data.table}s}

\item{shares}{a \code{numeric} with length corresponding to the number of component stocks in the ETF. The entries are the stock holdings of the ETF in the corresponding stock. The order of these entries should correspond to the order the stocks are listed in the \code{list} passed in the \code{pData} argument.}

\item{outstanding}{number of shares outstanding of the ETF}

\item{nonEquity}{aggregated value of the additional components (like cash, money-market funds, bonds, etc.) of the ETF which are not included in the components in \code{pData}.}

\item{ETFNAME}{a \code{character} denoting which entry in the \code{pData} list is the ETF. Default is \code{"ETF"}}

\item{unrestricted}{a \code{logical} denoting whether to use the unrestricted estimator, which is an extension that also affects the diagonal. Default is \code{FALSE}}

\item{targetBeta}{a \code{character}, one of \code{c("HY", "VAB", "expert")} (default) denoting which target beta to use, only the first entry will be used. A value \code{"HY"} means using the Hayashi-Yoshida estimator to estimate the
empirical beta. A value of \code{"VAB"} denotes using the variance adjusted beta. A value of \code{"expert"} denotes using a user-supplied target beta, which can be supplied in the \code{expertBeta} argument.}

\item{expertBeta}{a \code{numeric} containing the user supplied expert beta used when \code{targetBeta} is \code{"expert"}. The \code{expertBeta} must be of length equal to the number of assets in the ETF. Default is \code{NULL}}

\item{preEstimator}{a \code{function} which estimates the integrated covariance matrix. Default is \code{\link{rCov}}}

\item{noiseRobustEstimator}{a \code{function} which estimates the integrated (co)variance and is robust to microstructure noise (only the diagonal will be estimated).
This function is only used when \code{noiseCorrection} is \code{TRUE}. Default is \code{\link{rTSCov}}}

\item{noiseCorrection}{a \code{logical} which denotes whether to use the extension of the estimator which corrects for microstructure noise by using the \code{noiseRobustEstimator} function. Default is \code{FALSE}}

\item{returnL}{a \code{logical} which denotes whether to return the \code{L} matrix. Default is \code{FALSE}}

\item{...}{extra arguments passed to \code{preEstimator} and \code{noiseRobustEstimator}.}
}
\description{
The Beta Adjusted Covariance (BAC) equals the pre-estimator plus a minimal adjustment matrix such that the covariance-implied stock-ETF beta equals a target beta.

The BAC estimator works by applying a minimum correction factor to a pre-estimated covariance matrix such that a target beta derived from the ETF is reached.

Let 
\deqn{
    \bar{\beta}
} 
denote the implied beta derived from the pre-estimator, and
\deqn{
    \beta_{\bullet}
}
denote the target beta, then the correction factor is calculated as:

\deqn{
 L\left(\bar{\beta}-\beta_{\bullet}\right),
}

where
\deqn{
    L=\left(I_{d^{2}}-\frac{1}{2}{\cal Q}\right)\bar{W}^{\prime}\left(I_{d^{2}}\left(\sum_{k=1}^{d}\frac{\sum_{k=1}^{n_{k}}\left(w_{t_{m-1}^{k}}^{k}\right)^{2}}{n_{k}}\right)-\frac{\bar{W}{\cal Q}\bar{W}^{\prime}}{2}\right)^{-1},
}
where \eqn{d} is the number of assets in the ETF, and \eqn{n_{k}} is the number of trades in the \eqn{k}th asset, and
\deqn{
    \bar{W}^{k}=\left(0_{\left(k-1\right)d}^{\prime},\frac{1}{n_{1}}\sum_{m=1}^{n_{1}}w_{t_{m-1}^{1}}^{1},\dots,\frac{1}{n_{d}}\sum_{m=1}^{n_{d}}w_{t_{m-1}^{d}}^{d},0_{\left(d-k\right)d}^{\prime}\right),
}
where \eqn{w_{t_{m-1}^{k}}^{k}} is the weight of the \eqn{k}th asset in the ETF. 

and 
\deqn{
     {\cal Q}^{\left(i-1\right)d+j}
} 
is defined by the following two cases:

\eqn{
    \left(0_{\left(i-1\right)d+j-1}^{\prime},1,0_{\left(d-i+1\right)d-j}^{\prime}\right)+\left(0_{\left(j-1\right)d+i-1}^{\prime},-1,0_{\left(d-j+1\right)d-i}^{\prime}\right) \quad \textrm{if }i\neq j;
}
 
\eqn{
    0_{d^{2}}^{\prime} \quad \textrm{otherwise}.
}

\eqn{\bar{W}^k} has dimensions \eqn{d \times d^2} and \eqn{{\cal Q}^{\left(i-1\right)d+j}} has dimensions \eqn{d^2 \times d^2}.

The Beta-Adjusted Covariance is then 
\deqn{
\Sigma^{\textrm{BAC}} = \Sigma - L\left(\bar{\beta}-\beta_{\bullet}\right),
}

where \eqn{\Sigma} is the pre-estimated covariance matrix.
}
\examples{
\dontrun{
# Since we don't have any data in this package that is of the required format we must simulate it.
library(xts)
library(highfrequency)
# Set the seed for replication
set.seed(123)
iT <- 23400 # Number of observations
# Simulate returns
rets <- mvtnorm::rmvnorm(iT * 3 + 1, mean = rep(0,4), 
                         sigma = matrix(c(0.1, -0.03 , 0.02, 0.04,
                                          -0.03, 0.05, -0.03, 0.02,
                                          0.02, -0.03, 0.05, -0.03,  
                                          0.04, 0.02, -0.03, 0.08), ncol = 4))
# We assume that the assets don't trade in a synchronous manner
w1 <- rets[sort(sample(1:nrow(rets), size = nrow(rets) * 0.5)), 1]
w2 <- rets[sort(sample(1:nrow(rets), size = nrow(rets) * 0.75)), 2]
w3 <- rets[sort(sample(1:nrow(rets), size = nrow(rets) * 0.65)), 3]
w4 <- rets[sort(sample(1:nrow(rets), size = nrow(rets) * 0.8)), 4]
w5 <- rnorm(nrow(rets) * 0.9, mean = 0, sd = 0.005)
timestamps1 <- seq(34200, 57600, length.out = length(w1))
timestamps2 <- seq(34200, 57600, length.out = length(w2))
timestamps3 <- seq(34200, 57600, length.out = length(w3))
timestamps4 <- seq(34200, 57600, length.out = length(w4))
timestamps4 <- seq(34200, 57600, length.out = length(w4))
timestamps5 <- seq(34200, 57600, length.out = length(w5))

w1 <- xts(w1 * c(0,sqrt(diff(timestamps1) / (max(timestamps1) - min(timestamps1)))),
          as.POSIXct(timestamps1, origin = "1970-01-01"), tzone = "UTC")
w2 <- xts(w2 * c(0,sqrt(diff(timestamps2) / (max(timestamps2) - min(timestamps2)))),
          as.POSIXct(timestamps2, origin = "1970-01-01"), tzone = "UTC")
w3 <- xts(w3 * c(0,sqrt(diff(timestamps3) / (max(timestamps3) - min(timestamps3)))),
          as.POSIXct(timestamps3, origin = "1970-01-01"), tzone = "UTC")
w4 <- xts(w4 * c(0,sqrt(diff(timestamps4) / (max(timestamps4) - min(timestamps4)))),
          as.POSIXct(timestamps4, origin = "1970-01-01"), tzone = "UTC")
w5 <- xts(w5 * c(0,sqrt(diff(timestamps5) / (max(timestamps5) - min(timestamps5)))),
          as.POSIXct(timestamps5, origin = "1970-01-01"), tzone = "UTC")

p1  <- exp(cumsum(w1))
p2  <- exp(cumsum(w2))
p3  <- exp(cumsum(w3))
p4  <- exp(cumsum(w4))

weights <- runif(4) * 1:4
weights <- weights / sum(weights)
p5 <- xts(rowSums(cbind(w1 * weights[1], w2 * weights[2], w3 * weights[3], w4 * weights[4]),
                   na.rm = TRUE),
                   index(cbind(p1, p2, p3, p4)))
p5 <- xts(cumsum(rowSums(cbind(p5, w5), na.rm = TRUE)), index(cbind(p5, w5)))

p5 <- exp(p5[sort(sample(1:length(p5), size = nrow(rets) * 0.9))])


BAC <- rBACov(pData = list(
                     "ETF" = p5, "STOCK 1" = p1, "STOCK 2" = p2, "STOCK 3" = p3, "STOCK 4" = p4
                   ), shares = 1:4, outstanding = 1, nonEquity = 0, ETFNAME = "ETF", 
                   unrestricted = FALSE, preEstimator = "rCov", noiseCorrection = FALSE, 
                   returnL = FALSE, K = 2, J = 1)

# Noise robust version of the estimator
noiseRobustBAC <- rBACov(pData = list(
                     "ETF" = p5, "STOCK 1" = p1, "STOCK 2" = p2, "STOCK 3" = p3, "STOCK 4" = p4
                   ), shares = 1:4, outstanding = 1, nonEquity = 0, ETFNAME = "ETF", 
                   unrestricted = FALSE, preEstimator = "rCov", noiseCorrection = TRUE, 
                   noiseRobustEstimator = rHYCov, returnL = FALSE, K = 2, J = 1)

# Use the Variance Adjusted Beta method
# Also use a different pre-estimator.
VABBAC <- rBACov(pData = list(
                     "ETF" = p5, "STOCK 1" = p1, "STOCK 2" = p2, "STOCK 3" = p3, "STOCK 4" = p4
                   ), shares = 1:4, outstanding = 1, nonEquity = 0, ETFNAME = "ETF", 
                   unrestricted = FALSE, targetBeta = "VAB", preEstimator = "rHYov", 
                   noiseCorrection = FALSE, returnL = FALSE, Lin = FALSE, L = 0, K = 2, J = 1)                    
                   
}
}
\references{
Boudt, K., Dragun, K., Omauri, S., and Vanduffel, S. (2021) Beta-Adjusted Covariance Estimation (working paper).
}
\seealso{
\code{\link{ICov}} for a list of implemented estimators of the integrated covariance.
}
\author{
Emil Sjoerup, (Kris Boudt and Kirill Dragun for the Python version)
}
