% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ts-model-tuner.R
\name{ts_model_auto_tune}
\alias{ts_model_auto_tune}
\title{Time Series Model Tuner}
\usage{
ts_model_auto_tune(
  .modeltime_model_id,
  .calibration_tbl,
  .splits_obj,
  .drop_training_na = TRUE,
  .date_col,
  .value_col,
  .tscv_assess = "12 months",
  .tscv_skip = "6 months",
  .slice_limit = 6,
  .facet_ncol = 2,
  .grid_size = 30,
  .num_cores = 1,
  .best_metric = "rmse"
)
}
\arguments{
\item{.modeltime_model_id}{The .model_id from a calibrated modeltime table.}

\item{.calibration_tbl}{A calibrated modeltime table.}

\item{.splits_obj}{The time_series_split object.}

\item{.drop_training_na}{A boolean that will drop NA values from the training(splits)
data}

\item{.date_col}{The column that holds the date values.}

\item{.value_col}{The column that holds the time series values.}

\item{.tscv_assess}{A character expression like "12 months". This gets passed to
\code{\link[timetk:time_series_cv]{timetk::time_series_cv()}}}

\item{.tscv_skip}{A character expression like "6 months". This gets passed to
\code{\link[timetk:time_series_cv]{timetk::time_series_cv()}}}

\item{.slice_limit}{An integer that gets passed to \code{\link[timetk:time_series_cv]{timetk::time_series_cv()}}}

\item{.facet_ncol}{The number of faceted columns to be passed to plot_time_series_cv_plan}

\item{.grid_size}{An integer that gets passed to the \code{\link[dials:grid_max_entropy]{dials::grid_latin_hypercube()}}
function.}

\item{.num_cores}{The default is 1, you can set this to any integer value as long
as it is equal to or less than the available cores on your machine.}

\item{.best_metric}{The default is "rmse" and this can be set to any default dials
metric. This must be passed as a character.}
}
\value{
A list object with multiple items.
}
\description{
This function will create a tuned model. It uses the \code{\link[=ts_model_spec_tune_template]{ts_model_spec_tune_template()}}
under the hood to get the generic template that is used in the grid search.
}
\details{
This function can work with the following parsnip/modeltime engines:
\itemize{
\item "auto_arima"
\item "auto_arima_xgboost"
\item "ets"
\item "croston"
\item "theta"
\item "stlm_ets"
\item "tbats"
\item "stlm_arima"
\item "nnetar"
\item "prophet"
\item "prophet_xgboost"
\item "lm"
\item "glmnet"
\item "stan"
\item "spark"
\item "keras"
\item "earth"
\item "xgboost"
}

This function returns a list object with several items inside of it. There are
three categories of items that are inside of the list.
\itemize{
\item \code{data}
\item \code{model_info}
\item \code{plots}
}

The \code{data} section has the following items:
\itemize{
\item \code{calibration_tbl} This is the calibration data passed into the function.
\item \code{calibration_tuned_tbl} This is a calibration tibble that has used the
tuned workflow.
\item \code{tscv_data_tbl} This is the tibble of the time series cross validation.
\item \code{tuned_results} This is a tuning results tibble with all slices from the
time series cross validation.
\item \code{best_tuned_results_tbl} This is a tibble of the parameters for the best
test set with the chosen metric.
\item \code{tscv_obj} This is the actual time series cross validation object returned
from \code{\link[timetk:time_series_cv]{timetk::time_series_cv()}}
}

The \code{model_info} section has the following items:
\itemize{
\item \code{model_spec} This is the original modeltime/parsnip model specification.
\item \code{model_spec_engine} This is the engine used for the model specification.
\item \code{model_spec_tuner} This is the tuning model template returned from \code{\link[=ts_model_spec_tune_template]{ts_model_spec_tune_template()}}
\item \code{plucked_model} This is the model that we have plucked from the calibration tibble
for tuning.
\item \code{wflw_tune_spec} This is a new workflow with the \code{model_spec_tuner} attached.
\item \code{grid_spec} This is the grid search specification for the tuning process.
\item \code{tuned_tscv_wflw_spec} This is the final tuned model where the workflow and
model have been finalized. This would be the model that you would want to
pull out if you are going to work with it further.
}

The \code{plots} section has the following items:
\itemize{
\item \code{tune_results_plt} This is a static ggplot of the grid search.
\item \code{tscv_pl} This is the time series cross validation plan plot.
}
}
\examples{
\dontrun{
suppressPackageStartupMessages(library(modeltime))
suppressPackageStartupMessages(library(timetk))
suppressPackageStartupMessages(library(dplyr))
suppressPackageStartupMessages(library(healthyR.data))
suppressPackageStartupMessages(library(tidymodels))

data <- healthyR_data \%>\%
    filter(ip_op_flag == "I") \%>\%
    select(visit_end_date_time) \%>\%
    rename(date_col = visit_end_date_time) \%>\%
    summarise_by_time(
        .date_var = date_col
        , .by     = "month"
        , visits  = n()
    ) \%>\%
    mutate(date_col = as.Date(date_col)) \%>\%
    filter_by_time(
        .date_var     = date_col
        , .start_date = "2012"
        , .end_date   = "2019"
    )

splits <- time_series_split(
    data
    , date_col
    , assess = 12
    , skip = 3
    , cumulative = TRUE
)

rec_objs <- ts_auto_recipe(
  .data = data
  , .date_col = date_col
  , .pred_col = visits
)

wfsets <- healthyR.ts::ts_wfs_mars(
  .model_type = "earth"
  , .recipe_list = rec_objs
)

wf_fits <- wfsets \%>\%
  modeltime_fit_workflowset(
    data = training(splits)
    , control = control_fit_workflowset(
     allow_par = TRUE
     , verbose = TRUE
    )
  )

models_tbl <- wf_fits \%>\%
  filter(.model != "NULL")

calibration_tbl <- models_tbl \%>\%
  modeltime_calibrate(new_data = testing(splits))

output <- healthyR.ts::ts_model_auto_tune(
  .modeltime_model_id = 1,
  .calibration_tbl = calibration_tbl,
  .splits_obj = splits,
  .drop_training_na = TRUE,
  .date_col = date_col,
  .value_col = visits,
  .tscv_assess = "12 months",
  .tscv_skip = "3 months",
  .num_cores = parallel::detectCores() - 1
)
}

}
\seealso{
Other Model Tuning: 
\code{\link{ts_model_spec_tune_template}()}
}
\author{
Steven P. Sanderson II, MPH
}
\concept{Model Tuning}
