% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/levinson.R
\name{levinson}
\alias{levinson}
\title{Durbin-Levinson Recursion}
\usage{
levinson(acf, p = NROW(acf))
}
\arguments{
\item{acf}{autocorrelation function for lags 0 to \code{p}, specified as a
vector or matrix. If r is a matrix, the function finds the coefficients for
each column of \code{acf} and returns them in the rows of \code{a}.}

\item{p}{model order, specified as a positive integer. Default:
\code{NROW(acf) - 1}.}
}
\value{
A \code{list} containing the following elements:
  \describe{
    \item{a}{vector or matrix containing \code{(p+1)} autoregression
    coefficients. If \code{x} is a matrix, then each row of a corresponds to
    a column of \code{x}. \code{a} has \code{p + 1} columns.}
    \item{e}{white noise input variance, returned as a vector. If \code{x} is
    a matrix, then each element of e corresponds to a column of \code{x}.}
    \item{k}{Reflection coefficients defining the lattice-filter embodiment
    of the model returned as vector or a matrix. If \code{x} is a matrix,
    then each column of \code{k} corresponds to a column of \code{x}.
    \code{k} has \code{p} rows.}
  }
}
\description{
Use the Durbin-Levinson algorithm to compute the coefficients of an
autoregressive linear process.
}
\details{
\code{levinson} uses the Durbin-Levinson algorithm to solve:
\deqn{toeplitz(acf(1:p)) * x = -acf(2:p+1)} The solution \code{c(1, x)} is
the denominator of an all pole filter approximation to the signal \code{x}
which generated the autocorrelation function acf.

From ref [2]: Levinson recursion or Levinson–Durbin recursion is a procedure
in linear algebra to recursively calculate the solution to an equation
involving a Toeplitz matrix. Other methods to process data include Schur
decomposition and Cholesky decomposition. In comparison to these, Levinson
recursion (particularly split Levinson recursion) tends to be faster
computationally, but more sensitive to computational inaccuracies like
round-off errors.
}
\examples{
## Estimate the coefficients of an autoregressive process given by
## x(n) = 0.1x(n-1) - 0.8x(n-2) - 0.27x(n-3) + w(n).
a <- c(1, 0.1, -0.8, -0.27)
v <- 0.4
w <- sqrt(v) * rnorm(15000)
x <- filter(1, a, w)
xc <- xcorr(x, scale = 'biased')
acf <- xc$R[-which(xc$lags < 0)]
lev <- levinson(acf, length(a) - 1)

}
\references{
[1] Steven M. Kay and Stanley Lawrence Marple Jr. (1981).
  Spectrum analysis – a modern perspective. Proceedings of the IEEE, Vol 69,
  1380-1419.\cr
  [2] \url{https://en.wikipedia.org/wiki/Levinson_recursion}
}
\author{
Paul Kienzle, \email{pkienzle@users.sf.net},\cr
 Peter V. Lanspeary, \email{pvl@mecheng.adelaide.edu.au}.\cr
 Conversion to R by Geert van Boxtel, \email{G.J.M.vanBoxtel@gmail.com}.
}
