library(groupdata2)
context("collapse_groups()")


test_that("testing summaries of method ascending-descending", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "participant_2" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "answer" = factor(sample(c("a", "b", "c", "d"), 60, replace = TRUE)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)
  df$session <- rep(c("1", "2", "3"), 20)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  # Method ascending
  # With size balancing

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    num_cols = "score",
    id_cols = "participant",
    balance_size = TRUE,
    method = "ascending",
  )

  ## Testing 'df_coll$.coll_groups'                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing is factor
  expect_true(
    is.factor(df_coll$.coll_groups))
  # Testing values
  expect_equal(
    xpectr::smpl(as.character(df_coll$.coll_groups), n = 30),
    c("3", "3", "2", "2", "2", "2", "2", "2", "3", "3", "3", "3", "1",
      "1", "3", "3", "3", "3", "1", "1", "1", "3", "3", "2", "2",
      "2", "2", "1", "1", "1"),
    fixed = TRUE)
  ## Finished testing 'df_coll$.coll_groups'                                ####

  summ <- summarize_balances(df_coll,
                             group_cols = ".coll_groups",
                             cat_cols = "answer",
                             num_cols = "score",
                             id_cols = "participant")

  # Check most of the group summaries are ascending in order
  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(11, 17, 25),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# participant"]],
    c(4, 7, 9),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(32, 54.41176, 50.84),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(352, 925, 1271),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(1, 5, 8),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(3, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(3, 4, 5),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(4, 4, 6),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# participant", "mean(score)",
      "sum(score)", "# answ_a", "# answ_b", "# answ_c", "# answ_d"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "integer", "numeric", "integer",
      "numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "integer", "double", "integer",
      "double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 10L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

  # Without size balancing
  xpectr::set_test_seed(42)
  df_coll_2 <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    num_cols = "score",
    id_cols = "participant",
    balance_size = FALSE,
    method = "ascending",
  )

  # Gets the same split
  expect_equal(
    df_coll$.coll_groups,
    df_coll_2$.coll_groups
  )

  # Method descending

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    num_cols = "score",
    id_cols = "participant",
    balance_size = TRUE,
    method = "descending",
  )



  ## Testing 'df_coll$.coll_groups'                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing is factor
  expect_true(
    is.factor(df_coll$.coll_groups))
  # Testing values
  expect_equal(
    xpectr::smpl(as.character(df_coll$.coll_groups), n = 30),
    c("1", "1", "2", "2", "2", "2", "2", "2", "1", "1", "1", "1", "3",
      "3", "1", "1", "1", "1", "3", "3", "3", "1", "1", "2", "2",
      "2", "2", "3", "3", "3"),
    fixed = TRUE)
  ## Finished testing 'df_coll$.coll_groups'                                ####

  summ <- summarize_balances(df_coll,
                             group_cols = ".coll_groups",
                             cat_cols = "answer",
                             num_cols = "score",
                             id_cols = "participant")


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(25, 17, 11),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# participant"]],
    c(9, 7, 4),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(50.84, 54.41176, 32),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(1271, 925, 352),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(8, 5, 1),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(6, 4, 3),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(5, 4, 3),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(6, 4, 4),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# participant", "mean(score)",
      "sum(score)", "# answ_a", "# answ_b", "# answ_c", "# answ_d"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "integer", "numeric", "integer",
      "numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "integer", "double", "integer",
      "double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 10L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

})

test_that("testing summaries of cat_levels", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "participant_2" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "answer" = factor(sample(c("a", "b", "c", "d"), 60, replace = TRUE)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)
  df$session <- rep(c("1", "2", "3"), 20)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  # We use method ascending as its
  # pattern is likely easier to spot than balance

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    cat_levels = ".minority",
    method = "ascending",
    balance_size = FALSE
  )

  ## Testing 'df_coll$.coll_groups'                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing is factor
  expect_true(
    is.factor(df_coll$.coll_groups))
  # Testing values
  expect_equal(
    xpectr::smpl(as.character(df_coll$.coll_groups), n = 30),
    c("3", "3", "1", "1", "1", "2", "2", "2", "3", "3", "1", "1", "2",
      "2", "2", "2", "1", "1", "3", "3", "3", "2", "2", "3", "3",
      "3", "3", "2", "2", "2"),
    fixed = TRUE)
  ## Finished testing 'df_coll$.coll_groups'                                ####

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    cat_cols = "answer"
  )

  # Only the minor cat_level ("c") is in ascending order!

  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(15, 20, 18),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(5, 5, 4),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(6, 5, 2),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(0, 5, 7),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(4, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# answ_a", "# answ_b", "# answ_c",
      "# answ_d"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "numeric", "numeric",
      "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "double", "double",
      "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 7L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

  # Set minority level manually

  xpectr::set_test_seed(42)
  df_coll_2 <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    cat_levels = "c", # c is the minority class
    method = "ascending",
    balance_size = FALSE
  )

  expect_equal(
    df_coll$.coll_groups,
    df_coll_2$.coll_groups
  )


  # .majority
  # We method balance as all classes were kind
  # of ascending with the "ascending"

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    cat_levels = ".majority",
    method = "balance",
    balance_size = FALSE
  )


  ## Testing 'df_coll$.coll_groups'                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing is factor
  expect_true(
    is.factor(df_coll$.coll_groups))
  # Testing values
  expect_equal(
    xpectr::smpl(as.character(df_coll$.coll_groups), n = 30),
    c("1", "1", "2", "2", "2", "1", "1", "1", "1", "1", "3", "3", "3",
      "3", "2", "2", "3", "3", "3", "3", "3", "2", "2", "1", "1",
      "1", "1", "3", "3", "3"),
    fixed = TRUE)
  ## Finished testing 'df_coll$.coll_groups'                                ####


  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    cat_cols = "answer"
  )


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(19, 14, 20),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(6, 4, 4),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(2, 5, 6),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(7, 2, 3),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(4, 3, 7),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# answ_a", "# answ_b", "# answ_c",
      "# answ_d"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "numeric", "numeric",
      "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "double", "double",
      "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 7L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

  # Set specific level

  xpectr::set_test_seed(42)
  df_coll_2 <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    cat_levels = "a", # a is the first majority class
    method = "balance",
    balance_size = FALSE
  )

  expect_equal(
    df_coll$.coll_groups,
    df_coll_2$.coll_groups
  )


  # Weighted cat levels
  # We should see the "b" level be pretty balanced
  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    cat_cols = "answer",
    cat_levels = c("a" = 2, "b" = 1000, "c"=1),
    method = "balance",
    balance_size = FALSE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    cat_cols = "answer"
  )


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(11, 22, 20),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(2, 6, 6),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(4, 5, 4), # Fairly balanced yes!
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(2, 6, 4),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(3, 5, 6),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# answ_a", "# answ_b", "# answ_c",
      "# answ_d"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "numeric", "numeric",
      "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "double", "double",
      "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 7L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####


  # List of weights per cat col with multiple cat cols
  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df %>% dplyr::mutate(session = factor(.data$session)),
    n = 3,
    group_cols = ".folds",
    cat_cols = c("answer", "session"),
    cat_levels = list("answer" = c("a" = 2, "b" = 1000, "c" = 1),
                      "session" = c("1" = 1, "2" = 1000)),
    method = "ascending",
    balance_size = FALSE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    cat_cols = c("answer","session")
  )

  # Note: We can't expect to see clear signs of them working
  # as they may cancel out, etc.
  # But both "b" and "2" are ascending, so that's something!


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(10, 18, 25),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_a"]],
    c(1, 5, 8),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_b"]],
    c(1, 6, 6),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_c"]],
    c(4, 3, 5),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# answ_d"]],
    c(4, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# sess_1"]],
    c(3, 6, 8),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# sess_2"]],
    c(4, 6, 9),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["# sess_3"]],
    c(3, 6, 8),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "# answ_a", "# answ_b", "# answ_c",
      "# answ_d", "# sess_1", "# sess_2", "# sess_3"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "numeric", "numeric",
      "numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "double", "double",
      "double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 10L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

})

test_that("testing summaries of balance_size options", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "answer" = factor(sample(c("a", "b", "c", "d"), 60, replace = TRUE))
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)


  # No size balancing

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "balance",
    balance_size = FALSE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups"
  )

  expect_equal(summ$Groups$`# rows`,
               c(17L, 23L, 13L))

  # ascending
  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "ascending",
    balance_size = FALSE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups"
  )

  expect_equal(summ$Groups$`# rows`,
               c(11, 22, 20))

  # WITH size balancing

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "balance",
    balance_size = TRUE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups"
  )

  # Different for the same seed
  expect_equal(summ$Groups$`# rows`,
               c(14L, 21L, 18L))

  # ascending
  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "ascending",
    balance_size = TRUE
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups"
  )

  expect_equal(summ$Groups$`# rows`,
               c(9, 18, 26))

})

test_that("testing summaries of different group_aggregation_fn", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)


  # group_aggregation_fn: mean (default)

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "balance",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = mean
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    num_cols = "score"
  )


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(22, 16, 15),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(50.54545, 52.5625, 57.53333),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(1112, 841, 863),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "mean(score)", "sum(score)"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####

  # Relevant in comparison to later test
  spread_summ <- df_coll %>%
    dplyr::group_by(.data$.coll_groups) %>%
    dplyr::summarise(SD = sd(.data$score),
                     iqr = IQR(.data$score))


  ## Testing 'spread_summ'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(spread_summ),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    spread_summ[[".coll_groups"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    spread_summ[["SD"]],
    c(25.7196, 31.09977, 31.36619),
    tolerance = 1e-4)
  expect_equal(
    spread_summ[["iqr"]],
    c(36.75, 54.75, 47),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(spread_summ),
    c(".coll_groups", "SD", "iqr"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(spread_summ),
    c("factor", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(spread_summ),
    c("integer", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(spread_summ),
    c(3L, 3L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(spread_summ)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'spread_summ'                                         ####


  # group_aggregation_fn: sum

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "balance",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = sum
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    num_cols = "score"
  )

  # Different but not actually better balanced sums

  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(14, 21, 18),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(53.92857, 44.66667, 62.38889),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(755, 938, 1123),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "mean(score)", "sum(score)"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####


  # group_aggregation_fn: IQR
  # NOTE: The mean values might actually play a larger
  # role than the IQRs/SDs in the IQR/SD of the
  # collapsed groups, so this cannot guarantee
  # balanced IQRs/SDs

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "balance",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = IQR
  )

  summ <- df_coll %>%
    dplyr::group_by(.data$.coll_groups) %>%
    dplyr::summarise(SD = sd(.data$score),
                     iqr = IQR(.data$score))

  # The IQRs and SDs are less balanced
  # than when using mean()
  # But they are different for sure

  ## Testing 'summ'                                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ[[".coll_groups"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ[["SD"]],
    c(29.23815, 35.09606, 23.74006),
    tolerance = 1e-4)
  expect_equal(
    summ[["iqr"]],
    c(45, 58.25, 41),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ),
    c(".coll_groups", "SD", "iqr"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ),
    c("factor", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ),
    c("integer", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ),
    c(3L, 3L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ'                                                ####


  # Method ascending

  # Mean (default)

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "ascending",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = mean
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    num_cols = "score"
  )


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(12, 19, 22),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(38.33333, 45.57895, 67.72727),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(460, 866, 1490),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "mean(score)", "sum(score)"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####



  # Sum

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "ascending",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = sum
  )

  summ <- summarize_balances(
    data = df_coll,
    group_cols = ".coll_groups",
    num_cols = "score"
  )


  ## Testing 'summ$Groups'                                                  ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ$Groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ$Groups[[".group_col"]],
    structure(c(1L, 1L, 1L), .Label = ".coll_groups", class = "factor"))
  expect_equal(
    summ$Groups[[".group"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ$Groups[["# rows"]],
    c(10, 21, 22),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["mean(score)"]],
    c(38.8, 44.66667, 67.72727),
    tolerance = 1e-4)
  expect_equal(
    summ$Groups[["sum(score)"]],
    c(388, 938, 1490),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ$Groups),
    c(".group_col", ".group", "# rows", "mean(score)", "sum(score)"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ$Groups),
    c("factor", "factor", "integer", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ$Groups),
    c("integer", "integer", "integer", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ$Groups),
    c(3L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ$Groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ$Groups'                                         ####


  # IQR

  xpectr::set_test_seed(42)
  df_coll <- collapse_groups(
    data = df,
    n = 3,
    group_cols = ".folds",
    method = "ascending",
    balance_size = FALSE,
    num_cols = "score",
    group_aggregation_fn = IQR
  )

  summ <- df_coll %>%
    dplyr::group_by(.data$.coll_groups) %>%
    dplyr::summarise(SD = sd(.data$score),
                     iqr = IQR(.data$score))


  ## Testing 'summ'                                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ[[".coll_groups"]],
    structure(1:3, .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    summ[["SD"]],
    c(27.69586, 28.89417, 27.40917),
    tolerance = 1e-4)
  expect_equal(
    summ[["iqr"]],
    c(40.25, 45, 49.75), # Ascending ftw!
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ),
    c(".coll_groups", "SD", "iqr"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ),
    c("factor", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ),
    c("integer", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ),
    c(3L, 3L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ'                                                ####


})

test_that("testing calculate_summary_collapse_groups_()", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "diagnosis" = factor(rep(sample(c(1:3), 20, replace = TRUE), 3)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  prepped <- prepare_collapse_groups_run_(
    data = df,
    group_cols = ".folds",
    cat_cols = "diagnosis",
    num_cols = c("age", "score"),
    id_cols = "participant",
    weights = NULL,
    balance_size = TRUE
  )

  # Created expected summary
  expected_summ <- prepped[["data"]] %>%
    dplyr::group_by(.data$.____.folds) %>%
    dplyr::summarise(
      age = sum(age),
      score = sum(score),
      size = n(),
      participant = length(unique(participant))
    )

  cat_summ <- prepped[["data"]] %>%
    dplyr::group_by(.data$.____.folds) %>%
    dplyr::count(diagnosis) %>%
    tidyr::spread(key = "diagnosis", value = "n", fill = 0) %>%
    dplyr::ungroup() %>%
    dplyr::mutate(dplyr::across(where(is.numeric), .fns = ~ {
      (.x - mean(.x)) / sd(.x)
    }),
    diagnosis = (`1` + `2` + `3`) / 3,
    majority = `3`)

  expected_summ$diagnosis <- cat_summ$diagnosis
  expected_summ$majority <- cat_summ$majority

  summ <- calculate_summary_collapse_groups_(
    data = prepped[["data"]],
    tmp_old_group_var = prepped[["group_cols"]],
    cat_cols = prepped[["cat_cols"]],
    cat_levels = NULL,
    num_cols = prepped[["num_cols"]],
    group_aggregation_fn = sum,
    balance_size = TRUE,
    id_cols = prepped[["id_cols"]]
  )

  summ_ordered <- dplyr::arrange(summ, .____.folds)

  # Check that we get the same as the manual calculations
  expect_equal(sort(colnames(summ_ordered)),
               sort(colnames(expected_summ)[-7])) # Remove "majority" - it's for later
  expect_equal(summ_ordered$.____.folds, expected_summ$.____.folds)
  expect_equal(summ_ordered$diagnosis, expected_summ$diagnosis)
  expect_equal(summ_ordered$age, expected_summ$age)
  expect_equal(summ_ordered$score, expected_summ$score)
  expect_equal(summ_ordered$size, expected_summ$size)
  expect_equal(summ_ordered$participant, expected_summ$participant)


  ## Testing 'summ'                                                         ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(summ),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    summ[[".____.folds"]],
    structure(c(3L, 8L, 5L, 4L, 2L, 1L, 6L, 7L), .Label = c("1", "2",
                                                            "3", "4", "5", "6", "7", "8"), class = "factor"))
  expect_equal(
    summ[["diagnosis"]],
    c(-0.41471, 0.30371, -0.41471, 0.24455, 0.08321, 0.13948, 0.26193,
      -0.20346),
    tolerance = 1e-4)
  expect_equal(
    summ[["age"]],
    c(369, 667, 166, 528, 396, 213, 405, 237),
    tolerance = 1e-4)
  expect_equal(
    summ[["score"]],
    c(172, 416, 194, 398, 443, 255, 428, 407),
    tolerance = 1e-4)
  expect_equal(
    summ[["size"]],
    c(4, 8, 4, 9, 9, 6, 7, 6),
    tolerance = 1e-4)
  expect_equal(
    summ[["participant"]],
    c(2, 3, 2, 3, 3, 2, 3, 2),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(summ),
    c(".____.folds", "diagnosis", "age", "score", "size", "participant"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(summ),
    c("factor", "numeric", "integer", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(summ),
    c("integer", "double", "integer", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(summ),
    c(8L, 6L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(summ)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ'                                                ####

  # With majority cat_level only
  summ <- calculate_summary_collapse_groups_(
    data = prepped[["data"]],
    tmp_old_group_var = prepped[["group_cols"]],
    cat_cols = prepped[["cat_cols"]],
    cat_levels = ".majority",
    num_cols = prepped[["num_cols"]],
    group_aggregation_fn = sum,
    balance_size = TRUE,
    id_cols = prepped[["id_cols"]]
  )
  summ_ordered <- dplyr::arrange(summ, .____.folds)
  expect_equal(summ_ordered$diagnosis, expected_summ$majority)


  # Edge cases

  # diagnosis leads to zero-variance vector

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1, 10)),
    "age" = c(-5, 5, -5, 5, -5, 5, -5, 5, -5, 5),
    "diagnosis" = factor(c(1,1,2,2,1,1,2,2,1,1)),
    ".__.folds" = factor(c(1, 1, 2, 2, 3, 3, 4, 4, 5, 5))
  )


  ## Testing 'summ <- calculate_summary_collapse_groups_( data = df, tmp_o...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(summ <- calculate_summary_collapse_groups_(
    data = df,
    tmp_old_group_var = ".__.folds",
    cat_cols = "diagnosis",
    cat_levels = NULL,
    num_cols = "age",
    group_aggregation_fn = sum,
    balance_size = FALSE,
    id_cols = "participant"
  ), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['warnings']]),
    xpectr::strip("Combining the standardized level counts for the `cat_cols` column 'diagnosis' led to a zero-variance vector. Consider not balancing this column or change the included `cat_levels`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['messages']]),
    xpectr::strip(character(0)),
    fixed = TRUE)
  # Assigning output
  output_19148 <- xpectr::suppress_mw(summ <- calculate_summary_collapse_groups_(
    data = df,
    tmp_old_group_var = ".__.folds",
    cat_cols = "diagnosis",
    cat_levels = NULL,
    num_cols = "age",
    group_aggregation_fn = sum,
    balance_size = FALSE,
    id_cols = "participant"
  ))
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19148[[".__.folds"]],
    factor(c(1, 2, 3, 4, 5)),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["diagnosis"]],
    c(0, 0, 0, 0, 0),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["age"]],
    c(0, 0, 0, 0, 0),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["participant"]],
    c(1, 1, 1, 1, 1),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_19148),
    c(".__.folds", "diagnosis", "age", "participant"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("factor", "numeric", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("integer", "double", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    5:4)
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'summ <- calculate_summary_collapse_groups_( data = df, tmp_o...'     ####


})

test_that("testing scale_and_combine_()", {
  xpectr::set_test_seed(42)

  df = data.frame(
    "a" = c(1,2,3,4,5,6),
    "b" = c(1,2,3,4,5,6) * -1,
    "c" = c(1,2,3,4,5,6) * 10
  )
  w <- c("b"=2, "c"=1, "a"=5)

  # Expected
  w_norm <- w / sum(w)
  df_scaled <- df %>%
    dplyr::mutate(
      a = standardize_(a) * w_norm[["a"]],
      b = standardize_(b) * w_norm[["b"]],
      c = standardize_(c) * w_norm[["c"]],
      new = a + b + c
    )

  # Observed
  combined <- scale_and_combine_(
    data = df,
    weights = w,
    include_cols = c("a", "b", "c"),
    scale_fn=standardize_,
    col_name="new")$new

  expect_equal(df_scaled$new, combined)


  # Names in weights must match


  ## Testing 'scale_and_combine_( data = df, weights = c("...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(scale_and_combine_(
      data = df,
      weights = c("b"=2, "c"=1, "d"=5),
      include_cols = c("a", "b", "c"),
      scale_fn=standardize_,
      col_name="new"), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_19148[['error']], lowercase = TRUE),
    xpectr::strip("must be a permutation of set {'a','b','c'}", lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'scale_and_combine_( data = df, weights = c("...'     ####

  # Include cols

  ## Testing 'scale_and_combine_( data = df, weights = w, ...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(scale_and_combine_(
      data = df,
      weights = w,
      include_cols = c("a", "b", "d"),
      scale_fn=standardize_,
      col_name="new"), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_19148[['error']], lowercase = TRUE),
    xpectr::strip("must include the elements {'a','b','d'}.", lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'scale_and_combine_( data = df, weights = w, ...'     ####

})

test_that("testing add_new_groups_()", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "diagnosis" = factor(rep(sample(c(1:3), 20, replace = TRUE), 3)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  # One row per fold / input group
  xpectr::set_test_seed(42)
  new_groups = data.frame(
    ".folds" = factor(sample(1:8, 8)),
    "combined" = runif(8),
    ".coll_g" = factor(sample(c(1:4, 1:4), 8))
  ) %>%
    dplyr::group_by(.data$.coll_g)

  # Add .coll_g to df
  df_added_groups <- add_new_groups_(
    data = df,
    new_groups = new_groups,
    tmp_old_group_var = ".folds",
    col_name = ".coll_g"
  )

  expect_equal(
    colnames(df_added_groups),
    c("participant", "age", "diagnosis", "score", ".coll_g")
  )


  ## Testing 'df_added_groups'                                              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(df_added_groups),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(df_added_groups[["participant"]], n = 30),
    structure(c(1L, 10L, 11L, 12L, 13L, 14L, 14L, 16L, 16L, 16L, 17L,
                18L, 19L, 19L, 2L, 20L, 3L, 3L, 3L, 4L, 5L, 5L, 6L, 6L, 7L,
                8L, 8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6",
                                                "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", "17",
                                                "18", "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(df_added_groups[["age"]], n = 30),
    c(92, 65, 42, 91, 83, 23, 23, 80, 80, 80, 88, 10, 39, 39, 93, 46,
      29, 29, 29, 81, 62, 62, 50, 50, 70, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(df_added_groups[["diagnosis"]], n = 30),
    structure(c(3L, 3L, 3L, 3L, 2L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 3L,
                3L, 1L, 2L, 3L, 3L, 3L, 3L, 1L, 1L, 2L, 2L, 2L, 3L, 3L, 2L,
                2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    xpectr::smpl(df_added_groups[["score"]], n = 30),
    c(35, 18, 86, 73, 75, 56, 33, 47, 72, 64, 68, 24, 80, 32, 44, 85,
      4, 93, 74, 37, 42, 65, 91, 62, 21, 60, 5, 90, 50, 53),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(df_added_groups[[".coll_g"]], n = 30),
    structure(c(4L, 4L, 2L, 4L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 1L, 3L,
                3L, 4L, 3L, 1L, 1L, 1L, 3L, 3L, 3L, 1L, 1L, 2L, 3L, 3L, 1L,
                1L, 1L), .Label = c("1", "2", "3", "4"), class = "factor"))
  # Testing column names
  expect_equal(
    names(df_added_groups),
    c("participant", "age", "diagnosis", "score", ".coll_g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(df_added_groups),
    c("factor", "integer", "factor", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(df_added_groups),
    c("integer", "integer", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(df_added_groups),
    c(53L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(df_added_groups)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'df_added_groups'                                     ####


  ## Testing 'add_new_groups_( data = df, new_groups = new...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(add_new_groups_(
    data = df,
    new_groups = new_groups,
    tmp_old_group_var = ".folds",
    col_name = ".folds"
  ), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error']]),
    xpectr::strip("`tmp_old_group_var` and `col_name` were identical."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'add_new_groups_( data = df, new_groups = new...'     ####


  ## Testing 'add_new_groups_( data = df %>% dplyr::mutate...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(add_new_groups_(
    data = df %>%
      dplyr::mutate(.folds = as.numeric(.folds)),
    new_groups = new_groups,
    tmp_old_group_var = ".folds",
    col_name = ".coll_g"
  ), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error']]),
    xpectr::strip("Assertion on 'data[[tmp_old_group_var]]' failed: Must be of type 'factor', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'add_new_groups_( data = df %>% dplyr::mutate...'     ####


  ## Testing 'add_new_groups_( data = df, new_groups = new...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(add_new_groups_(
    data = df,
    new_groups = new_groups %>%
      dplyr::mutate(.folds = as.numeric(.folds)),
    tmp_old_group_var = ".folds",
    col_name = ".coll_g"
  ), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error']]),
    xpectr::strip("Assertion on 'new_groups[[tmp_old_group_var]]' failed: Must be of type 'factor', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'add_new_groups_( data = df, new_groups = new...'     ####


})

test_that("testing add_ordered_summary_groups_()", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "diagnosis" = factor(rep(sample(c(1:3), 20, replace = TRUE), 3)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  # One row per fold / input group
  xpectr::set_test_seed(42)
  summary <- data.frame(
    ".folds" = factor(sample(1:8, 8)),
    "combined" = runif(8)
  )

  # DESCENDING

  expected_grouping <- summary %>%
    dplyr::arrange(dplyr::desc(.data$combined)) %>%
    dplyr::mutate(.coll_g = factor(rep(1:4, each=2))) %>%
    dplyr::select(.folds, .coll_g) %>%
    dplyr::arrange(.folds)

  df_added <- add_ordered_summary_groups_(
    data = df,
    summary = summary,
    n = 4,
    group_cols = ".folds",
    num_col = "combined",
    method = "descending",
    col_name = ".coll_g"
  )

  observed_grouping <- df_added %>%
    dplyr::count(.folds, .coll_g) %>%
    dplyr::select(-.data$n) %>%
    dplyr::arrange(.folds)

  # Check we get the expected results
  expect_identical(
    as.data.frame(observed_grouping),
    as.data.frame(expected_grouping)
  )

  ## Testing 'observed_grouping'                                            ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(observed_grouping),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    observed_grouping[[".folds"]],
    structure(1:8, .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
              class = "factor"))
  expect_equal(
    observed_grouping[[".coll_g"]],
    structure(c(1L, 4L, 1L, 3L, 2L, 4L, 2L, 3L), .Label = c("1", "2",
                                                            "3", "4"), class = "factor"))
  # Testing column names
  expect_equal(
    names(observed_grouping),
    c(".folds", ".coll_g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(observed_grouping),
    c("factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(observed_grouping),
    c("integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(observed_grouping),
    c(8L, 2L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(observed_grouping)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'observed_grouping'                                   ####


  # ASCENDING

  expected_grouping <- summary %>%
    dplyr::arrange(.data$combined) %>%
    dplyr::mutate(.coll_g = factor(rep(1:4, each=2))) %>%
    dplyr::select(.folds, .coll_g) %>%
    dplyr::arrange(.folds)

  df_added <- add_ordered_summary_groups_(
    data = df,
    summary = summary,
    n = 4,
    group_cols = ".folds",
    num_col = "combined",
    method = "ascending",
    col_name = ".coll_g"
  )

  observed_grouping <- df_added %>%
    dplyr::count(.folds, .coll_g) %>%
    dplyr::select(-.data$n) %>%
    dplyr::arrange(.folds)

  # Check we get the expected results
  expect_identical(
    as.data.frame(observed_grouping),
    as.data.frame(expected_grouping)
  )


  ## Testing 'observed_grouping'                                            ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(observed_grouping),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    observed_grouping[[".folds"]],
    structure(1:8, .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
              class = "factor"))
  expect_equal(
    observed_grouping[[".coll_g"]],
    structure(c(4L, 1L, 4L, 2L, 3L, 1L, 3L, 2L), .Label = c("1", "2",
                                                            "3", "4"), class = "factor"))
  # Testing column names
  expect_equal(
    names(observed_grouping),
    c(".folds", ".coll_g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(observed_grouping),
    c("factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(observed_grouping),
    c("integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(observed_grouping),
    c(8L, 2L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(observed_grouping)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'observed_grouping'                                   ####

})

test_that("testing replace_forbidden_names_()", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "diagnosis" = factor(rep(sample(c(1:3), 20, replace = TRUE), 3)),
    "score" = sample(c(1:100), 20 * 3),
    "combined" = runif(60),
    "n" = rep(sample(c(1:100), 20), 3),
    ".folder" = rep(sample(c(1:100), 20), 3)
  )
  df <- df %>% dplyr::arrange(participant)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 23)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  replaced <- replace_forbidden_names_(
    data = df,
    data_group_cols = ".foldings",
    group_cols = ".folds",
    cat_cols = c("diagnosis", ".folds"),
    num_cols = c("n", "score"),
    id_cols = c("participant", "combined"),
    weights = c("score" = 3, "combined" = 8, ".folds" = 2),
    invert = FALSE)

  expect_equal(
    colnames(replaced$data),
    c("participant", "age", "diagnosis", "score", ".____combined",
      ".____n", ".____.folder", ".____.folds")
  )
  expect_equal(
    dplyr::group_vars(replaced$data),
    character(0)
  )
  expect_equal(
    replaced$data_group_cols,
    ".____.foldings"
  )
  expect_equal(
    replaced$group_cols,
    ".____.folds"
  )
  expect_equal(
    replaced$cat_cols,
    c("diagnosis", ".____.folds")
  )
  expect_equal(
    replaced$num_cols,
    c(".____n", "score")
  )
  expect_equal(
    replaced$id_cols,
    c("participant", ".____combined")
  )
  expect_equal(
    replaced$weights,
    c("score" = 3, ".____combined" = 8, ".____.folds" = 2)
  )


  # Grouped data
  replaced <- replace_forbidden_names_(
    data = dplyr::group_by(df, .folds),
    data_group_cols = ".foldings",
    group_cols = ".folds",
    cat_cols = c("diagnosis", ".folds"),
    num_cols = c("n", "score"),
    id_cols = c("participant", "combined"),
    weights = c("score" = 3, "combined" = 8, ".folds" = 2),
    invert = FALSE)

  expect_equal(
    colnames(replaced$data),
    c("participant", "age", "diagnosis", "score", ".____combined",
      ".____n", ".____.folder", ".____.folds")
  )
  expect_equal(
    dplyr::group_vars(replaced$data),
    ".____.folds"
  )

  ## INVERSE!

  inverse_replaced <- replace_forbidden_names_(
    data = replaced$data, # Still grouped
    data_group_cols = replaced$data_group_cols,
    group_cols = replaced$group_cols,
    cat_cols = replaced$cat_cols,
    num_cols = replaced$num_cols,
    id_cols = replaced$id_cols,
    weights = replaced$weights,
    invert = TRUE)

  expect_identical(
    as.data.frame(inverse_replaced$data),
    as.data.frame(df)
  )

  expect_equal(
    colnames(inverse_replaced$data),
    c("participant", "age", "diagnosis", "score", "combined",
      "n", ".folder", ".folds")
  )
  expect_equal(
    dplyr::group_vars(inverse_replaced$data),
    ".folds"
  )
  expect_equal(
    inverse_replaced$data_group_cols,
    ".foldings"
  )
  expect_equal(
    inverse_replaced$group_cols,
    ".folds"
  )
  expect_equal(
    inverse_replaced$cat_cols,
    c("diagnosis", ".folds")
  )
  expect_equal(
    inverse_replaced$num_cols,
    c("n", "score")
  )
  expect_equal(
    inverse_replaced$id_cols,
    c("participant", "combined")
  )
  expect_equal(
    inverse_replaced$weights,
    c("score" = 3, "combined" = 8, ".folds" = 2)
  )

})

test_that("fuzz testing collapse_groups() without auto-tune", {

  # Set seed
  xpectr::set_test_seed(42)

  # Create data frame
  df <- data.frame(
    "participant" = factor(rep(1:20, 3)),
    "participant_2" = factor(rep(1:20, 3)),
    "age" = rep(sample(c(1:100), 20), 3),
    "answer" = factor(sample(c("a", "b", "c", "d"), 60, replace = TRUE)),
    "score" = sample(c(1:100), 20 * 3)
  )
  df <- df %>% dplyr::arrange(participant)
  df$session <- rep(c("1", "2", "3"), 20)

  # Sample rows to get unequal sizes per participant
  df <- dplyr::sample_n(df, size = 53)

  # Create the initial groups (to be collapsed)
  df <- fold(
    data = df,
    k = 8,
    method = "n_dist",
    id_col = "participant"
  )

  # Ungroup the data frame
  # Otherwise `collapse_groups()` would be
  # applied to each fold separately!
  df <- dplyr::ungroup(df)

  # Generate expectations for 'collapse_groups'
  # Tip: comment out the gxs_function() call
  # so it is easy to regenerate the tests
  # xpectr::set_test_seed(42)
  # xpectr::gxs_function(
  #   fn = collapse_groups,
  #   args_values = list(
  #     "data" = list(df, c(1,2,3), 1, NA),
  #     "n" = list(3, 4, 1, 8, 9, "5", NA),
  #     "group_cols" = list(".folds", c(".folds", "participant"), "answer", 1, NA),
  #     "cat_cols" = list("answer", c("answer", "session"), "score", ".folds", 1, NA),
  #     "cat_levels" = list(NULL, ".majority", ".minority", "nope", c("a" = 2, "b" = 3),
  #                         c("nope" = 2, "b" = 3), list("answer" = c("a" = 2)),
  #                         list("session" = c("a" = 2)), list("sdfs" = c("a" = 2)), NA),
  #     "num_cols" = list("score", c("score", "age"), "answer", ".folds", 1, NA),
  #     "id_cols" = list("participant", c("participant_2", "participant"), "score", ".folds", 1, NA),
  #     "balance_size" = list(TRUE, FALSE, 1, NA),
  #     "auto_tune" = list(FALSE, 1, NA), # TRUE is tested elsewhere
  #     "weights" = list(NULL, c("size" = 2, "answer" = 7, "score" = 1, "participant" = 2),
  #                      c("size" = 3, "nope" = 5), c("participant" = 2,"participant_2" = 2),
  #                      list("size" = 2, "answer" = 7), c("size" = NA, "answer" = 7),
  #                      1, NA),
  #     "method" = list("balance", "ascending", "descending", "none", list("balance"), 1, NA),
  #     "group_aggregation_fn" = list(mean, sum, sd, 2, NA),
  #     "num_new_group_cols" = list(1, 2, "two", NA),
  #     "unique_new_group_cols_only" = list(TRUE, FALSE, 1, NA),
  #     "max_iters" = list(2, 0, "sdf", NA),
  #     "extreme_pairing_levels" = list(1, 2, 0, "str", NA),
  #     "combine_method" = list("avg_standardized", "avg_min_max_scaled", "nope", 1, NA),
  #     "col_name" = list(".collg", ".some_name.", ".folds", "answer", 1, NA),
  #     "parallel" = list(FALSE, 1, NA),
  #     "verbose" = list(FALSE, 1, NA)
  #   ),
  #   extra_combinations = list(
  #     list("cat_cols" = NULL, "num_cols" = NULL, "id_cols" = NULL, "balance_size" = FALSE),
  #     list("cat_cols" = "answer", "num_cols" = NULL, "id_cols" = NULL, "balance_size" = FALSE),
  #     list("cat_cols" = NULL, "num_cols" = "score", "id_cols" = NULL, "balance_size" = FALSE),
  #     list("cat_cols" = NULL, "num_cols" = NULL, "id_cols" = "participant", "balance_size" = FALSE),
  #     list("cat_cols" = NULL, "num_cols" = NULL, "id_cols" = NULL, "balance_size" = TRUE),
  #     list("auto_tune" = TRUE, "method" = "ascending")
  #   ),
  #   indentation = 2,
  #   copy_env = FALSE
  # )



  ## Testing 'collapse_groups'                                                ####
  ## Initially generated by xpectr
  # Testing different combinations of argument values

  # Testing collapse_groups(data = df, n = 3, group_cols...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19148[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19148[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19148[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19148[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19148[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19148[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19148[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19148[[".collg"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19148),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = c(1, 2, 3), n = 3, gr...
  # Changed from baseline: data = c(1, 2, 3)
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19370 <- xpectr::capture_side_effects(collapse_groups(data = c(1, 2, 3), n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = 1, n = 3, group_cols ...
  # Changed from baseline: data = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12861 <- xpectr::capture_side_effects(collapse_groups(data = 1, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12861[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12861[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = NA, n = 3, group_cols...
  # Changed from baseline: data = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18304 <- xpectr::capture_side_effects(collapse_groups(data = NA, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'logical'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = NULL, n = 3, group_co...
  # Changed from baseline: data = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16417 <- xpectr::capture_side_effects(collapse_groups(data = NULL, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16417[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16417[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 4, group_cols...
  # Changed from baseline: n = 4
  xpectr::set_test_seed(42)
  # Assigning output
  output_15190 <- collapse_groups(data = df, n = 4, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_15190),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_15190[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15190[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15190[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_15190[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15190[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_15190[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_15190[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_15190[[".collg"]], n = 30),
    structure(c(3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 4L, 4L, 3L,
      3L, 1L, 1L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 3L,
      3L, 3L), .Label = c("1", "2", "3", "4"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_15190),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_15190),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_15190),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_15190),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_15190)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 1, group_cols...
  # Changed from baseline: n = 1
  xpectr::set_test_seed(42)
  # Assigning output
  output_17365 <- collapse_groups(data = df, n = 1, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_17365),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_17365[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_17365[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_17365[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_17365[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_17365[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_17365[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_17365[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_17365[[".collg"]], n = 30),
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
      1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
      1L, 1L), .Label = "1", class = "factor"))
  # Testing column names
  expect_equal(
    names(output_17365),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17365),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17365),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17365),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17365)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 8, group_cols...
  # Changed from baseline: n = 8
  xpectr::set_test_seed(42)
  # Assigning output
  output_11346 <- collapse_groups(data = df, n = 8, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_11346),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_11346[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11346[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11346[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11346[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11346[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11346[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_11346[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_11346[[".collg"]], n = 30),
    structure(c(7L, 7L, 2L, 2L, 2L, 8L, 8L, 8L, 7L, 7L, 3L, 3L, 4L,
      4L, 5L, 5L, 3L, 3L, 6L, 6L, 6L, 5L, 5L, 1L, 1L, 1L, 1L, 4L,
      4L, 4L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11346),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11346),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11346),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11346),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11346)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 9, group_cols...
  # Changed from baseline: n = 9
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16569 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 9, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error']]),
    xpectr::strip("`data` subset had fewer `group_cols` groups (8) than `n` (9). If `data` was originally grouped, the `group_cols` within each of those subsets must contain `>= n` groups to collapse."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = "5", group_co...
  # Changed from baseline: n = "5"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17050 <- xpectr::capture_side_effects(collapse_groups(data = df, n = "5", group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'n': Must be of type 'numeric', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = NA, group_col...
  # Changed from baseline: n = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14577 <- xpectr::capture_side_effects(collapse_groups(data = df, n = NA, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14577[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'n': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14577[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = NULL, group_c...
  # Changed from baseline: n = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17191 <- xpectr::capture_side_effects(collapse_groups(data = df, n = NULL, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'n': Must be of type 'numeric', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_cols = c(".fold...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19346 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = c(".folds", "participant"), cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: 'participant'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_cols = "answer"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12554 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = "answer", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12554[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: 'answer'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12554[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_cols = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14622 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = 1, cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14622[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_cols': Must be of type 'character', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14622[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19400 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = NA, cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_cols': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_cols = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19782 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = NULL, cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19782[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_cols': Must be of type 'character', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19782[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = c("answer"...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11174 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = c("answer", "session"), cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11174[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable '`cat_cols` column data[['session']]': Must be of type 'factor', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11174[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = "score"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14749 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "score", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: 'score'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = ".folds"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15603 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = ".folds", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15603[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: '.folds'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15603[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19040 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = 1, cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'cat_cols': Must be of type 'character' (or 'NULL'), not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11387 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NA, cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'cat_cols': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_19888 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NULL, cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19888),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19888[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19888[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19888[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19888[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19888[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19888[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19888[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19888[[".collg"]], n = 30),
    structure(c(2L, 2L, 1L, 1L, 1L, 3L, 3L, 3L, 2L, 2L, 1L, 1L, 2L,
      2L, 3L, 3L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19888),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19888),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19888),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19888),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19888)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols, num_cols, i...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19466 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NULL, cat_levels = NULL, num_cols = NULL, id_cols = NULL, balance_size = FALSE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19466),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19466[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19466[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19466[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19466[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19466[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19466[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19466[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19466[[".collg"]], n = 30),
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 1L, 1L, 2L, 2L, 1L,
      1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19466),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19466),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19466),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19466),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19466)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols, num_cols, i...
  xpectr::set_test_seed(42)
  # Assigning output
  output_10824 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = NULL, id_cols = NULL, balance_size = FALSE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_10824),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_10824[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10824[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10824[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10824[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10824[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10824[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_10824[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_10824[[".collg"]], n = 30),
    structure(c(1L, 1L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 1L,
      1L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10824),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10824),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10824),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10824),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10824)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols, num_cols, i...
  xpectr::set_test_seed(42)
  # Assigning output
  output_15142 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NULL, cat_levels = NULL, num_cols = "score", id_cols = NULL, balance_size = FALSE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_15142),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_15142[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15142[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15142[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_15142[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_15142[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_15142[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_15142[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_15142[[".collg"]], n = 30),
    structure(c(3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 3L, 3L, 1L, 1L, 3L,
      3L, 2L, 2L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_15142),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_15142),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_15142),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_15142),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_15142)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols, num_cols, i...
  xpectr::set_test_seed(42)
  # Assigning output
  output_13902 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NULL, cat_levels = NULL, num_cols = NULL, id_cols = "participant", balance_size = FALSE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_13902),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_13902[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_13902[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_13902[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_13902[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_13902[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_13902[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_13902[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_13902[[".collg"]], n = 30),
    structure(c(2L, 2L, 1L, 1L, 1L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 1L,
      1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_13902),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_13902),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_13902),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_13902),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_13902)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_cols, num_cols, i...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19057 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = NULL, cat_levels = NULL, num_cols = NULL, id_cols = NULL, balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19057),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19057[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19057[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19057[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19057[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19057[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19057[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19057[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19057[[".collg"]], n = 30),
    structure(c(1L, 1L, 3L, 3L, 3L, 2L, 2L, 2L, 1L, 1L, 2L, 2L, 1L,
      1L, 3L, 3L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19057),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19057),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19057),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19057),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19057)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = ".majority"
  xpectr::set_test_seed(42)
  # Assigning output
  output_14469 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = ".majority", num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_14469),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_14469[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_14469[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_14469[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_14469[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_14469[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_14469[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_14469[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_14469[[".collg"]], n = 30),
    structure(c(2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 3L, 3L, 2L,
      2L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 3L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_14469),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_14469),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_14469),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_14469),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_14469)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = ".minority"
  xpectr::set_test_seed(42)
  # Assigning output
  output_18360 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = ".minority", num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_18360),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_18360[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18360[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18360[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18360[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18360[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18360[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_18360[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_18360[[".collg"]], n = 30),
    structure(c(3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 3L, 3L, 2L, 2L, 3L,
      3L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 2L, 2L, 2L, 2L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_18360),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_18360),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_18360),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_18360),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_18360)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = "nope"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17375 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = "nope", num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_17375[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'.minority','.majority','a','b','c','d'}",
        "must be a subset of set {'.minority','.majority','a','b','c','d'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17375[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = c(a = 2,...
  xpectr::set_test_seed(42)
  # Assigning output
  output_18110 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = c(a = 2, b = 3), num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_18110),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_18110[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18110[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18110[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18110[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18110[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18110[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_18110[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_18110[[".collg"]], n = 30),
    structure(c(3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 1L, 1L, 1L,
      1L, 3L, 3L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_18110),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_18110),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_18110),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_18110),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_18110)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = c(nope =...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13881 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = c(nope = 2, b = 3), num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_13881[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'.minority','.majority','a','b','c','d'}",
        "must be a subset of set {'.minority','.majority','a','b','c','d'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13881[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = list(ans...
  xpectr::set_test_seed(42)
  # Assigning output
  output_16851 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = list(answer = c(a = 2)), num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_16851),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_16851[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16851[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16851[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16851[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16851[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16851[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_16851[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_16851[[".collg"]], n = 30),
    structure(c(2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 3L, 3L, 2L,
      2L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 3L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16851),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16851),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16851),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16851),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16851)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = list(ses...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10039 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = list(session = c(a = 2)), num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10039[['error']]),
    xpectr::strip("1 assertions failed:\n * when `cat_levels` is a list, its names must be equal to those in `cat_cols`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10039[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = list(sdf...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18329 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = list(sdfs = c(a = 2)), num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18329[['error']]),
    xpectr::strip("1 assertions failed:\n * when `cat_levels` is a list, its names must be equal to those in `cat_cols`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18329[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: cat_levels = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10073 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NA, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10073[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(is_checkmate_v2_1(),
             "Assertion on 'cat_levels' failed: One of the following must apply:\n * checkmate::check_character(cat_levels): Contains missing values (element 1)\n * checkmate::check_numeric(cat_levels): Must have names\n * checkmate::check_list(cat_levels): Must be of type 'list' (or 'NULL'), not 'logical'.",
             "Assertion failed: One of the following must apply:\n * checkmate::check_character(cat_levels): Contains missing values (element 1)\n * checkmate::check_numeric(cat_levels): Must have names\n * checkmate::check_list(cat_levels): Must be of type 'list' (or 'NULL'), not 'logical'."),
      lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10073[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = c("score",...
  xpectr::set_test_seed(42)
  # Assigning output
  output_12076 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = c("score", "age"), id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_12076),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_12076[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12076[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12076[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_12076[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12076[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_12076[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_12076[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_12076[[".collg"]], n = 30),
    structure(c(3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L, 2L, 1L,
      1L, 1L, 1L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_12076),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_12076),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_12076),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_12076),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_12076)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = "answer"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19066 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "answer", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19066[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: 'answer'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19066[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = ".folds"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16117 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = ".folds", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16117[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: '.folds'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16117[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13795 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = 1, id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13795[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_cols': Must be of type 'character' (or 'NULL'), not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13795[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14357 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = NA, id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14357[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_cols': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14357[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_cols = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_10374 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = NULL, id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_10374),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_10374[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10374[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10374[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10374[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10374[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10374[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_10374[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_10374[[".collg"]], n = 30),
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 3L,
      3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10374),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10374),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10374),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10374),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10374)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = c("particip...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19735 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = c("participant_2", "participant"), balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19735),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19735[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19735[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19735[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19735[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19735[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19735[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19735[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19735[[".collg"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19735),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19735),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19735),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19735),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19735)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = "score"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14317 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "score", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14317[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: 'score'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14317[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = ".folds"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19575 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = ".folds", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19575[['error']]),
    xpectr::strip("1 assertions failed:\n * All columns in 'c(group_cols, cat_cols, num_cols, id_cols)' must be unique. Found duplicates: '.folds'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19575[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18877 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = 1, balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18877[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'id_cols': Must be of type 'character' (or 'NULL'), not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18877[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16399 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = NA, balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16399[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'id_cols': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16399[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: id_cols = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_19709 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = NULL, balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19709),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19709[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19709[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19709[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19709[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19709[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19709[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19709[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19709[[".collg"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19709),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19709),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19709),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19709),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19709)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: balance_size = FALSE
  xpectr::set_test_seed(42)
  # Assigning output
  output_16188 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = FALSE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_16188),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_16188[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16188[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16188[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16188[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16188[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16188[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_16188[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_16188[[".collg"]], n = 30),
    structure(c(3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L,
      3L, 1L, 1L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16188),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16188),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16188),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16188),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16188)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: balance_size = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13334 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = 1, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13334[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance_size': Must be of type 'logical flag', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13334[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: balance_size = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13467 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = NA, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13467[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance_size': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13467[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: balance_size = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13984 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = NULL, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13984[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance_size': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13984[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: auto_tune = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17846 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = 1, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17846[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'auto_tune': Must be of type 'logical flag', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17846[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: auto_tune = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10389 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = NA, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10389[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'auto_tune': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10389[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: auto_tune = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17487 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = NULL, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17487[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'auto_tune': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17487[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: auto_tune, method
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16772 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = TRUE, weights = NULL, method = "ascending", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16772[['error']]),
    xpectr::strip("1 assertions failed:\n * when `method` != 'balance', `auto_tune` must be disabled."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16772[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = c(size = 2,...
  xpectr::set_test_seed(42)
  # Assigning output
  output_11712 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = c(size = 2, answer = 7, score = 1, participant = 2), method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_11712),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_11712[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11712[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11712[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11712[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11712[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11712[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_11712[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_11712[[".collg"]], n = 30),
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 3L,
      3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11712),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11712),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11712),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11712),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11712)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = c(size = 3,...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12610 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = c(size = 3, nope = 5), method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_12610[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'size','answer','score','participant'}",
        "must be a subset of set {'size','answer','score','participant'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12610[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = c(participa...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15144 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = c(participant = 2, participant_2 = 2), method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_15144[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'size','answer','score','participant'}",
        "must be a subset of set {'size','answer','score','participant'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15144[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = list(size =...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16756 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = list(size = 2, answer = 7), method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16756[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'weights': Must be of type 'numeric' (or 'NULL'), not 'list'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16756[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = c(size = NA...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19828 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = c(size = NA, answer = 7), method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19828[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'weights': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19828[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17595 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = 1, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17595[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'weights': Must have names."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17595[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: weights = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15664 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NA, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15664[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'weights': Must have names."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15664[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = "ascending"
  xpectr::set_test_seed(42)
  # Assigning output
  output_18496 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "ascending", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_18496),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_18496[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18496[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18496[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18496[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_18496[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_18496[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_18496[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_18496[[".collg"]], n = 30),
    structure(c(3L, 3L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 1L,
      1L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 3L, 3L, 2L, 2L, 2L, 2L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_18496),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_18496),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_18496),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_18496),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_18496)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = "descending"
  xpectr::set_test_seed(42)
  # Assigning output
  output_11894 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "descending", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_11894),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_11894[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11894[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11894[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11894[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11894[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11894[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_11894[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_11894[[".collg"]], n = 30),
    structure(c(1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 3L,
      3L, 1L, 1L, 1L, 1L, 3L, 3L, 3L, 1L, 1L, 2L, 2L, 2L, 2L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11894),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11894),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11894),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11894),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11894)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = "none"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12712 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "none", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_12712[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'balance','ascending','descending'}",
        "must be a subset of set {'balance','ascending','descending'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12712[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = list("balance")
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18281 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = list("balance"), group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18281[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'method': Must be of type 'string', not 'list'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18281[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16932 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = 1, group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16932[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'method': Must be of type 'string', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16932[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12405 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = NA, group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12405[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'method': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12405[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: method = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10429 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = NULL, group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10429[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'method': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10429[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_aggregation_fn ...
  xpectr::set_test_seed(42)
  # Assigning output
  output_11404 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = sum, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_11404),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_11404[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11404[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11404[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11404[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11404[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11404[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_11404[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_11404[[".collg"]], n = 30),
    structure(c(2L, 2L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L,
      3L, 1L, 1L, 3L, 3L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 3L,
      3L, 3L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11404),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11404),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11404),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11404),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11404)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_aggregation_fn ...
  xpectr::set_test_seed(42)
  # Assigning output
  output_12163 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = sd, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_12163),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_12163[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12163[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12163[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_12163[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_12163[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_12163[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_12163[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_12163[[".collg"]], n = 30),
    structure(c(2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 1L,
      1L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_12163),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_12163),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_12163),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_12163),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_12163)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_aggregation_fn = 2
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14793 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = 2, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14793[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_aggregation_fn': Must be a function, not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14793[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_aggregation_fn ...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11974 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = NA, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11974[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_aggregation_fn': Must be a function, not 'logical'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11974[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: group_aggregation_fn ...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17193 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = NULL, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17193[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'group_aggregation_fn': Must be a function, not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17193[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_new_group_cols = 2
  xpectr::set_test_seed(42)
  # Assigning output
  output_10078 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 2, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_10078),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_10078[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10078[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10078[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10078[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10078[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_10078[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_10078[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_10078[[".collg_1"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_10078[[".collg_2"]], n = 30),
    structure(c(2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 2L, 2L, 1L, 1L, 1L,
      1L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10078),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg_1", ".collg_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10078),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10078),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10078),
    c(53L, 9L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10078)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_new_group_cols = ...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13754 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = "two", unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13754[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_new_group_cols': Must be of type 'number', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13754[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_new_group_cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10015 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = NA, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10015[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_new_group_cols': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10015[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: num_new_group_cols = ...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15816 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = NULL, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15816[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_new_group_cols': Must be of type 'number', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15816[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: unique_new_group_cols...
  xpectr::set_test_seed(42)
  # Assigning output
  output_11579 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = FALSE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_11579),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_11579[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11579[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11579[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11579[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_11579[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_11579[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_11579[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_11579[[".collg"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11579),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11579),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11579),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11579),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11579)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: unique_new_group_cols...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13590 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = 1, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13590[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unique_fold_cols_only': Must be of type 'logical flag', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13590[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: unique_new_group_cols...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16456 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = NA, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16456[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unique_fold_cols_only': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16456[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: unique_new_group_cols...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17758 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = NULL, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17758[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unique_fold_cols_only': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17758[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: max_iters = 0
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15636 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 0, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15636[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'max_iters': Must be >= 1."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15636[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: max_iters = "sdf"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12337 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = "sdf", extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12337[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'max_iters': Must be of type 'count', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12337[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: max_iters = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10899 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = NA, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10899[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'max_iters': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10899[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: max_iters = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10856 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = NULL, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10856[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'max_iters': Must be of type 'count', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10856[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: extreme_pairing_level...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13052 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 2, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13052[['error']]),
    xpectr::strip("`num_col`: The (subset of) data is too small to perform 2 levels of extreme pairing. Decrease `extreme_pairing_levels`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13052[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: extreme_pairing_level...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16674 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 0, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16674[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'extreme_pairing_levels': Must be >= 1."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16674[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: extreme_pairing_level...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10002 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = "str", combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10002[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'extreme_pairing_levels': Must be of type 'count', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10002[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: extreme_pairing_level...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12085 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = NA, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12085[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'extreme_pairing_levels': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12085[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: extreme_pairing_level...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19330 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = NULL, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19330[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'extreme_pairing_levels': Must be of type 'count', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19330[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: combine_method = "avg...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19256 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_min_max_scaled", col_name = ".collg", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_19256),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_19256[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19256[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19256[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19256[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_19256[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_19256[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_19256[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_19256[[".collg"]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19256),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".collg"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19256),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19256),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19256),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19256)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: combine_method = "nope"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17340 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "nope", col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_17340[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'avg_standardized','avg_min_max_scaled'}",
        "must be a subset of set {'avg_standardized','avg_min_max_scaled'}"
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17340[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: combine_method = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13330 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = 1, col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13330[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'combine_method': Must be of type 'string', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13330[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: combine_method = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15150 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = NA, col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15150[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'combine_method': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15150[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: combine_method = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17439 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = NULL, col_name = ".collg", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17439[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'combine_method': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17439[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = ".some_name."
  xpectr::set_test_seed(42)
  # Assigning output
  output_16191 <- collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".some_name.", parallel = FALSE, verbose = FALSE)
  # Testing class
  expect_equal(
    class(output_16191),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    xpectr::smpl(output_16191[["participant"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16191[["participant_2"]], n = 30),
    structure(c(10L, 11L, 12L, 13L, 13L, 15L, 16L, 16L, 17L, 17L, 18L,
      18L, 2L, 2L, 20L, 20L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 6L, 6L, 8L,
      8L, 9L, 9L, 9L), .Label = c("1", "2", "3", "4", "5", "6", "7",
      "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", "18",
      "19", "20"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16191[["age"]], n = 30),
    c(65, 42, 91, 83, 83, 40, 80, 80, 88, 88, 10, 10, 93, 93, 46, 46,
      29, 29, 81, 81, 81, 62, 62, 50, 50, 13, 13, 61, 61, 61),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16191[["answer"]], n = 30),
    structure(c(4L, 3L, 4L, 2L, 2L, 1L, 3L, 3L, 1L, 1L, 1L, 2L, 2L,
      4L, 3L, 3L, 1L, 4L, 4L, 3L, 4L, 1L, 4L, 3L, 4L, 4L, 3L, 3L,
      2L, 4L), .Label = c("a", "b", "c", "d"), class = "factor"))
  expect_equal(
    xpectr::smpl(output_16191[["score"]], n = 30),
    c(28, 61, 1, 52, 19, 40, 27, 35, 91, 90, 43, 5, 18, 25, 86, 51,
      17, 36, 63, 69, 30, 75, 41, 54, 73, 9, 95, 13, 8, 39),
    tolerance = 1e-4)
  expect_equal(
    xpectr::smpl(output_16191[["session"]], n = 30),
    c("1", "1", "1", "2", "1", "3", "3", "2", "1", "3", "1", "3", "2",
      "3", "2", "1", "2", "1", "1", "3", "2", "1", "3", "1", "2",
      "1", "2", "3", "1", "2"),
    fixed = TRUE)
  expect_equal(
    xpectr::smpl(output_16191[[".folds"]], n = 30),
    structure(c(6L, 6L, 7L, 7L, 7L, 4L, 4L, 4L, 6L, 6L, 2L, 2L, 1L,
      1L, 8L, 8L, 2L, 2L, 5L, 5L, 5L, 8L, 8L, 3L, 3L, 3L, 3L, 1L,
      1L, 1L), .Label = c("1", "2", "3", "4", "5", "6", "7", "8"),
      class = "factor"))
  expect_equal(
    xpectr::smpl(output_16191[[".some_name."]], n = 30),
    structure(c(2L, 2L, 3L, 3L, 3L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 2L,
      2L, 1L, 1L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
      2L, 2L), .Label = c("1", "2", "3"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16191),
    c("participant", "participant_2", "age", "answer", "score", "session",
      ".folds", ".some_name."),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16191),
    c("factor", "factor", "integer", "factor", "integer", "character",
      "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16191),
    c("integer", "integer", "integer", "integer", "integer", "character",
      "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16191),
    c(53L, 8L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16191)),
    character(0),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = ".folds"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16262 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".folds", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16262[['error']]),
    xpectr::strip("1 assertions failed:\n * `col_name` is already a column in `data`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16262[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = "answer"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12171 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = "answer", parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12171[['error']]),
    xpectr::strip("1 assertions failed:\n * `col_name` is already a column in `data`."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12171[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12165 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = 1, parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12165[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'col_name': Must be of type 'string', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12165[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13889 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = NA, parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13889[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'col_name': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13889[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: col_name = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19424 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = NULL, parallel = FALSE, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19424[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'col_name': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19424[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: parallel = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19626 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = 1, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19626[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'parallel': Must be of type 'logical flag', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19626[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: parallel = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17398 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = NA, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17398[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'parallel': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17398[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: parallel = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17332 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = NULL, verbose = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17332[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'parallel': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17332[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: verbose = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15357 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = 1), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15357[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'verbose': Must be of type 'logical flag', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15357[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: verbose = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10022 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = NA), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10022[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'verbose': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10022[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing collapse_groups(data = df, n = 3, group_cols...
  # Changed from baseline: verbose = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16089 <- xpectr::capture_side_effects(collapse_groups(data = df, n = 3, group_cols = ".folds", cat_cols = "answer", cat_levels = NULL, num_cols = "score", id_cols = "participant", balance_size = TRUE, auto_tune = FALSE, weights = NULL, method = "balance", group_aggregation_fn = mean, num_new_group_cols = 1, unique_new_group_cols_only = TRUE, max_iters = 2, extreme_pairing_levels = 1, combine_method = "avg_standardized", col_name = ".collg", parallel = FALSE, verbose = NULL), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16089[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'verbose': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16089[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  ## Finished testing 'collapse_groups'                                       ####
  #

})

