/*
 * gravmagsubs.cpp -- calling routines for gravmag prism vertical subroutines
 *                    and auxiliary functions
 *
 * These subroutines include:
 *   print_error()
 *   check_error_code()
 *   check_array_lengths_equal()
 *   check_array_all_less()
 *   rectprismgrav1_bycell()
 *   rectprismmag1_bycell()
 *   rectprismgrav1_total()
 *   rectprismmag1_total()
 *   rectprismgrav()
 *   rectprismmag()
 *
 *
 * CITATION
 * =======
 *
 * To cite this software in publications, please use:
 *
 *  Cronkite-Ratcliff, C., Phelps, G., Scheirer, D., 2022, gravmagsubs:
 *  Gravitational and magnetic attraction of 3-D vertical rectangular
 *  prisms, U.S. Geological Survey software release, version 1.0,
 *  https://doi.org/10.5066/P9HBSPRM
 *
 * 
 */


#include "gravmag_primitive_subs.h"
#include <Rcpp.h>
#include <cmath>
#include <stdexcept>
#include <string>

#ifdef _OPENMP
#include <omp.h>
#endif
using namespace Rcpp;

// Prints an error depending on the value of boolean variables
// station_in_prism and other_error.
void print_error(bool station_in_prism, bool other_error)
{
    if (station_in_prism)
        Rcout << "WARNING: At least one station is within the prism.\n";
    if (other_error) {
        Rcout << "WARNING: At least one value is NaN.\n";
        Rcout << "Please make sure that no stations are within the prism or are positioned below the corner of a prism.\n";
    }
} 

// Sets boolean variables station_in_prism or other_error as true depending
// on the value of error_code as returned from rect_prism_mag1().
void check_error_code(int error_code, bool& station_in_prism, bool& other_error)
{
    if (error_code == -1 || error_code == -11 || error_code == -12) {
        station_in_prism = true;
    } else if (error_code == -10) {
        other_error = true;
    }
}

// Checks if the lengths of two NumericVector arrays a and b are equal.
// If the lengths are not equal, raises an error.
void check_array_lengths_equal(const NumericVector& a, const NumericVector& b,
                               std::string name_a, std::string name_b)
{
    R_xlen_t a_size = a.size();
    R_xlen_t b_size = b.size();
    
    std::string err_msg;
    
    if (a_size != b_size) {
        err_msg = name_a + " and " + name_b + " must have the same length";
        stop(err_msg);
    }
}

// Checks that each value in NumericVector array a is less than the value at
// the corresponding index in NumericVector array b.
// If not, raises an error.
void check_array_all_less(const NumericVector& a, const NumericVector& b,
                          std::string name_a, std::string name_b)
{
    check_array_lengths_equal(a, b, name_a, name_b);
    R_xlen_t a_size = a.size();

    if (a_size == 1 && a[0] >= b[0])
        stop(name_a + " must be less than " + name_b);

    std::string err_msg = "Values of " + name_a + " must be less than corresponding values of " + name_b;

    for (R_xlen_t k = 0; k < a_size; k++) {
        if (a[k] >= b[k])
            stop(err_msg);
    }
}

/************************ rectprismgrav1_bycell *********************/
/*
 * Returns an M-by-N matrix where the element i,j represents
 * the gravity anomaly generated by the j-th prism as observed
 * at the i-th station.
 * 
 * Arguments:
 *      xstation  - double   - x-coordinate of station, in km, positive east
 *      ystation  - double   - y-coordinate of station, in km, positive north
 *      zstation  - double   - z-coordinate of station, in km, positive up
 *      xmin      - double   - x-coordinate minimum of prism, in km, positive east
 *      xmax      - double   - x-coordinate maximum of prism, in km, positive east
 *      ymin      - double   - y-coordinate minimum of prism, in km, positive north
 *      ymax      - double   - y-coordinate maximum of prism, in km, positive north
 *      zdeep     - double   - z-coordinate of bottom of prism, in km, positive up
 *      zshallow  - double   - z-coordinate of top of prism, in km, positive up
 *      deltarho  - double   - density contrast, in g/cc
 * 
 * Calls function rect_prism_grav1().
 *
 */

 NumericMatrix rectprismgrav1_bycell(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector deltarho)
{
    R_xlen_t xstation_size = xstation.size();
    R_xlen_t xmin_size = xmin.size();

    double xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, drho, xloc, yloc, zloc;
    double init;

    R_xlen_t ix = 0;

    NumericMatrix gravanom (xstation_size, xmin_size);

    int result = 0;

    bool other_error = false;
    bool station_in_prism = false;
 
    #pragma omp parallel for collapse(2) private(init, xloc, yloc, zloc, xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, drho, ix, result) shared(gravanom, other_error, station_in_prism)
    for (R_xlen_t k = 0; k < xstation_size; k++) {
        for (R_xlen_t j = 0; j < xmin_size; j++) {

            ix = j * xstation_size + k;

            xloc = xstation[k];
            yloc = ystation[k];
            zloc = zstation[k];
            xoff1 = xmin[j];
            yoff1 = ymin[j];
            xoff2 = xmax[j];
            yoff2 = ymax[j];
            zoff1 = zdeep[j];
            zoff2 = zshallow[j];
            drho = deltarho[j];

            result = rect_prism_grav1(xloc, yloc, zloc, xoff1, xoff2,
                                      yoff1, yoff2, zoff1, zoff2, drho, &init, 0);

            if (result < 0) {
                if (!other_error || !station_in_prism) {
                    check_error_code(result, station_in_prism, other_error);
                }
            }

            gravanom[ix] = init;
        }
    }

    print_error(station_in_prism, other_error);
 
    return gravanom;
}
/************************ rectprismgrav1_bycell *********************/


/************************ rectprismmag1_bycell **********************/
/*
 * Returns an M-by-N matrix where the element i,j represents
 * the magnetic anomaly generated by the j-th prism as observed
 * at the i-th station.
 *
 * Arguments:
 *      xstation   - double   - x-coordinate of station, in km, positive east
 *      ystation   - double   - y-coordinate of station, in km, positive north
 *      zstation   - double   - z-coordinate of station, in km, positive up
 *      xmin       - double   - x-coordinate minimum of prism, in km, positive east
 *      xmax       - double   - x-coordinate maximum of prism, in km, positive east
 *      ymin       - double   - y-coordinate minimum of prism, in km, positive north
 *      ymax       - double   - y-coordinate maximum of prism, in km, positive north
 *      z_deep     - double   - z-coordinate of bottom of prism, in km, positive up
 *      z_shallow  - double   - z-coordinate of top    of prism, in km, positive up
 *      magstr     - double   - magnetization of body, in A/m (both induced and remanent)
 *      magnincl   - double   - magnetization inclination of body, in deg, positive below horizontal
 *      magndecl   - double   - magnetization declination of body, in deg, positive east of true north
 *      fieldtotal - double   - Earth's field intensity, in nT
 *      fieldincl  - double   - Earth's field inclination, in deg, positive below horizon
 *      fielddecl  - double   - Earth's field declination, in deg, positive east of true north
 *
 * Calls function rect_prism_mag1().
 *
 */
 
NumericMatrix rectprismmag1_bycell(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector magstr, NumericVector magincl, NumericVector magdecl,
    NumericVector fieldtotal, NumericVector fieldincl, NumericVector fielddecl)
{
    R_xlen_t xstation_size = xstation.size();
    R_xlen_t xmin_size = xmin.size();
    
    double xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, xloc, yloc, zloc;
    double mintn, mincl, mdecl, ftotl, fincl, fdecl;
    double init;
    
    R_xlen_t ix = 0;
    
    NumericMatrix maganom (xstation_size, xmin_size);

    int result = 0;

    bool other_error = false;
    bool station_in_prism = false;
    
    #pragma omp parallel for collapse(2) private(init, xloc, yloc, zloc, xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, mintn, mincl, mdecl, ftotl, fincl, fdecl, ix, result) shared(maganom, other_error, station_in_prism)
    for (R_xlen_t k = 0; k < xstation_size; k++) {
        for (R_xlen_t j = 0; j < xmin_size; j++) {
          
            ix = j * xstation_size + k;
            
            xloc = xstation[k];
            yloc = ystation[k];
            zloc = zstation[k];
            xoff1 = xmin[j];
            yoff1 = ymin[j];
            xoff2 = xmax[j];
            yoff2 = ymax[j];
            zoff1 = zdeep[j];
            zoff2 = zshallow[j];
            mintn = magstr[j];
            mincl = magincl[j];
            mdecl = magdecl[j];
            ftotl = fieldtotal[j];
            fincl = fieldincl[j];
            fdecl = fielddecl[j];
            
            result = rect_prism_mag1(xloc, yloc, zloc, xoff1, xoff2,
                                     yoff1, yoff2, zoff1, zoff2,
                                     mintn, mincl, mdecl,
                                     ftotl, fincl, fdecl, &init, 0);
            
            if (result < 0) {
                if (!other_error || !station_in_prism) {
                    check_error_code(result, station_in_prism, other_error);
                }
            }

            maganom[ix] = init;
        }
    }

    print_error(station_in_prism, other_error);
    
    return maganom;
}
/************************ rectprismmag1_bycell **********************/


/************************ rectprismgrav1_total **********************/
/*
 * Returns an M-length vector where the i-th element represents
 * the total gravity anomaly generated by N prisms as observed
 * at the i-th station.
 *
 * Arguments:
 *      xstation  - double   - x-coordinate of station, in km, positive east
 *      ystation  - double   - y-coordinate of station, in km, positive north
 *      zstation  - double   - z-coordinate of station, in km, positive up
 *      xmin      - double   - x-coordinate minimum of prism, in km, positive east
 *      xmax      - double   - x-coordinate maximum of prism, in km, positive east
 *      ymin      - double   - y-coordinate minimum of prism, in km, positive north
 *      ymax      - double   - y-coordinate maximum of prism, in km, positive north
 *      zdeep     - double   - z-coordinate of bottom of prism, in km, positive up
 *      zshallow  - double   - z-coordinate of top of prism, in km, positive up
 *      deltarho  - double   - density contrast, in g/cc
 *
 * Calls function rect_prism_grav1().
 *
 */
 
NumericVector rectprismgrav1_total(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector deltarho)
{
    R_xlen_t xstation_size = xstation.size();
    R_xlen_t xmin_size = xmin.size();

    double xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, drho, xloc, yloc, zloc;
    double init = 0;

    NumericVector gravanom (xstation_size);

    int result = 0;

    bool other_error = false;
    bool station_in_prism = false;
 
    #pragma omp parallel for collapse(2) private(init, xloc, yloc, zloc, xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, drho, result) shared(gravanom, other_error, station_in_prism)
    for (R_xlen_t k = 0; k < xstation_size; k++) {
        for (R_xlen_t j = 0; j < xmin_size; j++) {


            xloc = xstation[k];
            yloc = ystation[k];
            zloc = zstation[k];
            xoff1 = xmin[j];
            yoff1 = ymin[j];
            xoff2 = xmax[j];
            yoff2 = ymax[j];
            zoff1 = zdeep[j];
            zoff2 = zshallow[j];
            drho = deltarho[j];

            result = rect_prism_grav1(xloc, yloc, zloc, xoff1, xoff2,
                                      yoff1, yoff2, zoff1, zoff2, drho, &init, 0);

            if (result < 0) {
                if (!other_error || !station_in_prism) {
                    check_error_code(result, station_in_prism, other_error);
                }
            }

            #pragma omp atomic
            gravanom[k] += init;
        }
    }

    print_error(station_in_prism, other_error);
 
    return gravanom;
}
/************************ rectprismgrav1_total **********************/


/************************ rectprismmag1_total ***********************/
/*
 * Returns an M-length vector where the i-th element represents
 * the total magnetic anomaly generated by N prisms as observed
 * at the i-th station.
 *
 * Arguments:
 *      xstation   - double   - x-coordinate of station, in km, positive east
 *      ystation   - double   - y-coordinate of station, in km, positive north
 *      zstation   - double   - z-coordinate of station, in km, positive up
 *      xmin       - double   - x-coordinate minimum of prism, in km, positive east
 *      xmax       - double   - x-coordinate maximum of prism, in km, positive east
 *      ymin       - double   - y-coordinate minimum of prism, in km, positive north
 *      ymax       - double   - y-coordinate maximum of prism, in km, positive north
 *      z_deep     - double   - z-coordinate of bottom of prism, in km, positive up
 *      z_shallow  - double   - z-coordinate of top    of prism, in km, positive up
 *      magstr     - double   - magnetization of body, in A/m (both induced and remanent)
 *      magnincl   - double   - magnetization inclination of body, in deg, positive below horizontal
 *      magndecl   - double   - magnetization declination of body, in deg, positive east of true north
 *      fieldtotal - double   - Earth's field intensity, in nT
 *      fieldincl  - double   - Earth's field inclination, in deg, positive below horizon
 *      fielddecl  - double   - Earth's field declination, in deg, positive east of true north
 *
 * Calls function rect_prism_mag1().
 *
 */
 
NumericVector rectprismmag1_total(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector magstr, NumericVector magincl, NumericVector magdecl,
    NumericVector fieldtotal, NumericVector fieldincl, NumericVector fielddecl)
{
    R_xlen_t xstation_size = xstation.size();
    R_xlen_t xmin_size = xmin.size();
    
    double xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, xloc, yloc, zloc;
    double mintn, mincl, mdecl, ftotl, fincl, fdecl;
    double init;
    
    NumericVector maganom (xstation_size);

    bool other_error = false;
    bool station_in_prism = false;

    int result = 0;
    
    #pragma omp parallel for collapse(2) private(init, xloc, yloc, zloc, xoff1, yoff1, xoff2, yoff2, zoff1, zoff2, mintn, mincl, mdecl, ftotl, fincl, fdecl, result) shared(maganom, other_error, station_in_prism)
    for (R_xlen_t k = 0; k < xstation_size; k++) {
        for (R_xlen_t j = 0; j < xmin_size; j++) {

            xloc = xstation[k];
            yloc = ystation[k];
            zloc = zstation[k];
            xoff1 = xmin[j];
            yoff1 = ymin[j];
            xoff2 = xmax[j];
            yoff2 = ymax[j];
            zoff1 = zdeep[j];
            zoff2 = zshallow[j];
            mintn = magstr[j];
            mincl = magincl[j];
            mdecl = magdecl[j];
            ftotl = fieldtotal[j];
            fincl = fieldincl[j];
            fdecl = fielddecl[j];
            
            result = rect_prism_mag1(xloc, yloc, zloc, xoff1, xoff2,
                                     yoff1, yoff2, zoff1, zoff2,
                                     mintn, mincl, mdecl,
                                     ftotl, fincl, fdecl, &init, 0);

            if (result < 0) {
                if (!other_error || !station_in_prism) {
                    check_error_code(result, station_in_prism, other_error);
                }
            }

            #pragma omp atomic
            maganom[k] += init;
        }
    }

    print_error(station_in_prism, other_error);
    
    return maganom;
}
/************************ rectprismmag1_total ***********************/


/************************ rectprismgrav *****************************/
/*
 * Generates R function rectprismgrav().
 *
 * Calculates the gravitational attraction of 3-D rectangular prisms.
 * Calculates anomalies of N prisms at M observation stations.
 *
 * Stations cannot be positioned on the edge of a prism.
 *
 * Arguments:
 *      xstation: vector of length M with the x-coordinates of each station,
 *                in km, positive east
 *      ystation: vector of length M with the y-coordinates of each station,
 *                in km, positive north
 *      zstation: vector of length M with the z-coordinates of each station,
 *                in km, positive up
 *      xmin: vector of length N with the minimum x-coordinates of each prism,
 *            in km, positive east
 *      xmax: vector of length N with the maximum x-coordinates of each prism,
 *            in km, positive east
 *      ymin: vector of length N with the minimum y-coordinates of each prism,
 *            in km, positive north
 *      ymax: vector of length N with the maximum y-coordinates of each prism,
 *            in km, positive north
 *      zdeep: vector of length N with the bottom z-coordinates of each prism,
 *             in km, positive up
 *      zshallow: vector of length N with the top z-coordinates of each prism,
 *                in km, positive up
 *      deltarho: vector of length N with the density contrast of each prism,
 *                in g/cc
 *      bycell: if bycell=true, this function returns M-by-N matrix with anomaly
 *              values generated by individual prisms (default false)
 *
 * Returns a matrix of length M rows.
 *
 * If bycell=false, there will be M rows and 1 column, and the element in 
 * each row represents the total gravity anomaly generated by all N prisms
 * as observed at the M-th station.
 *
 * If bycell=true, the matrix will have M rows and N columns, with the element
 * i,j representing the anomaly value generated by j-th prism as observed
 * at the i-th station.
 *
 */

// [[Rcpp::export]]
NumericMatrix rectprismgrav(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector deltarho, bool bycell=false)
{
    check_array_lengths_equal(xstation, ystation, "xstation", "ystation");
    check_array_lengths_equal(xstation, zstation, "xstation", "zstation");

    check_array_all_less(xmin, xmax, "xmin", "xmax");
    check_array_all_less(ymin, ymax, "ymin", "ymax");
    check_array_all_less(zdeep, zshallow, "zdeep", "zshallow");

    check_array_lengths_equal(xmin, ymin, "xmin", "ymin");
    check_array_lengths_equal(xmin, zshallow, "xmin", "zshallow");
    check_array_lengths_equal(xmin, deltarho, "xmin", "deltarho");

    if (bycell) {
        return rectprismgrav1_bycell(xstation, ystation, zstation,
                                    xmin, xmax, ymin, ymax,
                                    zdeep, zshallow, deltarho);
    }
    NumericVector v = rectprismgrav1_total(
                                    xstation, ystation, zstation,
                                    xmin, xmax, ymin, ymax,
                                    zdeep, zshallow, deltarho);
    R_xlen_t xstation_size = xstation.size();
    return NumericMatrix(xstation_size, 1, v.begin());
}
/************************ rectprismgrav *****************************/


/************************ rectprismmag ******************************/
/*
 * Generates R function rectprismmag().
 *
 * Calculates the magnetic effect of 3-D rectangular prisms.
 * Calculates the anomalies of N prisms at M observation points.
 *
 * Returns total field magnetic anomaly in nT.
 *
 * N.B. Demagnetization effects are ignored in this subroutine.
 *
 * Arguments:
 *      xstation: vector of length M with the x-coordinates of each station,
 *                in km, positive east
 *      ystation: vector of length M with the y-coordinates of each station,
 *                in km, positive north
 *      zstation: vector of length M with the z-coordinates of each station,
 *                in km, positive up
 *      xmin: vector of length N with the minimum x-coordinates of each prism,
 *            in km, positive east
 *      xmax: vector of length N with the maximum x-coordinates of each prism,
 *            in km, positive east
 *      ymin: vector of length N with the minimum y-coordinates of each prism,
 *            in km, positive north
 *      ymax: vector of length N with the maximum y-coordinates of each prism,
 *            in km, positive north
 *      zdeep: vector of length N with the bottom z-coordinates of each prism,
 *             in km, positive up
 *      zshallow: vector of length N with the top z-coordinates of each prism,
 *                in km, positive up
 *      suscvolsi: vector of length N with the volume susceptibility (unitless)
 *      nrmstr: vector of length N with the remanent magnetization of each prism,
 *              in A/m (remanent)
 *      nrmincl: vector of length N with the inclination angle of the remanent
 *               magnetization of each prism, in degrees, positive below horizontal
 *      nrmdecl: vector of length N with the declination angle of the remanent
 *               magnetization of each prism, in degrees, positive east of
 *               true north
 *      fieldtotal: vector of length N with the Earth's field intensity at each
 *                  prism, in nT
 *      fieldincl: vector of length N with the Earth's field inclination angle
 *                 at each prism, in degrees, positive below horizontal
 *      fielddecl: vector of length N with the Earth's field declination angle 
 *                 at each prism, in degrees, positive east of true north
 *      bycell: if bycell=true, this function returns M-by-N matrix with anomaly
 *              values generated by individual prisms (default false)
 *
 * Returns a matrix of length M rows.
 *
 * If bycell=false, there will be M rows and 1 column, and the element in 
 * each row represents the total magnetic anomaly generated by all N prisms
 * as observed at the M-th station.
 *
 * If bycell=true, the matrix will have M rows and N columns, with the element
 * i,j representing the anomaly value generated by j-th prism as observed
 * at the i-th station.
 *
 */

// [[Rcpp::export]]
NumericMatrix rectprismmag(
    NumericVector xstation, NumericVector ystation, NumericVector zstation,
    NumericVector xmin, NumericVector xmax,
    NumericVector ymin, NumericVector ymax,
    NumericVector zdeep, NumericVector zshallow,
    NumericVector suscvolsi, NumericVector nrmstr,
    NumericVector nrmincl, NumericVector nrmdecl,
    NumericVector fieldtotal, NumericVector fieldincl, NumericVector fielddecl,
    bool bycell=false)
{
    const double mu0 = 400 * M_PI;

    check_array_lengths_equal(xstation, ystation, "xstation", "ystation");
    check_array_lengths_equal(xstation, zstation, "xstation", "zstation");

    check_array_all_less(xmin, xmax, "xmin", "xmax");
    check_array_all_less(ymin, ymax, "ymin", "ymax");
    check_array_all_less(zdeep, zshallow, "zdeep", "zshallow");

    check_array_lengths_equal(xmin, ymin, "xmin", "ymin");
    check_array_lengths_equal(xmin, zshallow, "xmin", "zshallow");

    check_array_lengths_equal(xmin, suscvolsi, "xmin", "suscvolsi");
    check_array_lengths_equal(xmin, nrmstr, "xmin", "nrmstr");
    check_array_lengths_equal(xmin, nrmincl, "xmin", "nrmincl");
    check_array_lengths_equal(xmin, nrmdecl, "xmin", "nrmdecl");
    check_array_lengths_equal(xmin, fieldtotal, "xmin", "fieldtotal");
    check_array_lengths_equal(xmin, fieldincl, "xmin", "fieldincl");
    check_array_lengths_equal(xmin, fielddecl, "xmin", "fielddecl");

    double induced_magnetization = 0;
    double magnetization_N = 0;
    double magnetization_E = 0;
    double magnetization_up = 0;
    double magnetization_horiz = 0;

    R_xlen_t num_cells = xmin.size();
    NumericVector magnetization = NumericVector(num_cells, 0.);
    NumericVector magnetization_incl = NumericVector(num_cells, 0.);
    NumericVector magnetization_decl = NumericVector(num_cells, 0.);
    NumericVector external_intensity = NumericVector(num_cells, 0.);

    for (R_xlen_t k = 0; k < num_cells; k++) {
        if (nrmstr[k] != 0 && suscvolsi[k] == 0) {
            magnetization[k] = nrmstr[k];
            magnetization_incl[k] = nrmincl[k];
            magnetization_decl[k] = nrmdecl[k];
        }
        else {
            induced_magnetization = suscvolsi[k] * fieldtotal[k] / mu0;
            if (nrmstr[k] == 0 && suscvolsi[k] != 0) {
                magnetization[k] = induced_magnetization;
                magnetization_incl[k] = fieldincl[k];
                magnetization_decl[k] = fielddecl[k];
            }
            else if (nrmstr[k] != 0 && suscvolsi[k] != 0) {
                magnetization_N     = nrmstr[k] * cos(nrmincl[k]*M_PI/180.) * cos(nrmdecl[k]*M_PI/180.) + induced_magnetization * cos(fieldincl[k]*M_PI/180.) * cos(fielddecl[k]*M_PI/180.) ;
                magnetization_E     = nrmstr[k] * cos(nrmincl[k]*M_PI/180.) * sin(nrmdecl[k]*M_PI/180.) + induced_magnetization * cos(fieldincl[k]*M_PI/180.) * sin(fielddecl[k]*M_PI/180.) ;
                magnetization_up    = nrmstr[k] * sin(nrmincl[k]*M_PI/180.) + induced_magnetization * sin(fieldincl[k]*M_PI/180.);
                magnetization_horiz = sqrt(magnetization_N*magnetization_N + magnetization_E*magnetization_E) ;
                magnetization[k]    = sqrt(magnetization_N*magnetization_N + magnetization_E*magnetization_E + magnetization_up*magnetization_up) ;
                magnetization_incl[k] = atan2(magnetization_up, magnetization_horiz) * 180. / M_PI ;
                magnetization_decl[k] = atan2(magnetization_E, magnetization_N) * 180. / M_PI ;
            }
        }
    }
  
    if (bycell) {
        return rectprismmag1_bycell(xstation, ystation, zstation,
                                    xmin, xmax, ymin, ymax, zdeep, zshallow,
                                    magnetization, magnetization_incl, magnetization_decl,
                                    external_intensity, fieldincl, fielddecl);
    }
    NumericVector v = rectprismmag1_total(
                                    xstation, ystation, zstation,
                                    xmin, xmax, ymin, ymax, zdeep, zshallow,
                                    magnetization, magnetization_incl, magnetization_decl,
                                    external_intensity, fieldincl, fielddecl);
    R_xlen_t xstation_size = xstation.size();
    return NumericMatrix(xstation_size, 1, v.begin());
}
/************************ rectprismmag ******************************/

