% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/graph_create.R
\name{graph_create}
\alias{graph_create}
\title{Create the initial graph for a multiple comparison procedure}
\usage{
graph_create(hypotheses, transitions, hyp_names = NULL)
}
\arguments{
\item{hypotheses}{A numeric vector of hypothesis weights in a graphical
multiple comparison procedure. Must be a vector of values between 0 & 1
(inclusive). The length should match the row and column lengths of
\code{transitions}. The sum of hypothesis weights should not exceed 1.}

\item{transitions}{A numeric matrix of transition weights between hypotheses
in a graphical multiple comparison procedure. Must be a square matrix of
values between 0 & 1 (inclusive). The row and column lengths should match
the length of \code{hypotheses}. Each row (Transition weights leaving a
hypothesis) can sum to no more than 1. The diagonal entries (Transition
weights from a hypothesis to itself) must be all 0s.}

\item{hyp_names}{(Optional) A character vector of hypothesis names. If not
provided, names from \code{hypotheses} and \code{transitions} will be used. If names
are not specified, hypotheses will be named sequentially as H1, H2, .......}
}
\value{
An S3 object of class \code{initial_graph} with a list of 2 elements:
\itemize{
\item Hypothesis weights \code{hypotheses}.
\item Transition weights \code{transitions}.
}
}
\description{
A graphical multiple comparison procedure is represented by 1) a vector of
initial hypothesis weights \code{hypotheses}, and 2) a matrix of initial
transition weights \code{transitions}. This function creates the initial graph
object using hypothesis weights and transition weights.
}
\section{Validation of inputs}{

Inputs are also validated to make sure of the validity of the graph:
\itemize{
\item Hypothesis weights \code{hypotheses} are numeric.
\item Transition weights \code{transitions} are numeric.
\item Length of \code{hypotheses} and dimensions of \code{transitions} match.
\item Hypothesis weights \code{hypotheses} must be non-negative and sum to no more
than 1.
\item Transition weights \code{transitions}:
\itemize{
\item Values must be non-negative.
\item Rows must sum to no more than 1.
\item Diagonal entries must be all 0.
}
\item Hypothesis names \code{hyp_names} override names in \code{hypotheses} or
\code{transitions}.
}
}

\examples{
# A graphical multiple comparison procedure with two primary hypotheses (H1
# and H2) and two secondary hypotheses (H3 and H4)
# See Figure 1 in Bretz et al. (2011).
hypotheses <- c(0.5, 0.5, 0, 0)
transitions <- rbind(
  c(0, 0, 1, 0),
  c(0, 0, 0, 1),
  c(0, 1, 0, 0),
  c(1, 0, 0, 0)
)
hyp_names <- c("H11", "H12", "H21", "H22")
g <- graph_create(hypotheses, transitions, hyp_names)
g

# Explicit names override names in `hypotheses` (with a warning)
hypotheses <- c(h1 = 0.5, h2 = 0.5, h3 = 0, h4 = 0)
transitions <- rbind(
  c(0, 0, 1, 0),
  c(0, 0, 0, 1),
  c(0, 1, 0, 0),
  c(1, 0, 0, 0)
)
g <- graph_create(hypotheses, transitions, hyp_names)
g

# Use names in `transitions`
hypotheses <- c(0.5, 0.5, 0, 0)
transitions <- rbind(
  H1 = c(0, 0, 1, 0),
  H2 = c(0, 0, 0, 1),
  H3 = c(0, 1, 0, 0),
  H4 = c(1, 0, 0, 0)
)
g <- graph_create(hypotheses, transitions)
g

# Unmatched names in `hypotheses` and `transitions` (with an error)
hypotheses <- c(h1 = 0.5, h2 = 0.5, h3 = 0, h4 = 0)
transitions <- rbind(
  H1 = c(0, 0, 1, 0),
  H2 = c(0, 0, 0, 1),
  H3 = c(0, 1, 0, 0),
  H4 = c(1, 0, 0, 0)
)
try(
g <- graph_create(hypotheses, transitions)
)

# When names are not specified, hypotheses are numbered sequentially as
# H1, H2, ...
hypotheses <- c(0.5, 0.5, 0, 0)
transitions <- rbind(
  c(0, 0, 1, 0),
  c(0, 0, 0, 1),
  c(0, 1, 0, 0),
  c(1, 0, 0, 0)
)
g <- graph_create(hypotheses, transitions)
g
}
\references{
Bretz, F., Maurer, W., Brannath, W., and Posch, M. (2009). A graphical
approach to sequentially rejective multiple test procedures.
\emph{Statistics in Medicine}, 28(4), 586-604.

Bretz, F., Posch, M., Glimm, E., Klinglmueller, F., Maurer, W., and
Rohmeyer, K. (2011). Graphical approaches for multiple comparison
procedures using weighted Bonferroni, Simes, or parametric tests.
\emph{Biometrical Journal}, 53(6), 894-913.
}
\seealso{
\code{\link[=graph_update]{graph_update()}} for the updated graph after hypotheses being deleted
from the initial graph.
}
