% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.R
\name{predict}
\alias{predict}
\alias{predictive_interval}
\alias{posterior_linpred}
\alias{posterior_predict}
\alias{predictive_interval.glmmfields}
\alias{posterior_linpred.glmmfields}
\alias{posterior_predict.glmmfields}
\alias{predict.glmmfields}
\title{Predict from a glmmfields model}
\usage{
\method{predictive_interval}{glmmfields}(object, ...)

\method{posterior_linpred}{glmmfields}(object, ...)

\method{posterior_predict}{glmmfields}(object, ...)

\method{predict}{glmmfields}(
  object,
  newdata = NULL,
  estimate_method = c("median", "mean"),
  conf_level = 0.95,
  interval = c("confidence", "prediction"),
  type = c("link", "response"),
  return_mcmc = FALSE,
  iter = "all",
  ...
)
}
\arguments{
\item{object}{An object returned by \code{\link[=glmmfields]{glmmfields()}}.}

\item{...}{Ignored currently}

\item{newdata}{Optionally, a data frame to predict on}

\item{estimate_method}{Method for computing point estimate ("mean" or
"median")}

\item{conf_level}{Probability level for the credible intervals.}

\item{interval}{Type of interval calculation. Same as for
\code{\link[stats:predict.lm]{stats::predict.lm()}}.}

\item{type}{Whether the predictions are returned on "link" scale or
"response" scale (Same as for \code{\link[stats:predict.glm]{stats::predict.glm()}}).}

\item{return_mcmc}{Logical. Should the full MCMC draws be returned for the
predictions?}

\item{iter}{Number of MCMC iterations to draw. Defaults to all.}
}
\description{
These functions extract posterior draws or credible intervals. The helper
functions are named to match those in the \pkg{rstanarm} package and call the
function \code{predict()} with appropriate argument values.
}
\examples{
\donttest{
library(ggplot2)

# simulate:
set.seed(1)
s <- sim_glmmfields(
  n_draws = 12, n_knots = 12, gp_theta = 2.5,
  gp_sigma = 0.2, sd_obs = 0.1
)

# fit:
# options(mc.cores = parallel::detectCores()) # for parallel processing
m <- glmmfields(y ~ 0,
  data = s$dat, time = "time",
  lat = "lat", lon = "lon",
  nknots = 12, iter = 800, chains = 1
)

# Predictions:
# Link scale credible intervals:
p <- predict(m, type = "link", interval = "confidence")
head(p)

# Prediction intervals on new observations (include observation error):
p <- predictive_interval(m)
head(p)

# Posterior prediction draws:
p <- posterior_predict(m, iter = 100)
dim(p) # rows are iterations and columns are data elements

# Draws from the linear predictor (not in link space):
p <- posterior_linpred(m, iter = 100)
dim(p) # rows are iterations and columns are data elements

# Use the `tidy` method to extract parameter estimates as a data frame:
head(tidy(m, conf.int = TRUE, conf.method = "HPDinterval"))

# Make predictions on a fine-scale spatial grid:
pred_grid <- expand.grid(
  lat = seq(min(s$dat$lat), max(s$dat$lat), length.out = 25),
  lon = seq(min(s$dat$lon), max(s$dat$lon), length.out = 25),
  time = unique(s$dat$time)
)
pred_grid$prediction <- predict(m,
  newdata = pred_grid, type = "response", iter = 100,
  estimate_method = "median"
)$estimate

ggplot(pred_grid, aes(lon, lat, fill = prediction)) +
  facet_wrap(~time) +
  geom_raster() +
  scale_fill_gradient2()
}
}
