% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pairwise_comparisons.R
\name{pairwise_comparisons}
\alias{pairwise_comparisons}
\title{Multiple pairwise comparison tests with tidy data}
\usage{
pairwise_comparisons(
  data,
  x,
  y,
  subject.id = NULL,
  type = "parametric",
  paired = FALSE,
  var.equal = FALSE,
  tr = 0.2,
  bf.prior = 0.707,
  p.adjust.method = "holm",
  k = 2L,
  ...
)
}
\arguments{
\item{data}{A dataframe (or a tibble) from which variables specified are to
be taken. Other data types (e.g., matrix,table, array, etc.) will \strong{not}
be accepted.}

\item{x}{The grouping (or independent) variable from the dataframe \code{data}. In
case of a repeated measures or within-subjects design, if \code{subject.id}
argument is not available or not explicitly specified, the function assumes
that the data has already been sorted by such an id by the user and creates
an internal identifier. So if your data is \strong{not} sorted, the results
\emph{can} be inaccurate when there are more than two levels in \code{x} and there
are \code{NA}s present. The data is expected to be sorted by user in
subject-1,subject-2, ..., pattern.}

\item{y}{The response (or outcome or dependent) variable from the
dataframe \code{data}.}

\item{subject.id}{Relevant in case of a repeated measures or within-subjects
design (\code{paired = TRUE}, i.e.), it specifies the subject or repeated
measures identifier. \strong{Important}: Note that if this argument is \code{NULL}
(which is the default), the function assumes that the data has already been
sorted by such an id by the user and creates an internal identifier. So if
your data is \strong{not} sorted and you leave this argument unspecified, the
results \emph{can} be inaccurate when there are more than two levels in \code{x} and
there are \code{NA}s present.}

\item{type}{A character specifying the type of statistical approach:
\itemize{
\item \code{"parametric"}
\item \code{"nonparametric"}
\item \code{"robust"}
\item \code{"bayes"}
}

You can specify just the initial letter.}

\item{paired}{Logical that decides whether the experimental design is
repeated measures/within-subjects or between-subjects. The default is
\code{FALSE}.}

\item{var.equal}{a logical variable indicating whether to treat the
    two variances as being equal. If \code{TRUE} then the pooled
    variance is used to estimate the variance otherwise the Welch
    (or Satterthwaite) approximation to the degrees of freedom is used.}

\item{tr}{Trim level for the mean when carrying out \code{robust} tests. In case
of an error, try reducing the value of \code{tr}, which is by default set to
\code{0.2}. Lowering the value might help.}

\item{bf.prior}{A number between \code{0.5} and \code{2} (default \code{0.707}), the prior
width to use in calculating Bayes factors and posterior estimates. In
addition to numeric arguments, several named values are also recognized:
\code{"medium"}, \code{"wide"}, and \code{"ultrawide"}, corresponding to \emph{r} scale values
of 1/2, sqrt(2)/2, and 1, respectively. In case of an ANOVA, this value
corresponds to scale for fixed effects.}

\item{p.adjust.method}{Adjustment method for \emph{p}-values for multiple
comparisons. Possible methods are: \code{"holm"} (default), \code{"hochberg"},
\code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"}, \code{"none"}.}

\item{k}{Number of digits after decimal point (should be an integer)
(Default: \code{k = 2L}).}

\item{...}{Additional arguments passed to other methods.}
}
\value{
A tibble dataframe containing two columns corresponding to group
levels being compared with each other (\code{group1} and \code{group2}) and \code{p.value}
column corresponding to this comparison. The dataframe will also contain a
\code{p.value.label} column containing a \emph{label} for this \emph{p}-value, in case
this needs to be displayed in \code{ggsignif::geom_ggsignif}. In addition to
these common columns across the different types of statistics, there will
be additional columns specific to the \code{type} of test being run.

This function provides a unified syntax to carry out pairwise comparison
tests and internally relies on other packages to carry out these tests. For
more details about the included tests, see the documentation for the
respective functions:
\itemize{
\item \emph{parametric} : \code{\link[stats:pairwise.t.test]{stats::pairwise.t.test()}} (paired) and
\code{\link[PMCMRplus:gamesHowellTest]{PMCMRplus::gamesHowellTest()}} (unpaired)
\item \emph{non-parametric} :
\code{\link[PMCMRplus:durbinAllPairsTest]{PMCMRplus::durbinAllPairsTest()}} (paired) and
\code{\link[PMCMRplus:kwAllPairsDunnTest]{PMCMRplus::kwAllPairsDunnTest()}} (unpaired)
\item \emph{robust} :
\code{\link[WRS2:rmanova]{WRS2::rmmcp()}} (paired) and \code{\link[WRS2:t1way]{WRS2::lincon()}} (unpaired)
\item \emph{Bayes Factor} : \code{\link[BayesFactor:ttestBF]{BayesFactor::ttestBF()}}
}
}
\description{
Calculate parametric, non-parametric, robust, and Bayes Factor pairwise
comparisons between group levels with corrections for multiple testing.
}
\examples{
\donttest{
if (require("PMCMRplus")) {
  # for reproducibility
  set.seed(123)
  library(ggstatsplot)
  library(statsExpressions) # for data

  # show all columns and make the column titles bold
  # as a user, you don't need to do this; this is just for the package website
  options(tibble.width = Inf, pillar.bold = TRUE, pillar.neg = TRUE, pillar.subtle_num = TRUE)

  #------------------- between-subjects design ----------------------------

  # parametric
  # if `var.equal = TRUE`, then Student's t-test will be run
  pairwise_comparisons(
    data            = mtcars,
    x               = cyl,
    y               = wt,
    type            = "parametric",
    var.equal       = TRUE,
    paired          = FALSE,
    p.adjust.method = "none"
  )

  # if `var.equal = FALSE`, then Games-Howell test will be run
  pairwise_comparisons(
    data            = mtcars,
    x               = cyl,
    y               = wt,
    type            = "parametric",
    var.equal       = FALSE,
    paired          = FALSE,
    p.adjust.method = "bonferroni"
  )

  # non-parametric (Dunn test)
  pairwise_comparisons(
    data            = mtcars,
    x               = cyl,
    y               = wt,
    type            = "nonparametric",
    paired          = FALSE,
    p.adjust.method = "none"
  )

  # robust (Yuen's trimmed means *t*-test)
  pairwise_comparisons(
    data            = mtcars,
    x               = cyl,
    y               = wt,
    type            = "robust",
    paired          = FALSE,
    p.adjust.method = "fdr"
  )

  # Bayes Factor (Student's *t*-test)
  pairwise_comparisons(
    data   = mtcars,
    x      = cyl,
    y      = wt,
    type   = "bayes",
    paired = FALSE
  )

  #------------------- within-subjects design ----------------------------

  # parametric (Student's *t*-test)
  pairwise_comparisons(
    data            = bugs_long,
    x               = condition,
    y               = desire,
    subject.id      = subject,
    type            = "parametric",
    paired          = TRUE,
    p.adjust.method = "BH"
  )

  # non-parametric (Durbin-Conover test)
  pairwise_comparisons(
    data            = bugs_long,
    x               = condition,
    y               = desire,
    subject.id      = subject,
    type            = "nonparametric",
    paired          = TRUE,
    p.adjust.method = "BY"
  )

  # robust (Yuen's trimmed means t-test)
  pairwise_comparisons(
    data            = bugs_long,
    x               = condition,
    y               = desire,
    subject.id      = subject,
    type            = "robust",
    paired          = TRUE,
    p.adjust.method = "hommel"
  )

  # Bayes Factor (Student's *t*-test)
  pairwise_comparisons(
    data       = bugs_long,
    x          = condition,
    y          = desire,
    subject.id = subject,
    type       = "bayes",
    paired     = TRUE
  )
}
}
}
\references{
For more, see:
\url{https://indrajeetpatil.github.io/ggstatsplot/articles/web_only/pairwise.html}
}
