% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fortify-network.R
\name{fortify.network}
\alias{fortify.network}
\title{Fortify method for networks of class \code{\link[network]{network}}}
\usage{
\method{fortify}{network}(
  model,
  data = NULL,
  layout = "fruchtermanreingold",
  weights = NULL,
  arrow.gap = ifelse(network::is.directed(model), 0.025, 0),
  by = NULL,
  scale = TRUE,
  stringsAsFactors = getOption("stringsAsFactors"),
  ...
)
}
\arguments{
\item{model}{an object of class \code{\link[network]{network}}.}

\item{data}{not used by this method.}

\item{layout}{a network layout supplied by \code{\link[sna]{gplot.layout}},
such as \code{"fruchtermanreingold"} (the default), or a two-column matrix
with as many rows as there are nodes in the network, in which case the
matrix is used as nodes coordinates.}

\item{weights}{the name of an edge attribute to use as edge weights when
computing the network layout, if the layout supports such weights (see
'Details').
Defaults to \code{NULL} (no edge weights).}

\item{arrow.gap}{a parameter that will shorten the network edges in order to
avoid overplotting edge arrows and nodes; defaults to \code{0} when the
network is undirected (no edge shortening), or to \code{0.025} when the
network is directed. Small values near \code{0.025} will generally achieve
good results when the size of the nodes is reasonably small.}

\item{by}{a character vector that matches an edge attribute, which will be
used to generate a data frame that can be plotted with
\code{\link[ggplot2]{facet_wrap}} or \code{\link[ggplot2]{facet_grid}}. The
nodes of the network will appear in all facets, at the same coordinates.
Defaults to \code{NULL} (no faceting).}

\item{scale}{whether to (re)scale the layout coordinates. Defaults to
\code{TRUE}, but should be set to \code{FALSE} if \code{layout} contains
meaningful spatial coordinates, such as latitude and longitude.}

\item{stringsAsFactors}{whether vertex and edge attributes should be
converted to factors if they are of class \code{character}. Defaults to
the value of \code{getOption("stringsAsFactors")}, which is \code{TRUE} by
default: see \code{\link[base]{data.frame}}.}

\item{...}{additional parameters for the \code{layout} argument; see
\code{\link[sna]{gplot.layout}} for available options.}
}
\value{
a \code{\link[base]{data.frame}} object.
}
\description{
See the vignette at \url{https://briatte.github.io/ggnetwork/} for a
description of both this function and the rest of the \code{ggnetwork}
package.
}
\details{
\code{fortify.network} will return a warning if it finds duplicated
edges after converting the network to an edge list. Duplicated edges should
be eliminated in favour of single weighted edges before using a network
layout that supports edge weights, such as the Kamada-Kawai force-directed
placement algorithm.
}
\examples{
if (require(ggplot2) && require(network)) {

  # source: ?network::flo
  data(flo)

  # data example
  ggnetwork(flo)

  # plot example
  ggplot(ggnetwork(flo), aes(x, y, xend = xend, yend = yend)) +
    geom_edges(alpha = 0.5) +
    geom_nodes(size = 12, color = "white") +
    geom_nodetext(aes(label = vertex.names), fontface = "bold") +
    theme_blank()

  # source: ?network::emon
  data(emon)

  # data example
  ggnetwork(emon[[1]], layout = "target", niter = 100)

  # data example with edge weights
  ggnetwork(emon[[1]], layout = "kamadakawai", weights = "Frequency")

  # plot example with straight edges
  ggplot(
    ggnetwork(emon[[1]], layout = "kamadakawai", arrow.gap = 0.025),
    aes(x, y, xend = xend, yend = yend)
  ) +
    geom_edges(aes(color = Frequency),
      arrow = arrow(length = unit(10, "pt"), type = "closed")
    ) +
    geom_nodes(aes(size = Formalization)) +
    scale_color_gradient(low = "grey50", high = "tomato") +
    scale_size_area(breaks = 1:3) +
    theme_blank()

  # plot example with curved edges
  ggplot(
    ggnetwork(emon[[1]], layout = "kamadakawai", arrow.gap = 0.025),
    aes(x, y, xend = xend, yend = yend)
  ) +
    geom_edges(aes(color = Frequency),
      curvature = 0.1,
      arrow = arrow(length = unit(10, "pt"), type = "open")
    ) +
    geom_nodes(aes(size = Formalization)) +
    scale_color_gradient(low = "grey50", high = "tomato") +
    scale_size_area(breaks = 1:3) +
    theme_blank()

  # facet by edge attribute
  ggplot(
    ggnetwork(emon[[1]], arrow.gap = 0.02, by = "Frequency"),
    aes(x, y, xend = xend, yend = yend)
  ) +
    geom_edges(arrow = arrow(length = unit(5, "pt"), type = "closed")) +
    geom_nodes() +
    theme_blank() +
    facet_grid(. ~ Frequency, labeller = label_both)

  # user-provided layout
  ggplot(
    ggnetwork(emon[[1]], layout = matrix(runif(28), ncol = 2)),
    aes(x, y, xend = xend, yend = yend)
  ) +
    geom_edges(arrow = arrow(length = unit(5, "pt"), type = "closed")) +
    geom_nodes() +
    theme_blank()
}

}
