#' Return species lists
#'
#' While there are reasons why users may need to check every record meeting their
#' search criteria (i.e. using [atlas_occurrences()]), a common use case
#' is to simply identify which species occur in a specified region, time period,
#' or taxonomic group. This function returns a `data.frame` with one row
#' per species, and columns giving associated taxonomic information.
#'
#' @param request optional `data_request` object: generated by a call to
#' [galah_call()].
#' @param identify `data.frame`: generated by a call to
#' [galah_identify()].
#' @param filter `data.frame`: generated by a call to
#' [galah_filter()]
#' @param geolocate `string`: generated by a call to
#' [galah_geolocate()]
#' @param data_profile `string`: generated by a call to
#' [galah_apply_profile()]
#' @param refresh_cache `logical`: if set to `TRUE` and 
#' `galah_config(caching = TRUE)` then files cached from a previous query will 
#' be replaced by the current query
#' @return An object of class `tbl_df` and `data.frame` (aka a tibble), 
#' returning matching species The `data.frame` object has attributes listing of 
#' the user-supplied arguments of the `data_request` 
#' (i.e., identify, filter, geolocate, columns)
#' @details
#' The primary use case of this function is to extract species-level information
#' given a set of criteria defined by [search_taxa()],
#' [galah_filter()] or [galah_geolocate()]. If the purpose
#' is simply to get taxonomic information that is not restricted by filtering,
#' then [search_taxa()] is more efficient. Similarly, if counts are
#' required that include filter but without returning taxonomic detail, then
#' [atlas_counts()] is more efficient (see examples).
#' @section Examples:
#' ```{r, child = "man/rmd/setup.Rmd"}
#' ```
#' 
#' First, look up a genus of interest in the ALA with [search_taxa()]
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' search_taxa("Heleioporus")
#' ```
#' 
#' It's a good idea to find how many species there are for the taxon you are 
#' interested in - in our case, genus *Heleioporus* - with [atlas_counts()]
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' galah_call() |>
#'   galah_identify("Heleioporus") |>
#'   atlas_counts(type = "species")
#' ```
#'
#' Now get taxonomic information on all species within this genus with 
#' `atlas_species()`
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' # (every row is a species with associated taxonomic data)
#' galah_call() |>
#'   galah_identify("Heleioporus") |>
#'   atlas_species()
#' ```
#' 
#' You can also get taxonomic information on species by piping with `%>%` or 
#' `|>`. Just begin your query with [galah_call()]
#' 
#' ```{r, comment = "#>", collapse = TRUE}
#' galah_call() |>
#'   galah_identify("Heleioporus") |>
#'   atlas_species()
#' ```
#' 
#' @export
atlas_species <- function(request = NULL,
                          identify = NULL, 
                          filter = NULL, 
                          geolocate = NULL,
                          data_profile = NULL,
                          refresh_cache = FALSE
                          ) {
                            
  if(!is.null(request)){
    check_data_request(request)
    
    # update with any other named arguments that have been given
    current_call <- update_galah_call(request, 
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      refresh_cache = refresh_cache
    ) 
  } else {
    current_call <- galah_call(
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      refresh_cache = refresh_cache
    )
  }
     
  # subset to available arguments
  custom_call <- current_call[
     names(current_call) %in% names(formals(atlas_species_internal))]
  class(custom_call) <- "data_request"
  
  # check for caching
  caching <- getOption("galah_config")$caching
  cache_file <- cache_filename("species", unlist(custom_call))
  if (caching && file.exists(cache_file) && !refresh_cache) {
    return(read_cache_file(cache_file))
  }
  
  # run function using do.call
  result <- do.call(atlas_species_internal, custom_call)
  attr(result, "data_type") <- "species"
  attr(result, "data_request") <- custom_call
  
  # if caching requested, save
  if (caching) {
    write_cache_file(object = result, 
                     data_type = "species",
                     cache_file = cache_file)
  }
   
  result   
}
       


atlas_species_internal <- function(request, 
                                   identify, 
                                   filter, 
                                   geolocate,
                                   data_profile,
                                   refresh_cache
                                   ) {
  query <- list()

  if (missing(identify) & missing(filter) & missing(geolocate)) {
    warn("This query will return a list of all species in the ALA.")
  }

  # ensure profile works from galah_filter as well as galah_profile   
  if(is.null(data_profile)){
    if(is.null(filter)){
      profile <- NULL
    }else{
      profile <- extract_profile(filter)
    }
  }else{
    profile <- data_profile$data_profile
  }
  
  query <- build_query(identify, filter, geolocate, profile = profile)
  query$facets <- species_facets()
  query$lookup  <- "true"
  
  tmp <- tempfile()
  data <- atlas_url("records_species") |>
          atlas_download(params = query, cache_file = tmp)

  if(is.null(data)){
    system_down_message("atlas_species")
    return(tibble())
  }else{
  
    if(getOption("galah_config")$atlas == "Australia"){
      # overwrite file with fixed names
      names(data) <- rename_columns(names(data), type = "checklist")
      data <- data[, wanted_columns("checklist")]
    }
    return(data |> as_tibble())
  }
}
