% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xtractor.R
\name{Xtractor}
\alias{Xtractor}
\title{R6 Object for Feature Extraction.}
\format{\code{\link{R6Class}} object.}
\description{
\code{Xtractor} calculates features from raw data for each ID of a grouping variable individually. This process can be parallelized with the package future.
}
\section{Usage}{

\preformatted{
xtractor = Xtractor$new("xtractor")
}
}

\section{Arguments}{


For Xtractor$new():
\describe{
\item{\code{name}: }{(`character(1)`): A user defined name of the Xtractor. All necessary data will be saved on the path: ./fxtract_files/name/}
\item{\code{load}: }{(`logical(1)`): If TRUE, an existing Xtractor will be loaded.}
\item{\code{file.dir}: }{(`character(1)`): Path where all files of the Xtractor are saved. Default is the current working directory.}
}
}

\section{Details}{

All datasets and feature functions are saved in this R6 object.
Datasets will be saved as single RDS files (for each ID) and feature functions are calculated on each single dataset.
A big advantage of this method is that it scales nicely for larger datasets. Data is only read into RAM, when needed.
}

\section{Fields}{

\describe{
\item{\code{error_messages}: }{(`data.frame()`): Active binding. A dataframe with information about error messages.}
\item{\code{ids}: }{(`character()`): Active binding. A character vector with the IDs of the grouping variable.}
\item{\code{features}: }{(`character()`): Active binding. A character vector with the feature functions which were added.}
\item{\code{status}: }{(`data.frame()`): Active binding. A dataframe with an overview over which features are calculated on which datasets.}
\item{\code{results}: }{(`data.frame()`): Active binding. A dataframe with all calculated features of all IDs.}
}
}

\section{Methods}{

\describe{
\item{\code{add_data(data, group_by)}}{[data: (`data.frame` | `data.table`)] A dataframe or data.table which shall be added to the R6 object. \cr
 [group_by: (`character(1)`)] The grouping variable's name of the dataframe. \cr \cr
 This method writes single RDS files for each group.}
\item{\code{preprocess_data(fun)}}{[fun: (`function`)] A function, which has a dataframe as input and a dataframe as output. \cr \cr
 This method loads the RDS files and applies this function on them. The old RDS files are overwritten.}
\item{\code{remove_data(ids)}}{[ids: (`character()`)] One or many IDs of the grouping variable. \cr \cr
 This method deletes the RDS files of the given IDs.}
\item{\code{get_data(ids)}}{[ids: (`character()`)] One or many IDs of the grouping variable. \cr \cr
 This method returns one dataframe with the chosen IDs.}
\item{\code{add_feature(fun, check_fun)}}{[fun: (`function`)] A function, which has a dataframe as input and a named vector or list as output. \cr
 [check_fun: (`logical(1)`)] The function will be checked if it returns a vector or a  list. Defaults to \code{TRUE}. Disable, if calculation takes too long. \cr \cr
 This method adds the feature function to the R6 object. It writes an RDS file of the function which can be retrieved later.}
\item{\code{remove_feature(fun)}}{[fun: (`function | character(1)`)] A function (or the name of the function as character) which shall be removed. \cr \cr
 This method removes the function from the object and deletes all corresponding files and results.}
\item{\code{get_feature(fun)}}{[fun: (`character(1)`)] The name of a function as character. \cr \cr
 This method reads the RDS file of the function. Useful for debugging after loading an Xtractor.}
\item{\code{calc_features(features, ids)}}{[features: (`character()`)] A character vector of the names of the features which shall be calculated. Defaults to all features. \cr
[ids: (`character()`)] One or many IDs of the grouping variable. Defaults to all IDs. \cr \cr
This method calculates all features on the chosen IDs.}
\item{\code{retry_failed_features(features)}}{[features: (`character()`)] A character vector of the names of the features which shall be calculated. Defaults to all features. \cr \cr
This method retries calculation of failed features. Useful if calculation failed because of memory problems.}
\item{\code{plot()}}{[internal] method to print the R6 object.}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
# one feature function
dir = tempdir()
xtractor = Xtractor$new("xtractor", file.dir = dir)
xtractor$add_data(iris, group_by = "Species")
xtractor$ids
fun = function(data) {
  c(mean_sepal_length = mean(data$Sepal.Length))
}
xtractor$add_feature(fun)
xtractor$features
xtractor$calc_features()
xtractor$results
xtractor$status
xtractor

# failing function on only one ID
fun2 = function(data) {
  if ("setosa" \%in\% data$Species) stop("my error")
  c(sd_sepal_length = sd(data$Sepal.Length))
}
xtractor$add_feature(fun2)
xtractor$calc_features()
xtractor$results
xtractor$error_messages
xtractor

# remove feature function
xtractor$remove_feature("fun2")
xtractor$results
xtractor

# remove ID
xtractor$remove_data("setosa")
xtractor$results
xtractor$ids
xtractor

# get datasets and functions
fun3 = xtractor$get_feature("fun")
df = xtractor$get_data()
dplyr_wrapper(data = df, group_by = "Species", fun = fun3)

}
