#' Amplitude Boxplot
#'
#' This function constructs the amplitude boxplot
#'
#' @param fn matrix (\eqn{N} x \eqn{M}) of \eqn{M} aligned functions with \eqn{N} samples
#' @param fmedian vector of \eqn{M} samples of the median calculated using \code{\link{time_warping}} with median
#' @param qn matrix (\eqn{N} x \eqn{M}) of \eqn{M} of aligned srvfs
#' @param qmedian vector of \eqn{M} samples of the median calculated using \code{\link{time_warping}} with median
#' @param time vector of size \eqn{N} describing the sample points
#' @param alpha quantile value (default=.05, i.e., 95\%)
#' @param ka scalar for outlier cutoff (default=1)
#' @param showplot shows plots of functions (default = T)
#' @return Returns a list containing \item{median_y}{median function}
#' \item{Q1}{First quartile}
#' \item{Q3}{Second quartile}
#' \item{Q1a}{First quantile based on alpha}
#' \item{Q3a}{Second quantile based on alpha}
#' \item{minn}{minimum extreme function}
#' \item{maxx}{maximum extreme function}
#' \item{outlier_index}{indexes of outlier functions}
#' @keywords srvf alignment boxplot
#' @references Xie, W., S. Kurtek, K. Bharath, and Y. Sun  (2016). "A Geometric Approach to Visualization
#' of Variability in Functional Data." Journal of the American Statistical Association in press: 1-34.
#' @export
#' @examples
#' data("simu_warp_median")
#' data("simu_data")
#' out <- AmplitudeBoxplot(simu_warp_median$fn, simu_warp_median$fmean,
#'                         simu_warp_median$qn, simu_warp_median$mqn, simu_data$time)
AmplitudeBoxplot <- function(fn, fmedian, qn, qmedian, time, alpha=.05, ka=1,
                             showplot=T){
  M <- nrow(fn)
  N <- ncol(fn)
  lambda <- 0.5

  # translation
  translation <- rep(0,N)
  for (i in 1:N){
    translation[i] <- trapz(time, fn[,i]/(time[M]-time[1]))
  }

  # amplitude median
  median_y <- fmedian

  # compute amplitude distances
  dy <- rep(0, N)
  for (i in 1:N){
    dy[i] <- sqrt(trapz(time, (qmedian-qn[,i])^2))
  }
  dy_ordering <- sort(dy, index.return = T)$ix
  CR_50 <- dy_ordering[1:round(N/2)]  # 50% central region
  m <- max(dy[CR_50])  # maximal amplitude distance with 50% central region

  # identify amplitude quartiles
  angle <- matrix(0, length(CR_50), length(CR_50))
  energy <- matrix(0, length(CR_50), length(CR_50))
  for (i in 1:(length(CR_50)-1)){
    for (j in (i+1):length(CR_50)){
      q1 <- qn[,CR_50[i]] - qmedian
      q3 <- qn[,CR_50[j]] - qmedian
      q1 <- q1/sqrt(trapz(time,q1*q1))
      q3 <- q3/sqrt(trapz(time,q3*q3))
      angle[i,j] <- trapz(time,q1*q3)
      energy[i,j] <- (1-lambda) * (dy[CR_50[i]]/m + dy[CR_50[j]]/m) - lambda * (angle[i,j]+1)
    }
  }
  maxloc <- which(energy == max(energy), arr.ind = TRUE)

  Q1_index <- CR_50[maxloc[1,1]]
  Q3_index <- CR_50[maxloc[1,2]]
  Q1_q <- qn[,Q1_index]
  Q3_q <- qn[,Q3_index]
  Q1 <- fn[,Q1_index]
  Q3 <- fn[,Q3_index]

  # identify amplitude quantile
  dy_ordering <- sort(dy, index.return = T)$ix
  CR_alpha <- dy_ordering[1:round(N*(1-alpha))]  # (1-alpha)% central region
  m <- max(dy[CR_alpha])  # maximal amplitude distance with (1-alpha)% central region
  angle <- matrix(0, length(CR_alpha), length(CR_alpha))
  energy <- matrix(0, length(CR_alpha), length(CR_alpha))
  for (i in 1:(length(CR_alpha)-1)){
    for (j in (i+1):length(CR_alpha)){
      q1 <- qn[,CR_alpha[i]] - qmedian
      q3 <- qn[,CR_alpha[j]] - qmedian
      q1 <- q1/sqrt(trapz(time,q1*q1))
      q3 <- q3/sqrt(trapz(time,q3*q3))
      angle[i,j] <- trapz(time,q1*q3)
      energy[i,j] <- (1-lambda) * (dy[CR_alpha[i]]/m + dy[CR_alpha[j]]/m) - lambda * (angle[i,j]+1)
    }
  }
  maxloc <- which(energy == max(energy), arr.ind = TRUE)

  Q1a_index <- CR_alpha[maxloc[1,1]]
  Q3a_index <- CR_alpha[maxloc[1,2]]
  Q1a_q <- qn[,Q1a_index]
  Q3a_q <- qn[,Q3a_index]
  Q1a <- fn[,Q1a_index]
  Q3a <- fn[,Q3a_index]

  # compute amplitude whiskers
  IQR <- dy[Q1_index] + dy[Q3_index]
  v1 <- Q1_q - qmedian
  v3 <- Q3_q - qmedian
  upper_q <- Q3_q + ka * IQR * v3 / sqrt(trapz(time,v3*v3))
  lower_q <- Q1_q + ka * IQR * v1 / sqrt(trapz(time,v1*v1))
  upper <- cumtrapz(time,upper_q*abs(upper_q))
  lower <- cumtrapz(time,lower_q*abs(lower_q))

  upper_dis <- sqrt(trapz(time,(upper_q-qmedian)^2))
  lower_dis <- sqrt(trapz(time,(lower_q-qmedian)^2))
  whisker_dis <- max(c(upper_dis,lower_dis))

  # indentify amplitude outliers
  outlier_index <- c()
  for (i in 1:N){
    if (dy[dy_ordering[N+1-i]]> whisker_dis){
      outlier_index <- c(outlier_index, dy_ordering[N+1-i])
    } else {
      break
    }
  }

  # identify ampitude extremes
  distance_to_upper <- rep(Inf, N)
  distance_to_lower <- rep(Inf, N)
  out_50_CR <- setdiff(setdiff(1:N, CR_50), outlier_index)
  for (i in 1:length(out_50_CR)){
    j <- out_50_CR[i]
    distance_to_upper[j] = sqrt(trapz(time,(upper_q-qn[,j])^2))
    distance_to_lower[j] = sqrt(trapz(time,(lower_q-qn[,j])^2))
  }
  max_index <- which.min(distance_to_upper)
  min_index <- which.min(distance_to_lower)
  min_q <- qn[,min_index]
  max_q <- qn[,max_index]
  minn <- fn[,min_index]
  maxx <- fn[,max_index]

  if (showplot){
    ymin <- min(c(min(fmedian),min(Q1),min(Q3),min(maxx),min(minn)))
    ymax <- max(c(max(fmedian),max(Q1),max(Q3),max(maxx),max(minn)))
    plot(time, fmedian, col="black",xlab="Time",main="Amplitude Boxplot", type="l", ylim=c(ymin, ymax))
    lines(time, Q1, col="blue")
    lines(time, Q3, col="blue")
    lines(time, Q1a, col="green")
    lines(time, Q3a, col="green")
    lines(time, maxx, col="red")
    lines(time, minn, col="red")

    s <- seq(0,1,length.out=100)
    Fs2 <- matrix(0,length(time), 595)
    Fs2[,1] <- (1-s[1]) * minn + s[1] * Q1
    for (j in 2:100){
      Fs2[,j] <- (1-s[j]) * minn + s[j] * Q1a
      Fs2[,99+j] <- (1-s[j]) * Q1a + s[j] * Q1
      Fs2[,198+j] <- (1-s[j]) * Q1 + s[j] * fmedian
      Fs2[,297+j] <- (1-s[j]) * fmedian + s[j] * Q3
      Fs2[,396+j] <- (1-s[j]) * Q3 + s[j] * Q3a
      Fs2[,495+j] <- (1-s[j]) * Q3a + s[j] * maxx
    }
    d1<-sqrt(trapz(time,(qmedian-Q1_q)^2))
    d1a<-sqrt(trapz(time,(Q1_q-Q1a_q)^2))
    dl<-sqrt(trapz(time,(Q1a_q-min_q)^2))
    d3<-sqrt(trapz(time,(qmedian-Q3_q)^2))
    d3a<-sqrt(trapz(time,(Q3_q-Q3a_q)^2))
    du<-sqrt(trapz(time,(Q3a_q-max_q)^2))
    part1<-seq(-d1-d1a-dl,-d1-d1a,length.out=100)
    part2<-seq(-d1-d1a,-d1,length.out=100)
    part3<-seq(-d1,0,length.out=100)
    part4<-seq(0,d3,length.out=100)
    part5<-seq(d3,d3+d3a,length.out=100)
    part6<-seq(d3+d3a,d3+d3a+du,length.out=100)
    allparts<-c(part1,part2[2:100],part3[2:100],part4[2:100],part5[2:100],part6[2:100])

    if (requireNamespace("plot3Drgl", quietly = TRUE)) {
      p=plot3D::persp3D(x=time,y=allparts,z=Fs2,col=viridis(128),plot=F,main="Amplitude Surface Plot",ticktype="detailed",box=F)+
        plot3D::lines3D(x=time,y=rep(0,M),z=fmedian,col="black",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(-d1,M),z=Q1,col="blue",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(-d1-d1a,M),z=Q1a,col="green",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(-d1-d1a-dl, M),z=minn,col="red",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(d3, M),z=Q3,col="blue",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(d3+d3a, M),z=Q3a,col="green",lwd=6,add=T,plot=F)+
        plot3D::lines3D(x=time,y=rep(d3+d3a+du, M),z=maxx,col="red",lwd=6,add=T,plot=F)
      plot3Drgl::plotrgl()
      rgl::par3d("windowRect"= c(0,0,640,640))
      rgl::grid3d(c("x", "y+", "z"))
      rgl::axes3d(c('x--',"y--",'z'))
      rgl::title3d(xlab="Time",ylab="Distance")
    } else {
      image(time, allparts, Fs2, main="Surface Plot", ylab="", col=viridis(128))
      lines(time, rep(0, M), col="black", lwd=1)
      lines(time, rep(-d1, M), col="blue", lwd=1)
      lines(time, rep(-d1-d1a, M), col="green", lwd=1)
      lines(time, rep(-d1-d1a-dl, M), col="red", lwd=1)
      lines(time, rep(d3, M), col="blue", lwd=1)
      lines(time, rep(d3+d3a, M), col="green", lwd=1)
      lines(time, rep(d3+d3a+du, M), col="red", lwd=1)
    }

  }

  return(list(median_y=median_y,Q1=Q1,Q3=Q3,Q1a=Q1a,Q3a=Q3a,minn=minn,maxx=maxx,
              outlier_index=outlier_index))
}
