% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cleanGeom.r
\name{breakPolys,GVector-method}
\alias{breakPolys,GVector-method}
\alias{breakPolys}
\alias{fixBridges,GVector-method}
\alias{fixBridges}
\alias{fixDangles,GVector-method}
\alias{fixDangles}
\alias{fixLines,GVector-method}
\alias{fixLines}
\alias{remove0,GVector-method}
\alias{remove0}
\alias{removeAngles,GVector-method}
\alias{removeAngles}
\alias{removeBridges,GVector-method}
\alias{removeBridges}
\alias{removeDangles,GVector-method}
\alias{removeDangles}
\alias{removeDupCentroids,GVector-method}
\alias{removeDupCentroids}
\alias{removeDups,GVector-method}
\alias{removeDups}
\alias{removeSmallPolys,GVector-method}
\alias{removeSmallPolys}
\alias{snap,GVector-method}
\alias{snap}
\title{Fix issues with geometries of a vector}
\usage{
\S4method{breakPolys}{GVector}(x)

\S4method{fixBridges}{GVector}(x)

\S4method{fixDangles}{GVector}(x, tolerance = -1)

\S4method{fixLines}{GVector}(x)

\S4method{remove0}{GVector}(x)

\S4method{removeAngles}{GVector}(x)

\S4method{removeBridges}{GVector}(x)

\S4method{removeDangles}{GVector}(x, tolerance = -1)

\S4method{removeDupCentroids}{GVector}(x)

\S4method{removeDups}{GVector}(x)

\S4method{removeSmallPolys}{GVector}(x, tolerance)

\S4method{snap}{GVector}(x, tolerance)
}
\arguments{
\item{x}{A \code{GVector}.}

\item{tolerance}{Numeric or \code{NULL} (default): Minimum distance in map units (degrees for unprojected, usually meters for projected) or minimum area (in meters-squared, regardless of projection).}
}
\value{
A \code{GVector}.
}
\description{
These functions are intended to help fix geometric issues with a \code{GVector}. Note that the functionality of the \code{snap()} and \code{removeAreas()} functions can also be implemented when using \code{\link[=fast]{fast()}} to create a \code{GVector}.
\itemize{
\item \code{breakPolys()}: Break topologically clean areas. This is similar to \code{fixLines()}, except that it does not break loops. Topologically clean vectors may occur if the vector was imported from a format that does not enforce topology, such as a shapefile. Duplicate geometries are automatically removed after breaking.
\item \code{fixBridges()}: Change "bridges" to "islands" (which are topologically incorrect) within geometries to lines.
\item \code{fixDangles()}: Change "dangles" hanging off boundaries to lines if shorter than \code{tolerance} distance. If \code{tolerance} is <0, all dangles will be changed to lines.  Units of \code{tolerance} are in map units, or in degrees for unprojected CRSs. If \code{tolerance} <0, all dangles are removed, and the function will retain only closed loops and lines connecting loops. Dangles will be removed from longest to shortest.
\item \code{fixLines()}: Break lines at intersections and lines that form closed loops.
\item \code{remove0()}: Remove all boundaries and lines with a length of 0.
\item \code{removeAngles()}: Collapse lines that diverge at an angle that is computationally equivalent to 0. This tool often needs to be followed with the \code{break()} and \code{removeDups()} methods.
\item \code{removeBridges()}: Remove "bridges" to "islands" (which are topologically incorrect) within geometries.
\item \code{removeDangles()}: Remove "dangling" lines if shorter than \code{tolerance} distance. If \code{tolerance} is <0, all dangles will be removed. Units of \code{tolerance} are in map units, or in degrees for unprojected CRSs. If \code{tolerance} <0, all dangles are removed, and the function will retain only closed loops and lines connecting loops. Dangles will be removed from longest to shortest.
\item \code{removeDupCentroids()}: Remove duplicated area centroids. In \strong{GRASS}, closed polygons have their attributes mapped to a (hidden) centroid of the polygon.
\item \code{removeDups()}: Remove duplicated features and area centroids.
\item \code{removeSmallPolys()}: Remove polygons smaller than \code{tolerance}. Units of \code{tolerance} are in square meters (regardless of the CRS).
\item \code{snap()}: Snap lines/boundaries to each other if they are less than \code{tolerance} apart. Subsequent removal of dangles may be needed. Units of \code{tolerance} are map units, or degrees for unprojected CRSs.
}
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

# Example data
madRivers <- fastData("madRivers")
rivers <- fast(madRivers)
soam <- rivers[rivers$NAM == "SOAMIANINA"] # select one river for illustration

### Simplify geometry (remove nodes)

vr <- simplifyGeom(soam, tolerance = 2000)
dp <- simplifyGeom(soam, tolerance = 2000, method = "dp")
dpr <- simplifyGeom(soam, tolerance = 2000, method = "dpr", prop = 0.5)
rw <- simplifyGeom(soam, tolerance = 2000, method = "rw")

plot(soam, col = "black", lwd = 3)
plot(vr, col = "blue", add = TRUE)
plot(dp, col = "red", add = TRUE)
plot(dpr, col = "chartreuse", add = TRUE)
plot(rw, col = "orange", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "Vertex reduction",
      "Douglas-Peucker",
      "Douglas-Peucker reduction",
      "Reumann-Witkam"
	),
	col = c("black", "blue", "red", "chartreuse", "orange"),
	lwd = c(3, 1, 1, 1, 1)
)

### Smooth geometry

hermite <- smoothGeom(soam, dist = 2000, angle = 3)
chaiken <- smoothGeom(soam, method = "Chaiken", dist = 2000)

plot(soam, col = "black", lwd = 2)
plot(hermite, col = "blue", add = TRUE)
plot(chaiken, col = "red", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "Hermite",
      "Chaiken"
	),
	col = c("black", "blue", "red"),
	lwd = c(2, 1, 1, 1, 1)
)

### Clean geometry

# Has no effect on this vector!
noDangs <- removeDangles(soam, tolerance = 10000)

plot(soam, col = "black", lwd = 2)
plot(noDangs, col = "red", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "No dangles"
	),
	lwd = c(2, 1),
	col = c("black", "red")
)

}
}
\seealso{
\code{\link[terra:topology]{terra::topology()}}, \code{\link[=fillHoles]{fillHoles()}}, \code{\link[terra:topology]{terra::removeDupNodes()}}, \emph{Details} section in \code{\link[=fast]{fast()}}, \code{\link[=simplifyGeom]{simplifyGeom()}}, \code{\link[=smoothGeom]{smoothGeom()}}, \strong{GRASS} manual page for module \code{v.clean} (see \code{grassHelp("v.clean")})
}
