% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/undirected_sbm.R
\name{sbm}
\alias{sbm}
\title{Create an undirected stochastic blockmodel object}
\usage{
sbm(
  n,
  k = NULL,
  B = NULL,
  ...,
  pi = rep(1/k, k),
  edge_distribution = c("poisson", "bernoulli"),
  sort_nodes = TRUE
)
}
\arguments{
\item{n}{The number of nodes in the network. Must be
a positive integer. This argument is required.}

\item{k}{(mixing matrix) The number of blocks in the blockmodel.
Use when you don't want to specify the mixing-matrix by hand.
When \code{k} is specified, the elements of \code{B} are drawn
randomly from a \code{Uniform(0, 1)} distribution.
This is subject to change, and may not be reproducible.
\code{k} defaults to \code{NULL}. You must specify either \code{k}
or \code{B}, but not both.}

\item{B}{(mixing matrix) A \code{k} by \code{k} matrix of block connection
probabilities. The probability that a node in block \code{i} connects
to a node in community \code{j} is \code{Poisson(B[i, j])}. Must be square
a square matrix. \code{matrix} and \code{Matrix} objects are both
acceptable. If \code{B} is not symmetric, it will be
symmetrized via the update \code{B := B + t(B)}. Defaults to \code{NULL}.
You must specify either \code{k} or \code{B}, but not both.}

\item{...}{
  Arguments passed on to \code{\link[=undirected_factor_model]{undirected_factor_model}}
  \describe{
    \item{\code{expected_degree}}{If specified, the desired expected degree
of the graph. Specifying \code{expected_degree} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Do not specify both
\code{expected_degree} and \code{expected_density} at the same time.}
    \item{\code{expected_density}}{If specified, the desired expected density
of the graph. Specifying \code{expected_density} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Do not specify both
\code{expected_degree} and \code{expected_density} at the same time.}
  }}

\item{pi}{(relative block probabilities) Relative block
probabilities. Must be positive, but do not need to sum
to one, as they will be normalized internally.
Must match the dimensions of \code{B} or \code{k}. Defaults to
\code{rep(1 / k, k)}, or a balanced blocks.}

\item{edge_distribution}{Either \code{"poisson"} or \code{"bernoulli"}. The
default is \code{"poisson"}, in which case the SBM can be a
multigraph, i.e. multiple edges between the same two nodes
are allowed. If \code{edge_distribution == "bernoulli"} only a
single edge is allowed between any pair of nodes. See Section 2.3
of Rohe et al (2017) for details.}

\item{sort_nodes}{Logical indicating whether or not to sort the nodes
so that they are grouped by block. Useful for plotting.
Defaults to \code{TRUE}.}
}
\value{
An \code{undirected_sbm} S3 object, which is a subclass of the
\code{\link[=dcsbm]{dcsbm()}} object, with one additional field.
\itemize{
\item \code{edge_distribution}: Either "poisson" or "bernoulli".
}
}
\description{
To specify a stochastic blockmodel, you must specify
the number of nodes (via \code{n}), the mixing matrix (via \code{k} or \code{B}),
and the relative block probabilites (optional, via \code{pi}).
We provide sane defaults for most of these options to enable
rapid exploration, or you can invest the effort
for more control over the model parameters. We \strong{strongly recommend}
setting the \code{expected_degree} or \code{expected_density} argument
to avoid large memory allocations associated with
sampling large, dense graphs.
}
\details{
A stochastic block is equivalent to a degree-corrected
stochastic blockmodel where the degree heterogeneity parameters
have all been set equal to 1.
}
\examples{

set.seed(27)

lazy_sbm <- sbm(n = 1000, k = 5, expected_density = 0.01)
lazy_sbm

# by default we get a multigraph (i.e. multiple edges are
# allowed between the same two nodes). using bernoulli edges
# will with an adjacency matrix with only zeroes and ones

bernoulli_sbm <- sbm(
  n = 5000,
  k = 300,
  edge_distribution = "bernoulli",
  expected_degree = 80
)

bernoulli_sbm

edgelist <- sample_edgelist(bernoulli_sbm)
edgelist

A <- sample_sparse(bernoulli_sbm)

# only zeroes and ones!
sign(A)

}
\seealso{
Other stochastic block models: 
\code{\link{dcsbm}()},
\code{\link{planted_partition}()}

Other undirected graphs: 
\code{\link{dcsbm}()},
\code{\link{erdos_renyi}()},
\code{\link{planted_partition}()}
}
\concept{stochastic block models}
\concept{undirected graphs}
