% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/euler.R
\name{euler}
\alias{euler}
\alias{euler.default}
\alias{euler.data.frame}
\alias{euler.matrix}
\alias{euler.table}
\alias{euler.list}
\title{Area-Proportional Euler Diagrams}
\usage{
euler(combinations, ...)

\method{euler}{default}(combinations, input = c("disjoint", "union"),
  shape = c("circle", "ellipse"), control = list(), ...)

\method{euler}{data.frame}(combinations, weights = NULL, by = NULL, ...)

\method{euler}{matrix}(combinations, ...)

\method{euler}{table}(combinations, ...)

\method{euler}{list}(combinations, ...)
}
\arguments{
\item{combinations}{Set relationships as a named numeric vector, matrix, or
data.frame. (See the methods (by class) section for details.)}

\item{...}{Arguments passed down to other methods.}

\item{input}{The type of input: disjoint class combinations
(\code{disjoint}) or unions (\code{union}).}

\item{shape}{The geometric shape used in the diagram: \code{circle} or \code{ellipse}.}

\item{control}{A list of control parameters.
\itemize{
\item \code{extraopt}: Should the more thorough optimizer (currently
\code{\link[GenSA:GenSA]{GenSA::GenSA()}}) kick in (provided \code{extraopt_threshold} is exceeded)? The
default is \code{TRUE} for ellipses and three sets and \code{FALSE} otherwise.
\item \code{extraopt_threshold}: The threshold, in terms of \code{diagError}, for when
the extra optimizer kicks in. This will almost always slow down the
process considerably. A value of 0 means
that the extra optimizer will kick in if there is \emph{any} error. A value of
1 means that it will never kick in. The default is \code{0.001}.
\item \code{extraopt_control}: A list of control parameters to pass to the
extra optimizer, such as \code{max.call}. See \code{\link[GenSA:GenSA]{GenSA::GenSA()}}.
}}

\item{weights}{A numeric vector of weights of the same length as \code{by} and
the number of rows in \code{combinations}.}

\item{by}{A factor or character matrix to be used in \code{\link[base:by]{base::by()}} to
split the data.frame or matrix of set combinations.}
}
\value{
A list object of class \code{'euler'} with the following parameters.
\item{coefficients}{A matrix of \code{h} and \code{k} (x and y-coordinates for the
centers of the
shapes) and, for circles, \code{r} for radii or, for ellipses, semiaxes \code{a}
and \code{b} and rotation angle \code{phi}.}
\item{original.values}{Set relationships provided by the user.}
\item{fitted.values}{Set relationships in the solution.}
\item{residuals}{Residuals.}
\item{diagError}{The largest absolute residual in percentage points
between the original and fitted areas.}
\item{stress}{The stress of the solution, computed as the sum of squared
residuals over the total sum of squares.}
}
\description{
Fit Euler diagrams (a generalization of Venn diagrams) using numerical
optimization to find exact or approximate solutions to a specification of set
relationships. The shape of the diagram may be a circle or an ellipse.
}
\details{
If the input is a matrix or data frame and argument \code{by} is specified,
the function returns a list of euler diagrams.

The function minimizes the sums of squared errors between the disjoint areas
in the euler diagram and the user's input, namely

\deqn{\sum_{i=1}^{n} (y_i - \hat{y}_i) ^ 2,}{\sum (orig - fit) ^ 2,}

where \eqn{\hat{y}}{fit} are estimates of \eqn{y} that are currently being
explored.

The stress statistic from \pkg{venneuler} is returned to give an indication
of the goodness of the fit:

\deqn{
  \frac{
    \sum_{i=1}^{n} (y_i - \hat{y}_i) ^ 2}{\sum_{i=1}^{n} y_i ^ 2},
  }{
  \sum (fit - original) ^ 2 / \sum original ^ 2,
}

where \eqn{\hat{y}}{fit} are ordinary least squares estimates from the
regression of the fitted areas on the original areas that are currently being
explored.

\code{euler()} also returns \code{diagError} and \code{regionError} from
\emph{eulerAPE}. \code{regionError} is computed as

\deqn{
    \left| \frac{y_i}{\sum y_i} - \frac{\hat{y}_i}{\sum \hat{y}_i}\right|.
  }{
    max|fit / \sum fit  - original / \sum original|.
 }

\code{diagError} is simply the maximum of regionError.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: A named numeric vector, with
combinations separated by an ampersand, for instance \code{A&B = 10}.
Missing combinations are treated as being 0.

\item \code{data.frame}: A data.frame of logicals, two-level factors (see examples).

\item \code{matrix}: A matrix that can be converted to a data.frame of logicals
(as in the description above) via \code{\link[base:as.data.frame.matrix]{base::as.data.frame.matrix()}}.

\item \code{table}: A table with \code{max(dim(x)) < 3}.

\item \code{list}: A list of vectors, each vector giving the contents of
that set (with no duplicates). Vectors in the list do not need to be named.
}}

\examples{
# First fit the euler specification
fit <- euler(c("A" = 1, "B" = 0.4, "C" = 3, "A&B" = 0.2))

# Then plot it
plot(fit)

# Same result as above
euler(c("A" = 1, "B" = 0.4, "C" = 3,
        "A&B" = 0.2, "A&C" = 0, "B&C" = 0,
        "A&B&C" = 0))

# A euler diagram from a list of sample spaces (the list method)
euler(list(A = c("a", "ab", "ac", "abc"),
           B = c("b", "ab", "bc", "abc"),
           C = c("c", "ac", "bc", "abc")))

# Using the matrix method
mat <- cbind(A = sample(c(TRUE, TRUE, FALSE), size = 50, replace = TRUE),
             B = sample(c(TRUE, FALSE), size = 50, replace = TRUE))
euler(mat)

# Using grouping via the 'by' argument
dat <- data.frame(
  A = sample(c(TRUE, FALSE), size = 100, replace = TRUE),
  B = sample(c(TRUE, TRUE, FALSE), size = 100, replace = TRUE),
  gender = sample(c("Men", "Women"), size = 100, replace = TRUE),
  nation = sample(c("Sweden", "Denmark"), size = 100, replace = TRUE)
)

euler(dat[, 1:2], by = dat[, 3:4])

# A set with no perfect solution
euler(c("a" = 3491, "b" = 3409, "c" = 3503,
        "a&b" = 120, "a&c" = 114, "b&c" = 132,
        "a&b&c" = 50))

# The table method
plot(euler(as.table(apply(Titanic, 2:4, sum))))
}
\references{
Wilkinson L. Exact and Approximate Area-Proportional Circular
Venn and Euler Diagrams. IEEE Transactions on Visualization and Computer
Graphics (Internet). 2012 Feb (cited 2016 Apr 9);18(2):321-31. Available
from:
\url{http://doi.org/10.1109/TVCG.2011.56}

Micallef L, Rodgers P. eulerAPE: Drawing Area-Proportional 3-Venn Diagrams
Using Ellipses. PLOS ONE (Internet). 2014 Jul (cited 2016 Dec 10);9(7):e101717. Available from:
\url{http://dx.doi.org/10.1371/journal.pone.0101717}
}
\seealso{
\code{\link[=plot.euler]{plot.euler()}}, \code{\link[=print.euler]{print.euler()}}
}
