% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query.R
\name{EsgfQuery}
\alias{EsgfQuery}
\alias{query_esgf}
\title{Query CMIP6 data using ESGF search RESTful API}
\usage{
query_esgf(host = "https://esgf-node.llnl.gov/esg-search")
}
\arguments{
\item{host}{The URL to the ESGF Search API service. This should be the URL of
the ESGF search service excluding the final endpoint name. Usually
this is \verb{http://<hostname>/esg-search}. Default is set to the
\href{http://esgf-node.llnl.gov}{LLNL (Lawrence Livermore National Laboratory) Index Node},
which is \code{"https://esgf-node.llnl.gov/esg-search"}.}
}
\description{
The Earth System Grid Federation (ESGF) is an international collaboration for
the software that powers most global climate change research, notably
assessments by the Intergovernmental Panel on Climate Change (IPCC).

The ESGF search service exposes RESTful APIs that can be used by clients to
query the contents of the underlying search index, and return results
matching the given constraints. The documentation of the APIs can be found
using this \href{https://esgf.github.io/esg-search/ESGF_Search_RESTful_API.html}{link}

\code{EsgfQuery} is the workhorse for dealing with ESGF search services.
}
\section{\code{EsgfQuery} object}{
\code{query_esgf()} returns an \code{EsgfQuery} object, which is an \link[R6:R6Class]{R6}
object with quite a few methods that can be classified into 3 categories:
\itemize{
\item \strong{Value listing}: methods to list all possible values of facets, shards,
etc.
\item \strong{Parameter getter & setter}: methods to get the query parameter values or
set them before sending the actual query to the ESGF search services.
\item \strong{Query responses}: methods to collect results for the query response.
}
\subsection{Value listing}{

When creating an \code{EsgfQuery} object, a
\href{https://esgf.github.io/esg-search/ESGF_Search_RESTful_API.html#facet-listings}{facet listing query}
is sent to the index node to get all available facets and shards for the
default project (CMIP6).
\code{EsgfQuery} object provides three value-listing methods to extract data from
the response of the facet listing query:
\itemize{
\item \href{#method-EsgfQuery-list_all_facets}{\code{EsgfQuery$list_all_facets()}}:
List all available facet names.
\item \href{#method-EsgfQuery-list_all_shards}{\code{EsgfQuery$list_all_shards()}}:
List all available shards.
\item \href{#method-EsgfQuery-list_all_values}{\code{EsgfQuery$list_all_values()}}:
List all available values of a specific facet.
}
}

\subsection{Parameter getter & setter}{

The ESGF search services support a lot of parameters. The \code{EsgfQuery}
contains dedicated methods to set values for most of them, including:
\itemize{
\item Most common keywords:
\href{#method-EsgfQuery-facets}{\code{facets}},
\href{#method-EsgfQuery-offset}{\code{offset}},
\href{#method-EsgfQuery-limit}{\code{limit}},
\href{#method-EsgfQuery-fields}{\code{fields}},
\href{#method-EsgfQuery-type}{\code{type}},
\href{#method-EsgfQuery-replica}{\code{replica}},
\href{#method-EsgfQuery-latest}{\code{latest}},
\href{#method-EsgfQuery-distrib}{\code{distrib}}
and
\href{#method-EsgfQuery-shards}{\code{shards}}.
\item Most common facets:
\href{#method-EsgfQuery-project}{\code{project}},
\href{#method-EsgfQuery-activity_id}{\code{activity_id}},
\href{#method-EsgfQuery-experiment_id}{\code{experiment_id}},
\href{#method-EsgfQuery-source_id}{\code{source_id}},
\href{#method-EsgfQuery-variable_id}{\code{variable_id}},
\href{#method-EsgfQuery-frequency}{\code{frequency}},
\href{#method-EsgfQuery-variant_label}{\code{variant_label}},
\href{#method-EsgfQuery-nominal_resolution}{\code{nominal_resolution}}
and
\href{#method-EsgfQuery-data_node}{\code{data_node}}.
}

All methods act in a similar way:
\itemize{
\item If input is given, the corresponding parameter is set and the updated
\code{EsgfQuery} object is returned.
\itemize{
\item This makes it possible to chain different parameter setters, e.g.
\code{EsgfQuery$project("CMIP6")$frequency("day")$limit(1)} sets the parameter
\code{project}, \code{frequency} and \code{limit} sequentially.
\item For parameters that want character inputs, you can put a preceding \code{!} to
negate the constraints, e.g. \code{EsgfQuery$project(!"CMIP6")} searches for
all projects except for \code{CMIP6}.
}
\item If no input is given, the current parameter value is returned. For example,
directly calling \code{EsgfQuery$project()} returns the current value of the
\code{project} parameter. The returned value can be two types:
\itemize{
\item \code{NULL}, i.e. there is no constraint on the corresponding parameter
\item An \code{EsgfQueryParam} object which is essentially a list of three elements:
\itemize{
\item \code{value}: The input values
\item \code{negate}: Whether there is a preceding \code{!} in the input
\item \code{name}: The parameter name
}
}
}

Despite methods for specific keywords and facets, you can specify arbitrary
query parameters using
\href{#method-EsgfQuery-params}{\code{EsgfQuery$params()}} method. For
details on the usage, please see the
\href{#method-EsgfQuery-params}{documentation}.
}

\subsection{Query responses}{

The query is not sent unless related methods are called:
\itemize{
\item \href{#method-EsgfQuery-count}{\code{EsgfQuery$count()}}: Count the total
number of records that match the query.
\itemize{
\item You can return only the total number of matched record by calling
\code{EsgfQuery$count(facets = FALSE)}
\item You can also count the matched records for specified facets, e.g.
\code{EsgfQuery$count(facets = c("source_id", "activity_id"))}
}
\item \href{#method-EsgfQuery-collect}{\code{EsgfQuery$collect()}}: Collect the
query results and format it into a \link[data.table:data.table]{data.table}
}
}

\subsection{Other helpers}{

\code{EsgfQuery} object also provide several other helper functions:
\itemize{
\item \href{#method-EsgfQuery-build_cache}{\code{EsgfQuery$build_cache()}}:
By default, \code{EsgfQuery$build_cache()} is called when initialize a new
\code{EsgfQuery} object. So in general, there is no need to call this
separately. Basically, \code{EsgfQuery$build_cahce()} sends a
\href{https://esgf.github.io/esg-search/ESGF_Search_RESTful_API.html#facet-listings}{facet listing query}
to the index node and stores the response internally. The response contains
all available facets and shards and is used as a source for validating user
input for parameter setters.
\item \href{#method-EsgfQuery-url}{\code{EsgfQuery$url()}}: Returns the actual
query URL or the wget script URL which can be used to download all files
matching the given constraints..
\item \href{#method-EsgfQuery-response}{\code{EsgfQuery$response()}}: Returns the
actual response of
\href{#method-EsgfQuery-count}{\code{EsgfQuery$count()}} and
\href{#method-EsgfQuery-collect}{\code{EsgfQuery$collect()}}. It is a named
list generated from the JSON response using \code{\link[jsonlite:fromJSON]{jsonlite::fromJSON()}}.
\item \href{#method-EsgfQuery-print}{\code{EsgfQuery$print()}}: Print a summary
of the current \code{EsgfQuery} object including the host URL, the built time of
facet cache and all query parameters.
}
}
}

\examples{

## ------------------------------------------------
## Method `EsgfQuery$new`
## ------------------------------------------------

\dontrun{
q <- EsgfQuery$new(host = "https://esgf-node.llnl.gov/esg-search")
q
}

## ------------------------------------------------
## Method `EsgfQuery$build_cache`
## ------------------------------------------------

\dontrun{
q$build_cache()
}

## ------------------------------------------------
## Method `EsgfQuery$list_all_facets`
## ------------------------------------------------

\dontrun{
q$list_all_facets()
}

## ------------------------------------------------
## Method `EsgfQuery$list_all_shards`
## ------------------------------------------------

\dontrun{
q$list_all_shards()
}

## ------------------------------------------------
## Method `EsgfQuery$list_all_values`
## ------------------------------------------------

\dontrun{
q$list_all_values()
}

## ------------------------------------------------
## Method `EsgfQuery$project`
## ------------------------------------------------

\dontrun{
# get current value
q$project()

# set the parameter
q$project("CMIP6")

# negate the project constraints
q$project(!"CMIP6")

# remove the parameter
q$project(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$activity_id`
## ------------------------------------------------

\dontrun{
# get current value
q$activity_id()

# set the parameter
q$activity_id("ScenarioMIP")

# negate the constraints
q$activity_id(!c("CFMIP", "ScenarioMIP"))

# remove the parameter
q$activity_id(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$experiment_id`
## ------------------------------------------------

\dontrun{
# get current value
q$experiment_id()

# set the parameter
q$experiment_id(c("ssp126", "ssp585"))

# negate the constraints
q$experiment_id(!c("ssp126", "ssp585"))

# remove the parameter
q$experiment_id(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$source_id`
## ------------------------------------------------

\dontrun{
# get current value
q$source_id()

# set the parameter
q$source_id(c("BCC-CSM2-MR", "CESM2"))

# negate the constraints
q$source_id(!c("BCC-CSM2-MR", "CESM2"))

# remove the parameter
q$source_id(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$variable_id`
## ------------------------------------------------

\dontrun{
# get current value
q$variable_id()

# set the parameter
q$variable_id(c("tas", "pr"))

# negate the constraints
q$variable_id(!c("tas", "pr"))

# remove the parameter
q$variable_id(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$frequency`
## ------------------------------------------------

\dontrun{
# get current value
q$frequency()

# set the parameter
q$frequency(c("1hr", "day"))

# negate the constraints
q$frequency(!c("1hr", "day"))

# remove the parameter
q$frequency(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$variant_label`
## ------------------------------------------------

\dontrun{
# get current value
q$variant_label()

# set the parameter
q$variant_label(c("r1i1p1f1", "r1i2p1f1"))

# negate the constraints
q$variant_label(!c("r1i1p1f1", "r1i2p1f1"))

# remove the parameter
q$variant_label(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$nominal_resolution`
## ------------------------------------------------

\dontrun{
# get current value
q$nominal_resolution()

# set the parameter
q$nominal_resolution(c("100 km", "1x1 degree"))

# negate the constraints
q$nominal_resolution(!c("100 km", "1x1 degree"))

# remove the parameter
q$nominal_resolution(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$data_node`
## ------------------------------------------------

\dontrun{
# get current value
q$data_node()

# set the parameter
q$data_node("esg.lasg.ac.cn")

# negate the constraints
q$data_node(!"esg.lasg.ac.cn")

# remove the parameter
q$data_node(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$facets`
## ------------------------------------------------

\dontrun{
# get current value
q$facets()

# set the facets
q$facets(c("activity_id", "source_id"))

# use all available facets
q$facets("*")
}

## ------------------------------------------------
## Method `EsgfQuery$fields`
## ------------------------------------------------

\dontrun{
# get current value
q$fields()

# set the fields
q$fields(c("activity_id", "source_id"))

# use all available fields
q$fields("*")

# remove the parameter
# act the same as above because the default `fields` in ESGF search
# services is `*` if `fields` is not specified
q$fields(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$shards`
## ------------------------------------------------

\dontrun{
# get current value
q$shards()

# set the parameter
q$shards("localhost:8983/solr/datasets")

# negate the constraints
q$shards(!"localhost:8983/solr/datasets")

# only applicable for distributed queries
q$distrib(FALSE)$shards("localhost:8983/solr/datasets") # Error

# remove the parameter
q$shards(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$replica`
## ------------------------------------------------

\dontrun{
# get current value
q$replica()

# set the parameter
q$replica(TRUE)

# remove the parameter
q$replica(NULL)
}

## ------------------------------------------------
## Method `EsgfQuery$latest`
## ------------------------------------------------

\dontrun{
# get current value
q$latest()

# set the parameter
q$latest(TRUE)
}

## ------------------------------------------------
## Method `EsgfQuery$type`
## ------------------------------------------------

\dontrun{
# get current value
q$type()

# set the parameter
q$type("Dataset")
}

## ------------------------------------------------
## Method `EsgfQuery$limit`
## ------------------------------------------------

\dontrun{
# get current value
q$limit()

# set the parameter
q$limit(10L)

# `limit` is reset to 10,000 if input is greater than that
q$limit(10000L) # warning
}

## ------------------------------------------------
## Method `EsgfQuery$offset`
## ------------------------------------------------

\dontrun{
# get current value
q$offset()

# set the parameter
q$offset(0L)
}

## ------------------------------------------------
## Method `EsgfQuery$distrib`
## ------------------------------------------------

\dontrun{
# get current value
q$distrib()

# set the parameter
q$distrib(TRUE)
}

## ------------------------------------------------
## Method `EsgfQuery$params`
## ------------------------------------------------

\dontrun{
# get current values
# default is an empty list (`list()`)
q$params()

# set the parameter
q$params(table_id = c("3hr", "day"), member_id = "00")
q$params()

# reset existing parameters
q$frequency("day")
q$params(frequency = "mon")
q$frequency() # frequency value has been changed using $params()

# negating the constraints is also supported
q$params(table_id = !c("3hr", "day"))

# use NULL to remove all parameters
q$params(NULL)$params()
}

## ------------------------------------------------
## Method `EsgfQuery$url`
## ------------------------------------------------

\dontrun{
q$url()

# get the wget script URL
q$url(wget = TRUE)

# You can download the wget script using the URL directly. For
# example, the code below downloads the script and save it as
# 'wget.sh' in R's temporary folder:
download.file(q$url(TRUE), file.path(tempdir(), "wget.sh"), mode = "wb")

}

## ------------------------------------------------
## Method `EsgfQuery$count`
## ------------------------------------------------

\dontrun{
# get the total number of matched records
q$count(NULL) # or q$count(facets = FALSE)

# count records for specific facets
q$facets(c("activity_id", "source_id"))$count()

# same as above
q$count(facets = c("activity_id", "source_id"))
}

## ------------------------------------------------
## Method `EsgfQuery$collect`
## ------------------------------------------------

\dontrun{
q$fields("source_id")
q$collect()
}

## ------------------------------------------------
## Method `EsgfQuery$response`
## ------------------------------------------------

\dontrun{
q$response()
}

## ------------------------------------------------
## Method `EsgfQuery$print`
## ------------------------------------------------

\dontrun{
q$print()
}
}
\author{
Hongyuan Jia
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-EsgfQuery-new}{\code{EsgfQuery$new()}}
\item \href{#method-EsgfQuery-build_cache}{\code{EsgfQuery$build_cache()}}
\item \href{#method-EsgfQuery-list_all_facets}{\code{EsgfQuery$list_all_facets()}}
\item \href{#method-EsgfQuery-list_all_shards}{\code{EsgfQuery$list_all_shards()}}
\item \href{#method-EsgfQuery-list_all_values}{\code{EsgfQuery$list_all_values()}}
\item \href{#method-EsgfQuery-project}{\code{EsgfQuery$project()}}
\item \href{#method-EsgfQuery-activity_id}{\code{EsgfQuery$activity_id()}}
\item \href{#method-EsgfQuery-experiment_id}{\code{EsgfQuery$experiment_id()}}
\item \href{#method-EsgfQuery-source_id}{\code{EsgfQuery$source_id()}}
\item \href{#method-EsgfQuery-variable_id}{\code{EsgfQuery$variable_id()}}
\item \href{#method-EsgfQuery-frequency}{\code{EsgfQuery$frequency()}}
\item \href{#method-EsgfQuery-variant_label}{\code{EsgfQuery$variant_label()}}
\item \href{#method-EsgfQuery-nominal_resolution}{\code{EsgfQuery$nominal_resolution()}}
\item \href{#method-EsgfQuery-data_node}{\code{EsgfQuery$data_node()}}
\item \href{#method-EsgfQuery-facets}{\code{EsgfQuery$facets()}}
\item \href{#method-EsgfQuery-fields}{\code{EsgfQuery$fields()}}
\item \href{#method-EsgfQuery-shards}{\code{EsgfQuery$shards()}}
\item \href{#method-EsgfQuery-replica}{\code{EsgfQuery$replica()}}
\item \href{#method-EsgfQuery-latest}{\code{EsgfQuery$latest()}}
\item \href{#method-EsgfQuery-type}{\code{EsgfQuery$type()}}
\item \href{#method-EsgfQuery-limit}{\code{EsgfQuery$limit()}}
\item \href{#method-EsgfQuery-offset}{\code{EsgfQuery$offset()}}
\item \href{#method-EsgfQuery-distrib}{\code{EsgfQuery$distrib()}}
\item \href{#method-EsgfQuery-params}{\code{EsgfQuery$params()}}
\item \href{#method-EsgfQuery-url}{\code{EsgfQuery$url()}}
\item \href{#method-EsgfQuery-count}{\code{EsgfQuery$count()}}
\item \href{#method-EsgfQuery-collect}{\code{EsgfQuery$collect()}}
\item \href{#method-EsgfQuery-response}{\code{EsgfQuery$response()}}
\item \href{#method-EsgfQuery-print}{\code{EsgfQuery$print()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-new"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-new}{}}}
\subsection{Method \code{new()}}{
Create a new EsgfQuery object

When initialization, a
\href{https://esgf.github.io/esg-search/ESGF_Search_RESTful_API.html#facet-listings}{facet listing query}
is sent to the index node to get all available facets and shards.
This information will be used to validate inputs for \code{activity_id},
\code{scource_id} facets and etc.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$new(host = "https://esgf-node.llnl.gov/esg-search")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{host}}{The URL to the ESGF Search API service. This should be
the URL of the ESGF search service excluding the final
endpoint name. Usually this is \verb{http://<hostname>/esg-search}.
Default is to ses the \href{http://esgf-node.llnl.gov}{LLNL (Lawrence Livermore National Laboratory)} Index Node, which is
\code{"https://esgf-node.llnl.gov/esg-search"}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An \code{EsgfQuery} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q <- EsgfQuery$new(host = "https://esgf-node.llnl.gov/esg-search")
q
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-build_cache"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-build_cache}{}}}
\subsection{Method \code{build_cache()}}{
Build facet cache used for input validation

A facet cache is data that is fetched using a
\href{https://esgf.github.io/esg-search/ESGF_Search_RESTful_API.html#facet-listings}{facet listing query}
to the index node. It contains all available facets and shards that
can be used as parameter values within a specific project.

By default, \verb{$build_cache()} is called when initialize a new
\code{EsgfQuery} object for the default project (CMIP6). So in general,
there is no need to call this method, unless that you want to
rebuild the cache again with different projects after calling
\href{#method-EsgfQuery-project}{\code{$project()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$build_cache()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{EsgfQuery} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$build_cache()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-list_all_facets"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-list_all_facets}{}}}
\subsection{Method \code{list_all_facets()}}{
List all available facet names
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$list_all_facets()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$list_all_facets()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-list_all_shards"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-list_all_shards}{}}}
\subsection{Method \code{list_all_shards()}}{
List all available shards
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$list_all_shards()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$list_all_shards()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-list_all_values"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-list_all_values}{}}}
\subsection{Method \code{list_all_values()}}{
List all available values of a specific facet
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$list_all_values(facet)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{facet}}{A single string giving the facet name.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A named character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$list_all_values()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-project"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-project}{}}}
\subsection{Method \code{project()}}{
Get or set the \code{project} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$project(value = "CMIP6")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{"CMIP6"}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$project(!c("CMIP5", "CMIP6"))} searches for all \code{project}s except for \code{CMIP5} and \code{CMIP6}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$project()

# set the parameter
q$project("CMIP6")

# negate the project constraints
q$project(!"CMIP6")

# remove the parameter
q$project(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-activity_id"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-activity_id}{}}}
\subsection{Method \code{activity_id()}}{
Get or set the \code{activity_id} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$activity_id(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$activity_id(!c("C4MIP", "GeoMIP"))} searches for all \code{activity_id}s except for \code{C4MIP} and \code{GeoMIP}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$activity_id()

# set the parameter
q$activity_id("ScenarioMIP")

# negate the constraints
q$activity_id(!c("CFMIP", "ScenarioMIP"))

# remove the parameter
q$activity_id(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-experiment_id"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-experiment_id}{}}}
\subsection{Method \code{experiment_id()}}{
Get or set the \code{experiment_id} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$experiment_id(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$experiment_id(!c("ssp126", "ssp245"))} searches for all \code{experiment_id}s except for \code{ssp126} and \code{ssp245}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$experiment_id()

# set the parameter
q$experiment_id(c("ssp126", "ssp585"))

# negate the constraints
q$experiment_id(!c("ssp126", "ssp585"))

# remove the parameter
q$experiment_id(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-source_id"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-source_id}{}}}
\subsection{Method \code{source_id()}}{
Get or set the \code{source_id} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$source_id(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$source_id(!c("CESM2", "CESM2-FV2"))} searches for all \code{source_id}s except for \code{CESM2} and \code{CESM2-FV2}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$source_id()

# set the parameter
q$source_id(c("BCC-CSM2-MR", "CESM2"))

# negate the constraints
q$source_id(!c("BCC-CSM2-MR", "CESM2"))

# remove the parameter
q$source_id(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-variable_id"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-variable_id}{}}}
\subsection{Method \code{variable_id()}}{
Get or set the \code{variable_id} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$variable_id(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$variable_id(!c("tas", "pr"))} searches for all \code{variable_id}s except for \code{tas} and \code{pr}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$variable_id()

# set the parameter
q$variable_id(c("tas", "pr"))

# negate the constraints
q$variable_id(!c("tas", "pr"))

# remove the parameter
q$variable_id(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-frequency"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-frequency}{}}}
\subsection{Method \code{frequency()}}{
Get or set the \code{frequency} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$frequency(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$frequency(!c("day", "mon"))} searches for all \code{frequency}s except for \code{day} and \code{mon}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$frequency()

# set the parameter
q$frequency(c("1hr", "day"))

# negate the constraints
q$frequency(!c("1hr", "day"))

# remove the parameter
q$frequency(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-variant_label"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-variant_label}{}}}
\subsection{Method \code{variant_label()}}{
Get or set the \code{variant_label} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$variant_label(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$variant_label(!c("r1i1p1f1", "r2i1p1f1"))} searches for all \code{variant_label}s except for \code{r1i1p1f1} and \code{r2i1p1f1}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$variant_label()

# set the parameter
q$variant_label(c("r1i1p1f1", "r1i2p1f1"))

# negate the constraints
q$variant_label(!c("r1i1p1f1", "r1i2p1f1"))

# remove the parameter
q$variant_label(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-nominal_resolution"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-nominal_resolution}{}}}
\subsection{Method \code{nominal_resolution()}}{
Get or set the \code{nominal_resolution} facet parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$nominal_resolution(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$nominal_resolution(!c("50 km", "1x1 degree"))} searches for all \code{nominal_resolution}s except for \code{50 km} and \code{1x1 degree}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$nominal_resolution()

# set the parameter
q$nominal_resolution(c("100 km", "1x1 degree"))

# negate the constraints
q$nominal_resolution(!c("100 km", "1x1 degree"))

# remove the parameter
q$nominal_resolution(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-data_node"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-data_node}{}}}
\subsection{Method \code{data_node()}}{
Get or set the \code{data_node} parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$data_node(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The  parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.  Note that you can put a preceding \code{!} to negate the facet constraints. For example, \code{$data_node(!c("cmip.bcc.cma.cn", "esg.camscma.cn"))} searches for all \code{data_node}s except for \code{cmip.bcc.cma.cn} and \code{esg.camscma.cn}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$data_node()

# set the parameter
q$data_node("esg.lasg.ac.cn")

# negate the constraints
q$data_node(!"esg.lasg.ac.cn")

# remove the parameter
q$data_node(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-facets"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-facets}{}}}
\subsection{Method \code{facets()}}{
Get or set the \code{facets} parameter for facet counting query.

Note that \verb{$facets()} only affects
\href{#method-EsgfQuery-count}{\code{$count()}}
method when sending a query of facet counting.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$facets(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}. The special notation \code{"*"} can be used to indicate that all available facets should be considered.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$facets()

# set the facets
q$facets(c("activity_id", "source_id"))

# use all available facets
q$facets("*")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-fields"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-fields}{}}}
\subsection{Method \code{fields()}}{
Get or set the \code{fields} parameter.

By default, all available metadata fields are returned for each
query. \verb{$facets()} can be used to limit the number of fields returned
in the query response.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$fields(value = "*")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{"*"}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}. The special notation \code{"*"} can be used to indicate that all available fields should be considered.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$fields()

# set the fields
q$fields(c("activity_id", "source_id"))

# use all available fields
q$fields("*")

# remove the parameter
# act the same as above because the default `fields` in ESGF search
# services is `*` if `fields` is not specified
q$fields(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-shards"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-shards}{}}}
\subsection{Method \code{shards()}}{
Get or set the \code{shards} parameter.

By default, a distributed query targets all ESGF Nodes. \verb{$shards()}
can be used to execute a distributed search that targets only one or
more specific nodes.

All available shards can be retrieved using
\href{#method-EsgfQuery-list_all_shards}{\code{$list_all_shards()}}
method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$shards(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A character vector or \code{NULL}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$shards()

# set the parameter
q$shards("localhost:8983/solr/datasets")

# negate the constraints
q$shards(!"localhost:8983/solr/datasets")

# only applicable for distributed queries
q$distrib(FALSE)$shards("localhost:8983/solr/datasets") # Error

# remove the parameter
q$shards(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-replica"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-replica}{}}}
\subsection{Method \code{replica()}}{
Get or set the \code{replica} parameter.

By default, a query returns all records (masters and replicas)
matching the search criteria, i.e. \verb{$replica(NULL)}.
To return only master records, use \verb{$replica(FALSE)}; to return only
replicas, use \verb{$replica(TRUE)}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$replica(value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{NULL}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A flag or \code{NULL}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$replica()

# set the parameter
q$replica(TRUE)

# remove the parameter
q$replica(NULL)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-latest"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-latest}{}}}
\subsection{Method \code{latest()}}{
Get or set the \code{latest} parameter.

By default, a query to the ESGF search services returns only the very
last, up-to-date version of the matching records, i.e.
\verb{$latest(TRUE)}. You can use \verb{$latest(FALSE)} to return all versions.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$latest(value = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{TRUE}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A flag.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$latest()

# set the parameter
q$latest(TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-type"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-type}{}}}
\subsection{Method \code{type()}}{
Get or set the \code{type} parameter.

There are three types in total: \code{Dataset}, \code{File} or \code{Aggregation}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$type(value = "Dataset")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{"Dataset"}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A string.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$type()

# set the parameter
q$type("Dataset")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-limit"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-limit}{}}}
\subsection{Method \code{limit()}}{
Get or set the \code{limit} parameter.

\verb{$limit()} can be used to limit the number of records to return.
Note that the maximum number of records to return per query for ESGF
search services is 10,000. A warning is issued if input value is
greater than that. In this case, \code{limit} will be reset to 10,000.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$limit(value = 10L)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{10}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item An integer.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$limit()

# set the parameter
q$limit(10L)

# `limit` is reset to 10,000 if input is greater than that
q$limit(10000L) # warning
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-offset"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-offset}{}}}
\subsection{Method \code{offset()}}{
Get or set the \code{offset} parameter.

If the query returns records that exceed the
\href{#method-EsgfQuery-limit}{\code{limit}} number,
\verb{$offset()} can be used to paginate through the available results.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$offset(value = 0L)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{0}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item An integer.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$offset()

# set the parameter
q$offset(0L)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-distrib"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-distrib}{}}}
\subsection{Method \code{distrib()}}{
Get or set the \code{distrib} facet

By default, the query is sent to all ESGF Nodes, i.e.
\verb{$distrib(TRUE)}.
\verb{$distrib(FALSE)} can be used to execute the query only on the
target node.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$distrib(value = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{The facet parameter value. Default: \code{TRUE}.
There are two options:
\itemize{
\item If \code{value} is not given, current value is returned.
\item A flag.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{ \item If \code{value} is given, the modified \code{EsgfQuery} object. \item Otherwise, an \code{EsgfQueryParam} object which is essentially a list of three elements: \itemize{ \item \code{value}: input values. \item \code{negate}: Whether there is a preceding \code{!}. \item \code{name}: Parameter name. } }
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current value
q$distrib()

# set the parameter
q$distrib(TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-params"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-params}{}}}
\subsection{Method \code{params()}}{
Get or set other parameters.

\verb{$params()} can be used to specify other parameters that do not have
a dedicated method, e.g. \code{version}, \code{master_id}, etc. It can also be
used to overwrite existing parameter values specified using methods
like \href{#method-EsgfQuery-activity_id}{\code{$activity_id()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$params(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Parameter values to set. There are three options:
\itemize{
\item If not given, existing parameters that do not have a dedicated
method are returned.
\item If \code{NULL}, all existing parameters that do not have a dedicated
method are removed.
\item A named vector, e.g. \verb{$params(score = 1, table_id = "day")} will
set \code{score} to \code{1} and \code{table_id} to \code{day}.
The \code{!} notation can still be used to negate the constraints, e.g.
\verb{$params(table_id = !c("3hr", "day"))} searches for all \code{table_id}
except for \verb{3hr} and \code{day}.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{
\item If parameters are specified, the modified \code{EsgfQuery} object,
invisibly.
\item Otherwise, an empty list for \verb{$params(NULL)} or a list of
\code{EsgfQueryParam} objects.
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get current values
# default is an empty list (`list()`)
q$params()

# set the parameter
q$params(table_id = c("3hr", "day"), member_id = "00")
q$params()

# reset existing parameters
q$frequency("day")
q$params(frequency = "mon")
q$frequency() # frequency value has been changed using $params()

# negating the constraints is also supported
q$params(table_id = !c("3hr", "day"))

# use NULL to remove all parameters
q$params(NULL)$params()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-url"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-url}{}}}
\subsection{Method \code{url()}}{
Get the URL of actual query or wget script
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$url(wget = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{wget}}{Whether to return the URL of the wget script that can be
used to download all files matching the given constraints.
Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$url()

# get the wget script URL
q$url(wget = TRUE)

# You can download the wget script using the URL directly. For
# example, the code below downloads the script and save it as
# 'wget.sh' in R's temporary folder:
download.file(q$url(TRUE), file.path(tempdir(), "wget.sh"), mode = "wb")

}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-count"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-count}{}}}
\subsection{Method \code{count()}}{
Send a query of facet counting and fetch the results
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$count(facets = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{facets}}{\code{NULL}, a flag or a character vector. There are three
options:
\itemize{
\item If \code{NULL} or \code{FALSE}, only the total number of matched records is
returned.
\item If \code{TRUE}, the value of \href{#method-EsgfQuery-facets}{\code{$facets()}}
is used to limit the facets. This is the default value.
\item If a character vector, it is used to limit the facets.
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\itemize{
\item If \code{facets} equals \code{NULL} or \code{FALSE}, or \verb{$facets()} returns \code{NULL},
an integer.
\item Otherwise, a named list with the first element always being \code{total}
which is the total number of matched records. Other elements have
the same length as input facets and are all named integer vectors.
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get the total number of matched records
q$count(NULL) # or q$count(facets = FALSE)

# count records for specific facets
q$facets(c("activity_id", "source_id"))$count()

# same as above
q$count(facets = c("activity_id", "source_id"))
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-collect"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-collect}{}}}
\subsection{Method \code{collect()}}{
Send the actual query and fetch the results

\verb{$collect()} sends the actual query to the ESGF search services and
returns the results in a \link[data.table:data.table]{data.table::data.table}. The columns depend
on the value of query type and \code{fields} parameter.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$collect()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A \link[data.table:data.table]{data.table}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$fields("source_id")
q$collect()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-response"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-response}{}}}
\subsection{Method \code{response()}}{
Get the response of last sent query

The response of the last sent query is always stored internally and
can be retrieved using \verb{$response()}. It is a named list generated
from the JSON response using \code{\link[jsonlite:fromJSON]{jsonlite::fromJSON()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$response()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A named list.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$response()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EsgfQuery-print"></a>}}
\if{latex}{\out{\hypertarget{method-EsgfQuery-print}{}}}
\subsection{Method \code{print()}}{
Print a summary of the current \code{EsgfQuery} object

\verb{$print()} gives the summary of current \code{EsgfQuery} object including
the host URL, the built time of facet cache and all query parameters.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EsgfQuery$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The \code{EsgfQuery} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
q$print()
}
}
\if{html}{\out{</div>}}

}

}
}
