\name{epicurve}

\alias{epicurve.hours}
\alias{epicurve.dates}
\alias{epicurve.weeks}
\alias{epicurve.months}
\alias{epicurve.table}
%- Also NEED an '\alias' for EACH other topic documented here.

\title{
  Construct an epidemic curve
}

\description{
  Construct an epidemic curve
}

\usage{
epicurve.dates(x, format = "\%Y-\%m-\%d", strata = NULL,
               min.date, max.date, before = 7, after = 7,
               width = 1, space = 0, tick = TRUE,
               tick.offset = 0.5, segments = FALSE, ...)

epicurve.weeks(x, format = "\%Y-\%m-\%d", strata = NULL,
               min.date, max.date, before = 7, after = 7,
               width = 1, space = 0, tick = TRUE,
               tick.offset = 0.5, segments = FALSE,
               origin = as.Date("1970-01-01"), sunday = TRUE, ...)

epicurve.months(x, format = "\%Y-\%m-\%d", strata = NULL,
                min.date, max.date, before = 31, after = 31,
                width = 1, space = 0, tick = TRUE,
                tick.offset = 0.5, segments = FALSE,
                origin = as.Date("1970-01-01"), ...)

epicurve.hours(x, mindt, maxdt, strata = NULL, half.hour = FALSE,
               width = 1, space = 0, tick = TRUE,
               tick.offset = 0.5, segments = FALSE, ...) 

epicurve.table(x, width = 1, space = 0, tick = TRUE,
               tick.offset = 0.5, segments = FALSE, ...)
  

}
%- maybe also 'usage' for other objects documented here.

\arguments{
  \item{x}{
    character vector of dates
  }
  \item{format}{
    date format of \code{x}; default is of form "2004-08-10" 
  }
  \item{strata}{
    [optional] categorical vector (character or factor) for stratifying
    \code{x} 
  }
  \item{min.date}{
    [optional] minimum calendar date for plotting x-axis of an epidemic
    curve; should be of the form of "2004-08-10"; if no date is
    specified, then several days are subtracted from the minimum date in
    \code{x} as specified by the \code{before} option
  }
  \item{max.date}{
    [optional] maximum calendar date for plotting x-axis of an epidemic
    curve; should be of the form of "2004-08-10"; if no date is
    specified, then several days are added to the maximum date in
    \code{x} as specified by the \code{after} option
  }
  \item{before}{
    if \code{min.date} is not specified, then these number of days are
    subtracted from the minimum date in \code{x} for plotting minimum
    calendar date for epidemic curve
  }
  \item{after}{
    if \code{max.date} is not specified, then these number of days are
    added to the maximum date in \code{x} for plotting maximum calendar
    date for epidemic curve 
  }
  \item{mindt}{
    [required] Date-time object in standard format that will form the
    lower boundary of the hour or half-hour time categories. The
    \code{mindt} option must less than or equal to the minimum value in
    \code{x}, and must be rounded off to the nearst hour for hour
    categories (e.g., HH:00:00) or rounded  off to the nearest half-hour
    for half-hour categories (e.g., HH:30:00). 
  }
  \item{maxdt}{
    [required] Date-time object in standard format that will form the
    upper boundary of the hour or half-hour time categories. The
    \code{maxdt} option must greater than or equal to the minimum value
    in \code{x}, and must be rounded off to the nearst hour for hour
    categories (e.g., HH:00:00) or rounded off to the nearest half-hour
    for half-hour categories (e.g., HH:30:00). 
  }
  \item{half.hour}{
    Set to TRUE for half-hour categories in \code{epicurve.hours}.
  }
  
  \item{width}{
    width of bars in the epidemic curve; this value is passed to
    \code{barplot} function
  }
  \item{space}{
    space between bars in the epidemic curve; this value is passed to
    \code{barplot} function
  }
  \item{tick}{
    adds tick marks to the x-axis (default = TRUE)
  }
  \item{tick.offset}{
    offsets tick marks so that they plotted between the bars 
  }
  \item{segments}{
    segments bars so that each box represents one case
  }
  \item{origin}{
    allows user to specify an alternative origin for Julian dates that
    are generated by this function (default = "1970-01-01")
  }
  \item{sunday}{
    First day of the week is Sunday (default = TRUE); setting to FALSE
    makes Monday the first day of the week
  }
  \item{...}{
    options are passed to the \code{barplot} function
  }
  
}

\details{
  These functions makes plotting epidemic curves much easier in
  R. Normally, to plot an epidemic curve in R, one must do the
  following: (1) have disease onset dates in some calendar date format,
  (2) convert these onset dates into a factor with the levels specified
  by the range of calendar dates for the x-axis of the epidemic curve,
  (3) convert this factor into a table (with or without stratification),
  (4) use this table as an argument in the \code{barplot} function to plot the
  epidemic curve, and (5) make final adjustments (labels, titles, etc.).

  Why use the \code{barplot} function? Strictly speaking, an epidemic
  curve is a histogram displaying the distribution of onset times which
  are categorized into, for example, dates. However, histogram functions
  seems to work better for measurements that our continuous (e.g.,
  height, weight). In contrast, epidemic curves are constructed from
  onset time data that has been categorized into days, weeks, or
  months. For this type of categorical data, the \code{barplot} does a
  better job. The caveat, however, is that we need to specify the range
  of possible calendar dates, weeks, or months in order to construct an
  appropriate plot. To do this we convert the data into a factor with
  the levels specified by the possible calendar date values.

  To make this whole process much easier, and to generate additional
  data that can be use for labeling your epidemic curve, the
  \code{epicurve} functions were created.
}

\value{
  \item{epicurve.dates}{returns list:}
  \item{$dates}{
    input dates are converted to standard calendar date format
  }
  \item{$dates2}{
    input dates are also converted to a factor with levels determined by
    the calendar dates (\code{$cdates}) used to plot the epidemic curve
  }
  \item{$xvals}{
    x-axis numeric values used for plotting the epidemic curve; this
    comes from the \code{barplot} function
  }
  \item{$cdates}{
    the calendar dates used for plotting the epidemic curve
  }
  \item{$cmday}{
    the day of the mon (1-31) for the calendar dates used for plotting
    the x-axis of the epidemic curve  
  }
  \item{$cmonth}{
    the months (Jan, Feb, Mar, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }
  \item{$cyear}{
    the years (e.g., 1996, 2001, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }

  \item{epicurve.weeks}{returns list:}
  \item{$dates}{
    input dates are converted to standard calendar date format
  }
  \item{$firstday}{
    first day of the week is reported
  }
  \item{$week}{
    week of the year (1-53); note that week 52 or 53 can represent both
    last week of a year but also the first few days at the beginning of
    the year 
  }
  \item{$stratum}{
    the Julian date for the mid-week day of the \code{$week} value
  }
  \item{$stratum2}{
    the Julian date for the mid-week day of the \code{$week} value
    converted to a factor with levels determined by the Julian dates
    (\code{$cstratum}) used to plot the epidemic curve 
  }
  \item{$stratum3}{
    the mid-week day of the \code{$week} value converted to standard
    calendar dates  
  }
  \item{$xvals}{
    x-axis numeric values used for plotting the epidemic curve; this
    comes from the \code{barplot} function
  }
  \item{$cweek}{
    the week of the year used for plotting the x-axis of the epidemic
    curve    
  }
  \item{$cstratum}{
    the Julian date for the mid-week day of the \code{$cweek} value used
    for plotting the x-axis of the epidemic curve 
  }
  \item{$cstratum2}{
    the standard calendar date for the mid-week day of the \code{$cweek}
    value used for plotting the x-axis of the epidemic curve
  }
  \item{$cmday}{
    the day of the mon (1-31) for the calendar dates used for plotting
    the x-axis of the epidemic curve 
  }
  \item{$cmonth}{
    the months (Jan, Feb, Mar, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }
  \item{$cyear}{
    the years (e.g., 1996, 2001, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }

  \item{epicurve.months}{returns list:}
  \item{$dates}{
    input dates are converted to standard calendar date format
  }
  \item{$mon}{
    month of the year (1-12)
  }
  \item{$month}{
    month of the year (Jan, Feb, Mar, ...)
  }
  \item{$stratum}{
    the Julian date for the mid-month day of the \code{$mon} value
  }
  \item{$stratum2}{
    the Julian date for the mid-month day of the \code{$mon} value
    converted to a factor with levels determined by the Julian dates
    (\code{$cstratum})used to plot the epidemic curve
  }
  \item{$stratum3}{
    the mid-month day of the \code{$mon} value converted to standard
    calendar dates  
  }
  \item{$xvals}{
    x-axis numeric values used for plotting the epidemic curve; this
    comes from the \code{barplot} function
  }
  \item{$cmon}{
    the month of the year (1-12) used for plotting the x-axis of the
    epidemic curve
  }
  \item{$cmonth}{
    the months (Jan, Feb, Mar, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }
  \item{$cstratum}{
    the Julian date for the mid-month day of the \code{$cmonth} value
    used for plotting the x-axis of the epidemic curve 
  }
  \item{$cstratum2}{
    the standard calendar date for the mid-month day of the
    \code{$cmonth} value used for plotting the x-axis of the epidemic
    curve      
  }
  \item{$cmday}{
    the day of the mon (1-31) for the calendar dates used for plotting
    the x-axis of the epidemic curve 
  }
  \item{$cyear}{
    the years (e.g., 1996, 2001, ...) for the calendar dates used for
    plotting the x-axis of the epidemic curve
  }

  \item{epicurve.hours}{returns list:}
  \item{$ct}{
    Date-time object that contains the number of seconds since the
    beginning of 1970 as a numeric vector and produced by
    \code{\link{as.POSIXct}}. You can use \code{\link{as.POSIXlt}} to
    convert this output in human legible (already done by this
    function). 
  }
  \item{$sec}{seconds}
  \item{$min}{minutes}
  \item{$hour}{hours (0-23)}
  \item{$hour12}{hours (1-12)}
  \item{$stratum}{number of hours or 1/2 hours since beginning of 1970}
  \item{$stratum2}{
    factor (categorical variable) with number of hours of 1/2 hours
    since beginning of 1970 using \code{$cstratum} as the levels
  }
  \item{$stratum3}{
    factor (categorical variable) in standard date-time format
    indicating number of hours or 1/2 hours since beginning of 1970
    using 
  }
  \item{$xvals}{}
  \item{$cstratum}{levels for creating \code{$stratum2} factor}
  \item{$cstratum2}{levels for creating \code{$stratum3} factor}
  \item{$csec}{seconds from \code{$cstratum2}}
  \item{$cmin}{minutes from \code{$cstratum2}}
  \item{$chour}{hours from \code{$cstratum2} in 24-hour format}
  \item{$chour12}{hours from \code{$cstratum2} in 12-hour format}
  \item{$campm}{corresponding 'AM' or 'PM' for \code{$chour12}}
  \item{$campm2}{corresponding 'am' or 'pm' for \code{$chour12}}
  \item{$cweekday}{day of the week for \code{$cstratum2}}
  \item{$cwkday}{abbreviated day of the week for \code{$cstratum2}}
  \item{$cmday}{day of the month for \code{$cstratum2}}
  \item{$cmonth}{month for \code{$cstratum2}}
  \item{$cmon}{abbreviated month for \code{$cstratum2}}
  \item{$cyear}{year for \code{$cstratum2}}
  \item{$half.hour}{
    FALSE (default) for 1-hour categories; TRUE for 1/2-hour categories
  }
  
  \item{epicurve.table}{returns numeric vector:}
  \item{xvals}{
    x-axis numeric values used for plotting the epidemic curve; this
    comes from the \code{barplot} function 
  }
}

\references{none}

\author{Tomas Aragon, \email{aragon@berkeley.edu},
  \url{http://www.medepi.net/aragon}} 

\note{Visit \url{http://www.epitools.net} for the latest.}

\seealso{
  \code{\link[graphics]{barplot}}, \code{\link[base]{strptime}}
}
\examples{
##epicurve.dates
sampdates <- seq(as.Date("2004-07-15"), as.Date("2004-09-15"), 1)
x <- sample(sampdates, 100, rep=TRUE)
xs <- sample(c("Male","Female"), 100, rep=TRUE)
epicurve.dates(x)
epicurve.dates(x, strata = xs)
rr <- epicurve.dates(x, strata = xs, segments = TRUE,
                     axisnames = FALSE)
axis(1, at = rr$xvals, labels = rr$cmday, tick = FALSE, line = 0)
axis(1, at = rr$xvals, labels = rr$cmonth, tick = FALSE, line = 1)

##epicurve.weeks
sampdates <- seq(as.Date("2004-07-15"), as.Date("2004-09-15"), 1)
x <- sample(sampdates, 100, rep=TRUE)
xs <- sample(c("Male","Female"), 100, rep=TRUE)
epicurve.weeks(x)

epicurve.weeks(x, strata = xs)

rr <- epicurve.weeks(x, strata = xs, segments = TRUE)
rr


##epicurve.months
dates <- c("1/1/04", "1/2/04", "1/3/04", "1/4/04", "1/5/04",
"1/6/04", "1/7/04", "1/8/04", "1/9/04", "1/10/04", NA, "1/12/04",
"1/14/04", "3/5/04", "5/5/04", "7/6/04", "8/18/04", "12/13/05",
"1/5/05", "4/6/05", "7/23/05", "10/3/05")
aw <- as.month(dates, format = "\%m/\%d/\%y")
aw
aw2 <- as.month(dates, format = "\%m/\%d/\%y", min.date="2003-01-01")
aw2

##epicurve.hours
data(oswego)
## create vector with meal date and time
mdt <- paste("4/18/1940", oswego$meal.time)
mdt[1:10]
## convert into standard date and time
meal.dt <- strptime(mdt, "\%m/\%d/\%Y \%I:\%M \%p")
meal.dt[1:10]
## create vector with onset date and time
odt <- paste(paste(oswego$onset.date,"/1940",sep=""), oswego$onset.time)
odt[1:10]
## convert into standard date and time
onset.dt <- strptime(odt, "\%m/\%d/\%Y \%I:\%M \%p")
onset.dt[1:10]      

##set colors
col3seq.d <- c("#43A2CA", "#A8DDB5", "#E0F3DB")

par.fin <- par()$fin
par(fin=c(5,3.4))

##1-hour categories
xv <- epicurve.hours(onset.dt, "1940-04-18 12:00:00", "1940-04-19 12:00:00",
                     axisnames = FALSE, axes = FALSE, ylim = c(0,11),
                     col = col3seq.d[1], segments =  TRUE,
                     strata = oswego$sex)

hh <- xv$chour12==3 | xv$chour12== 6 | xv$chour12== 9
hh2 <- xv$chour12==12
hh3 <- xv$chour12==1
hlab <- paste(xv$chour12,xv$campm2,sep="")
hlab2 <- paste(xv$cmonth,xv$cmday)
axis(1, at = xv$xval[hh], labels = xv$chour12[hh], tick = FALSE, line = -.2)
axis(1, at = xv$xval[hh2], labels = hlab[hh2], tick = FALSE, line = -.2)
axis(1, at = xv$xval[hh3], labels = hlab2[hh3], tick = FALSE, line = 1.0)
axis(2, las = 1)
title(main = "Figure 1. Cases of Gastrointestinal Illness
by Time of Onset of Symptoms (Hour Category)
Oswego County, New York, April 18-19, 2004",
      xlab = "Time of Onset",
      ylab = "Cases")

##1/2-hour categories
xv <- epicurve.hours(onset.dt, "1940-04-18 12:00:00", "1940-04-19 12:00:00",
                     axisnames = FALSE, axes = FALSE, ylim = c(0,11),
                     col = col3seq.d[1], segments =  TRUE,
                     half.hour = TRUE, strata = oswego$sex)
hh <- xv$chour12==3 | xv$chour12== 6 | xv$chour12== 9
hh2 <- xv$chour12==12
hh3 <- xv$chour12==1
hlab <- paste(xv$chour12,xv$campm2,sep="")
hlab2 <- paste(xv$cmonth,xv$cmday)
axis(1, at = xv$xval[hh], labels = xv$chour12[hh], tick = FALSE, line = -.2)
axis(1, at = xv$xval[hh2], labels = hlab[hh2], tick = FALSE, line = -.2)
axis(1, at = xv$xval[hh3], labels = hlab2[hh3], tick = FALSE, line = 1.0)
axis(2, las = 1)
title(main = "Figure 2. Cases of Gastrointestinal Illness
by Time of Onset of Symptoms (1/2 Hour Category)
Oswego County, New York, April 18-19, 2004",
      xlab = "Time of Onset",
      ylab = "Cases")

par(fin=par.fin)


##epicurve.table
xvec <- c(1,2,3,4,5,4,3,2,1)
epicurve.table(xvec)

names(xvec) <- 1991:1999
epicurve.table(xvec)

xmtx <- rbind(xvec, xvec)
rownames(xmtx) <- c("Male", "Female")
epicurve.table(xmtx)

epicurve.table(xmtx, seg = TRUE)


}

\keyword{hplot}
