#' Extract cues from list of weightmatrices.
#' 
#' @export
#' @param wmlist A list with weightmatrices, generated by 
#' \code{\link{RWlearning}} or \code{\link{updateWeights}}.
#' @param extra.check Logical: whether or not to collect all cues from all 
#' weightmatrices in the list. Note that this slows down the process and 
#' should not result in different findings. Default is FALSE.
#' @return Vector with cues.
#' @seealso \code{\link{getOutcomes}}, \code{\link{getValues}}
#' @author Jacolien van Rij
#' @examples
#' # load example data:
#' data(dat)
#' # prepare training data:
#' dat$Cues <- paste("BG", dat$Shape, dat$Color, sep="_")
#' dat$Outcomes <- dat$Category
#' dat$Frequency <- dat$Frequency1
#' train <- createTrainingData(dat)
#' # learning:
#' wm <- RWlearning(train)
#' # retrieve cues from wm list:
#' getCues(wm)
#' # or this version (which takes more time):
#' system.time({getCues(wm, extra.check=TRUE)})
#' system.time({getCues(wm)})
#' 

getCues <- function(wmlist, extra.check=FALSE){
	cues <- rownames(wmlist[[length(wmlist)]])
	if(extra.check){
		allcues <- sort(unique(unlist(lapply(wmlist, function(x){return(rownames(x))}))))
		c1 <- cues[!cues %in% allcues]
		c2 <- allcues[!allcues %in% cues]

		if(length(c1)>0){
			warning(sprintf("Not all cues in allcues: %s\n", paste(c1, collapse=",")))
		}
		if(length(c2)>0){
			warning(sprintf("Not all allcues in cues: %s\n", paste(c2, collapse=",")))
		}
		return(sort(unique(c(cues, allcues))))
	}else{
		return(cues)
	}
}
