
#' LeafletJS Map
#'
#' Builds an interactive map of locations for any species
#' @param ee_obj A ecoengine object of type \code{observations} as generated by \code{\link{ee_observations}}
#' @param  dest Location where the html file and geojson file should be stored. Default is the temp directory
#' @param  title Title of the map.
#' @param  incl.data Default is \code{TRUE}. Writes geoJSON data into the html file to get around security restrictions in browsers like Google Chrome. Set to \code{FALSE} to read from a separate local geoJSON file.
#' @export
#' @keywords map
#' @import leafletR
#' @examples \dontrun{
#' lynx_data <- ee_observations(genus = "Lynx", georeferenced = TRUE, page = "all", quiet = TRUE)
#' ee_map(lynx_data, title = "Lynx distribution map")
#' # Now let's map out foxes
#' vulpes <- ee_observations(genus = "vulpes", georeferenced = TRUE, quiet = TRUE)
#' ee_map(vulpes, title = "Fox distributions")
#' # You can also map photos
#' ee_map(ee_photos(georeferenced = TRUE))
#'}
ee_map <- function(ee_obj, dest = tempdir(), title = "Ecoengine species map", incl.data = TRUE) {
	assert_that(ee_obj$type == "observations" | ee_obj$type == "photos")


	dest <- ifelse(is.null(dest), tempdir(), dest)
	species_data <- ee_obj$data
	lat_location <- which(names(species_data) == "latitude")
	lon_location <- which(names(species_data) == "longitude")

	species_data <- species_data[complete.cases(species_data[, lat_location:lon_location]), ]


	assert_that(nrow(species_data) > 0)



if(ee_obj$type == "observations") {	
ee_geo <- toGeoJSON(data = species_data, name = "temp", dest = dest, lat.lon = c(lat_location, lon_location))	
	num_species <- length(unique(species_data$scientific_name))
	cols <- c("#8D5C00", "#2F5CD7","#E91974", "#3CB619","#7EAFCC",
"#4F2755","#F5450E","#264C44","#3EA262","#FA43C9","#6E8604","#631D0E","#EE8099","#E5B25A",
"#0C3D8A","#9E4CD3","#195C7B","#9F8450","#7A0666","#BBA3C5","#F064B4","#108223","#553502",
"#17ADE7","#83C445","#C52817","#626302","#9F9215","#6CCD78","#BF3704")
	pal <- cols[1:num_species]
	sty <- styleCat(prop = "scientific_name", val = unique(species_data$scientific_name), style.val = pal, fill.alpha = 1, alpha = 1, rad = 4, leg = "Scientific Name")
	map <- leaflet(ee_geo, base.map="tls", style = sty, popup = "scientific_name", dest = dest, title = title, incl.data = incl.data)
	}
	else {
	ee_geo <- toGeoJSON(data = species_data, name = "temp", dest = dest, lat.lon = c(lat_location, lon_location))	
	map <- leaflet(ee_geo, base.map="tls", popup = "url", dest = dest, title = title, incl.data = incl.data)

	}
	browseURL(map)
}






