% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract_dynamic_raster.R
\name{extract_dynamic_raster}
\alias{extract_dynamic_raster}
\title{Extract temporally dynamic rasters of explanatory variables.}
\usage{
extract_dynamic_raster(
  dates,
  spatial.ext,
  datasetname,
  bandname,
  spatial.res.metres,
  GEE.math.fun,
  user.email,
  varname,
  temporal.res,
  temporal.direction,
  save.directory,
  save.drive.folder,
  resume = TRUE
)
}
\arguments{
\item{dates}{a character string, vector of dates in format "YYYY-MM-DD".}

\item{spatial.ext}{the spatial extent for the extracted raster. Object from which extent can be
extracted of class \code{SpatExtent}, \code{SpatRaster}, an \code{sf} polygon or a
numeric vector listing xmin, xmax, ymin and ymax in order.}

\item{datasetname}{a character string, the Google Earth Engine dataset to extract data from.}

\item{bandname}{a character string, the Google Earth Engine dataset bandname to extract data for.}

\item{spatial.res.metres}{a numeric value, specifying the spatial resolution in metres of the
raster to be extracted.}

\item{GEE.math.fun}{a character string, the mathematical function to compute across the specified
time frame from each projection date and for each cell.}

\item{user.email}{a character string, user email for initialising Google Drive.}

\item{varname}{optional; a character string, the unique name for the explanatory variable. Default
varname is “bandname_temporal.res_temporal.direction_ GEE.math.fun_raster".}

\item{temporal.res}{a numeric value, the temporal resolution in days to extract data across.}

\item{temporal.direction}{a character string, the temporal direction for extracting dynamic
variable data across relative to each projection date given. One of \code{prior} or \code{post}: can
be abbreviated.}

\item{save.directory}{optional; a character string, path to local directory to save extracted
rasters to.}

\item{save.drive.folder}{optional; a character string, Google Drive folder name to save extracted
rasters to. Folder must be uniquely named within your Google Drive. Do not provide path.}

\item{resume}{a logical indicating whether to search \code{save.directory} or \code{save.drive.folder} and
return to previous progress through projection dates.Default = TRUE.}
}
\value{
Returns details of successful explanatory variable extractions for each projection date.
}
\description{
Extract rasters for temporally dynamic explanatory variables at each projection date using Google
Earth Engine.
}
\details{
For each projection date, this function downloads rasters at a given spatial extent and
resolution for temporally dynamic explanatory variables. For each cell within the spatial
extent, the \code{GEE.math.fun} is calculated on the data extracted from across the specified number
of days prior or post the projection date. Rasters can be saved to Google Drive or a local
directory too. These rasters can be combined to create projection covariate data frames for
projecting dynamic species distribution and abundance at high spatiotemporal resolution.
}
\section{Google Earth Engine}{
\code{extract_dynamic_raster()} requires users to have installed the R package \code{rgee} (Aybar et al.,
2020) and initialised Google Earth Engine with valid log-in credentials. Please follow
instructions on the following website \url{https://cran.r-project.org/package=rgee}.
\itemize{
\item \code{datasetname} must be in the accepted Google Earth Engine catalogue layout (e.g.
“MODIS/006/MCD12Q1” or “UCSB-CHG/CHIRPS/DAILY”)
\item \code{bandname} must be as specified under the dataset in the Google Earth Engine catalogue (e.g.
“LC_Type5”, “precipitation”). For datasets and band names, see
\url{https://developers.google.com/earth-engine/datasets}.
}
}

\section{Google Drive}{
\code{extract_dynamic_raster()} also requires users to have installed the R package
\code{googledrive}(D'Agostino McGowan and Bryan, 2022) and initialised Google Drive with valid log-in
credentials, which must be stated using argument \code{user.email}. Please follow instructions on
\url{https://googledrive.tidyverse.org/} for initialising the \code{googledrive} package.

The \code{save.drive.folder} must be uniquely named within your Google Drive and do not provide the
path.

Note: When running this function a folder labelled "dynamicSDM_download_bucket" will be created in
your Google Drive. This will be emptied once the function has finished running and output rasters
will be found in the \code{save.drive.folder} or \code{save.directory}.
}

\section{Mathematical function}{
\code{GEE.math.fun} specifies the mathematical function to be calculated over the temporal period
from each projection date. Options are limited to Google Earth Engine ImageCollection Reducer
functions (\url{https://developers.google.com/earth-engine/apidocs/}) for which an analogous R
function is available. This includes: "allNonZero","anyNonZero", "count",
"first","firstNonNull", "last", "lastNonNull", "max","mean", "median","min", "mode","product",
"sampleStdDev", "sampleVariance", "stdDev", "sum" and "variance".
}

\section{Categorical data}{
Please be aware, at current this function does not support the extraction of temporally dynamic
variables for specific categories within categorical datasets.

When extracting from categorical datasets, be careful to choose appropriate mathematical
functions for such data. For instance, "first" or "last" may be more relevant that "sum" of land
cover classification numbers.
}

\section{Occasional rgee errors}{
As this function uses the \code{rgee} package to extract rasters from Google Earth Engine, below we
outline occasional \code{rgee} errors that may occur when extracting rasters:
\itemize{
\item "To avoid memory excess problems, ee_as_raster will not build Raster objects for large images"
}

This can be a sporadic error. It may be related to GEE server usage or internet
connection at the time you tested the function. Try restarting your R session or try again at
another time. Also, try clearing the files from the "dynamicSDM_download_bucket" in your Google
Drive.

This error could also be due to an issue with your input \code{spatial.res.metres}. This
function will extract rasters at all typical spatial resolutions of remote-sensing data and at
global extents. If this error persists, please ensure you have not accidentally given an
unrealistically high spatial resolution (e.g. \code{spatial.res.metres = 0.01} when you may be
confusing the spatial resolution in degrees with metres).
\itemize{
\item "Pixel type not supported: Type Long. Convert the image to a floating point type or a smaller
integer type, for example, using ee.Image.toDouble()"
}

This error appears when \code{rgee} has been given an input that cannot be extracted. Some common
causes include:
\itemize{
\item Inappropriate \code{GEE.math.fun} for extracting categorical data.
\item Dates or spatial extents that are not covered by the chosen GEE dataset. Remember to check
whether the first projection date minus \code{temporal.res} is still within the temporal extent of the
dataset.
}
}

\examples{
\dontshow{if (googledrive::drive_has_token()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
dates <- dynamic_proj_dates("2018-01-01", "2018-12-01", interval = 3,interval.level = "month")
\dontshow{
dates <- dates[1]
}

data("sample_extent_data")
user.email<-as.character(gargle::gargle_oauth_sitrep()$email)

extract_dynamic_raster(dates = dates,
                      datasetname = "UCSB-CHG/CHIRPS/DAILY",
                      bandname="precipitation",
                      user.email = user.email,
                      spatial.res.metres = 5566,
                      GEE.math.fun = "sum",
                      temporal.direction = "prior",
                      temporal.res = 56,
                      spatial.ext = sample_extent_data,
                      varname = "total_annual_precipitation_prior",
                      save.directory = tempdir())



\dontshow{\}) # examplesIf}
}
\references{
Aybar, C., Wu, Q., Bautista, L., Yali, R. and Barja, A., 2020. rgee: An R package for
interacting with Google Earth Engine. Journal of Open Source Software, 5(51), p.2272.

D'Agostino McGowan L., and Bryan J., 2022. googledrive: An Interface to Google Drive.
https://googledrive.tidyverse.org, https://github.com/tidyverse/googledrive.
}
