% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/case-match.R
\name{case_match}
\alias{case_match}
\title{A general vectorised \code{switch()}}
\usage{
case_match(.x, ..., .default = NULL, .ptype = NULL)
}
\arguments{
\item{.x}{A vector to match against.}

\item{...}{<\code{\link[rlang:dyn-dots]{dynamic-dots}}> A sequence of two-sided
formulas: \code{old_values ~ new_value}. The right hand side (RHS) determines
the output value for all values of \code{.x} that match the left hand side
(LHS).

The LHS must evaluate to the same type of vector as \code{.x}. It can be any
length, allowing you to map multiple \code{.x} values to the same RHS value.
If a value is repeated in the LHS, i.e. a value in \code{.x} matches to
multiple cases, the first match is used.

The RHS inputs will be coerced to their common type. Each RHS input will be
\link[vctrs:vector_recycling_rules]{recycled} to the size of \code{.x}.}

\item{.default}{The value used when values in \code{.x} aren't matched by any of
the LHS inputs. If \code{NULL}, the default, a missing value will be used.

\code{.default} is \link[vctrs:vector_recycling_rules]{recycled} to the size of
\code{.x}.}

\item{.ptype}{An optional prototype declaring the desired output type. If
not supplied, the output type will be taken from the common type of
all RHS inputs and \code{.default}.}
}
\value{
A vector with the same size as \code{.x} and the same type as the common type of
the RHS inputs and \code{.default} (if not overridden by \code{.ptype}).
}
\description{
This function allows you to vectorise multiple \code{\link[=switch]{switch()}} statements. Each
case is evaluated sequentially and the first match for each element
determines the corresponding value in the output vector. If no cases match,
the \code{.default} is used.

\code{case_match()} is an R equivalent of the SQL "simple" \verb{CASE WHEN} statement.
\subsection{Connection to \code{case_when()}}{

While \code{\link[=case_when]{case_when()}} uses logical expressions on the left-hand side of the
formula, \code{case_match()} uses values to match against \code{.x} with. The following
two statements are roughly equivalent:

\if{html}{\out{<div class="sourceCode">}}\preformatted{case_when(
  x \%in\% c("a", "b") ~ 1,
  x \%in\% "c" ~ 2,
  x \%in\% c("d", "e") ~ 3
)

case_match(
  x,
  c("a", "b") ~ 1,
  "c" ~ 2,
  c("d", "e") ~ 3
)
}\if{html}{\out{</div>}}
}
}
\examples{
x <- c("a", "b", "a", "d", "b", NA, "c", "e")

# `case_match()` acts like a vectorized `switch()`.
# Unmatched values "fall through" as a missing value.
case_match(
  x,
  "a" ~ 1,
  "b" ~ 2,
  "c" ~ 3,
  "d" ~ 4
)

# Missing values can be matched exactly, and `.default` can be used to
# control the value used for unmatched values of `.x`
case_match(
  x,
  "a" ~ 1,
  "b" ~ 2,
  "c" ~ 3,
  "d" ~ 4,
  NA ~ 0,
  .default = 100
)

# Input values can be grouped into the same expression to map them to the
# same output value
case_match(
  x,
  c("a", "b") ~ "low",
  c("c", "d", "e") ~ "high"
)

# `case_match()` isn't limited to character input:
y <- c(1, 2, 1, 3, 1, NA, 2, 4)

case_match(
  y,
  c(1, 3) ~ "odd",
  c(2, 4) ~ "even",
  .default = "missing"
)

# Setting `.default` to the original vector is a useful way to replace
# selected values, leaving everything else as is
case_match(y, NA ~ 0, .default = y)

starwars \%>\%
  mutate(
    # Replace missings, but leave everything else alone
    hair_color = case_match(hair_color, NA ~ "unknown", .default = hair_color),
    # Replace some, but not all, of the species
    species = case_match(
      species,
      "Human" ~ "Humanoid",
      "Droid" ~ "Robot",
      c("Wookiee", "Ewok") ~ "Hairy",
      .default = species
    ),
    .keep = "used"
  )
}
\seealso{
\code{\link[=case_when]{case_when()}}
}
