% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/welfareDecisionAnalysis.R
\name{welfareDecisionAnalysis}
\alias{welfareDecisionAnalysis}
\title{Analysis of the underlying welfare based decision problem.}
\usage{
welfareDecisionAnalysis(
  estimate,
  welfare,
  numberOfModelRuns,
  randomMethod = "calculate",
  functionSyntax = "data.frameNames",
  relativeTolerance = 0.05,
  verbosity = 0
)
}
\arguments{
\item{estimate}{\code{\link{estimate}} object describing the distribution of the input variables.}

\item{welfare}{either a \code{function} or a \code{list} with two \code{functions}, i.e.
\code{list(p1,p2)}. In the first case the function is the net benefit (or welfare) of project approval (PA) vs.
the status quo (SQ). In the second case the element \code{p1} is the function valuing the first
project and the element \code{p2} valuing the second project, viz. the welfare function of \code{p1}
and \code{p2} respectively.}

\item{numberOfModelRuns}{\code{integer}: The number of running the welfare model for the 
underlying Monte Carlo simulation.}

\item{randomMethod}{\code{character}: The method to be used to sample the distribution
representing the input estimate. For details see option \code{method} in 
\code{\link{random.estimate}}.}

\item{functionSyntax}{\code{character}: function syntax used in the welfare function(s). For 
details see \code{\link{mcSimulation}}.}

\item{relativeTolerance}{\code{numeric}: the relative tolerance level of deviation of the
generated confidence interval from the specified interval. If this deviation is greater than
\code{relativeTolerance} a warning is given.}

\item{verbosity}{\code{integer}: if \code{0} the function is silent; the larger the value the
more verbose is output information.}
}
\value{
An object of class \code{welfareDecisionAnalysis} with the following elements:
 \describe{
     \item{\code{$mcResult}}{The results of the Monte Carlo analysis of \code{estimate} 
     transformed by \code{welfare}} (cf. \code{\link{mcSimulation}}).
     \item{\code{$enbPa}}{Expected Net Benefit of project approval: ENB(PA)}
			\item{\code{$elPa}}{Expected Loss in case of project approval: EL(PA)}
			\item{\code{$elSq}}{Expected Loss in case of status quo: EL(SQ)}
 		\item{\code{$eol}}{Expected Opportunity Loss: EOL}
 		\item{\code{$optimalChoice}}{
 		    The optimal choice, i.e. either project approval (PA) or the status quo (SQ).
 		    }
}
}
\description{
The optimal choice between two different opportunities is calculated. Optimality is taken as 
maximizing expected welfare. Furthermore, the Expected Opportunity Loss (EOL) is calculated.
}
\details{
\subsection{The underlying decision problem and its notational framework}{ 
  We are considering a
  decision maker who can influence an ecological-economic system having two alternative decisions
  \eqn{d_1} and \eqn{d_2} at hand. We assume, that the system can be characterized by the 
  \eqn{n-}dimensional
  vector \eqn{X}. The characteristics \eqn{X}, are not necessarily known exactly to the decision maker.
  However, we assume furthermore that she is able to quantify this uncertainty which we call an
  \emph{\link{estimate}} of the characteristics. Mathematically, an estimate is a random variable with
  probability density \eqn{\rho_X}.
  
  Furthermore, the characteristics \eqn{X} determine the welfare \eqn{W(d)} according to the welfare
  function \eqn{w_d}: 
  \deqn{ 
      W_d = w_d (X) 
  }{
    W(d) = w_d (X) 
  } 
  Thus, the welfare of decision \eqn{d} is also a random
  variable whose probability distribution we call \eqn{\rho_{W_d}}{rho(W(d))}. The welfare function \eqn{w_d} values
  the decision \eqn{d} given a certain state \eqn{X} of the system. In other words, decision \eqn{d_2} is
  preferred over decision \eqn{d_1}, if and only if, the expected welfare of decision \eqn{d_2} is
  greater than the expected welfare (For a comprehensive
  discussion of the concept of social preference ordering and its representation by a welfare
  function cf. Gravelle and Rees (2004)) of decsion \eqn{d_1}, formally 
  \deqn{
    d_1 \prec d_2 \Leftrightarrow \textrm{E}[W_{d_1}] < \textrm{E}[W_{d_2}].
   }{
     d_1 < d_2 <==> E[W(d_1)] < E[W(d_2)].
   }
  This means the best decision \eqn{d^*}{d*} is the one which maximizes welfare: 
  \deqn{ 
    d^* := \arg \max_{d=d_1,d_2} \textrm{E}[W_d]
  }{
    d* := arg max (d=d_1,d_2) E[W(d)] 
  }
  
  This maximization principle has a dual minimization principle. We define the net benefit
  \eqn{\textrm{NB}_{d_1} := W_{d_1} - W_{d_2}}{NB(d_1) := W(d_1) - W(d_2)} as the difference 
  between the welfare of the two decision
  alternatives. A loss \eqn{L_d} is characterized if a decision \eqn{d} produces a negative net benefit.
  No loss occurs if the decision produces a positive net benefit. This is reflected in the formal
  definition 
  \deqn{
      L_d := - \textrm{NB}_d, \textrm{~if~} \textrm{NB}_d  < 0, \textrm{~and~} L_d := 0  
         \textrm{~otherwise}.
  }{
     L(d) :=  - NB(d) if NB(d)  < 0 and L(d) := 0 otherwise.
  }
  Using this notion it can be shown that the maximization of
  expected welfare is equivalent to the minimization of the expected loss 
  \eqn{\textrm{EL}_d := \textrm{E}[L_d]}{EL(d) := E[L(d)]}. 
  \subsection{The Expected Opportunity Loss (EOL)}{
    The use of this concept, here, is in line as described in Hubbard (2014). The Expected
    Opportunity Loss (\eqn{\textrm{EOL}}{EOL}) is defined as the expected loss for the best
    decision. The best decision minimizes the expected loss:
    \deqn{
      \textrm{EOL} := \min \left\{ \textrm{EL}_{d_1}, \textrm{EL}_{d_2}\right\}
     }{
      EOL := min \{ EL(d_1), EL(d_2) \}
     }
    
    The \eqn{\textrm{EOL}}{EOL} is always conditional on the available information which is
    characterized by the probability distribution of \eqn{X}
    \eqn{\rho_X}: \eqn{\textrm{EOL} = \textrm{EOL}(\rho_X)}{EOL = EOL(\rho_X)}.
  }
  \subsection{Special case: Status quo and project approval}{
    A very common actual binary decision problem is the question if a certain project shall be 
    approved versus continuing with business as usual, i.e. the status quo. It appears to be 
    natural to identify the status quo with zero welfare. This is a special case ( Actually, one
    can show, that this special case is equivalent to the discussion above.) of the binary
    decision problem that we are considering here. The two decision alternatives are
    \describe{
      \item{\eqn{d_1:}}{ project approval (PA) and }
      \item{\eqn{d_2:}}{ status quo (SQ),}
    }
    and the welfare of the approved project (or project outcome or yield of the project) is the
    random variable \eqn{W_\textrm{PA}}{W(PA)} with distribution 
    \eqn{P_{W_\textrm{PA}} = w_\textrm{PA}(P_X)}{P_W(PA) = w_PA(P_X)}:
      \deqn{
        W_\textrm{PA} \sim w_\textrm{PA}(P_X) = P_{W_\textrm{PA}}
       }{
       W(PA) ~ w_PA(P_X) = P_W(PA) 
       }
    and the welfare of the status quo serves as reference and is normalized to zero:
    \deqn{
      W_\textrm{SQ} \equiv 0,
    }{
      W(SQ) = 0
    }
    which implies zero expected welfare of the status quo:
    \deqn{
      \textrm{E}[W]_\textrm{SQ} 	= 0.
    }{
      E[W(SQ)] = 0.
    }
  }
  }
}
\examples{
#############################################################
# Example 1 (Creating the estimate from the command line):
#############################################################
# Create the estimate object:
variable=c("revenue","costs")
distribution=c("posnorm","posnorm")
lower=c(10000,  5000)
upper=c(100000, 50000)
costBenefitEstimate<-as.estimate(variable, distribution, lower, upper)
# (a) Define the welfare function without name for the return value:
profit<-function(x){
 x$revenue-x$costs
}
# Perform the decision analysis:
myAnalysis<-welfareDecisionAnalysis(estimate=costBenefitEstimate, 
                                    welfare=profit, 
                                    numberOfModelRuns=100000,
                                    functionSyntax="data.frameNames")
# Show the analysis results:
print(summary((myAnalysis)))
#############################################################
# (b) Define the welfare function with a name for the return value:
profit<-function(x){
 list(Profit=x$revenue-x$costs)
}
# Perform the decision analysis:
myAnalysis<-welfareDecisionAnalysis(estimate=costBenefitEstimate, 
                                    welfare=profit, 
                                    numberOfModelRuns=100000,
                                    functionSyntax="data.frameNames")
# Show the analysis results:
print(summary((myAnalysis)))
#############################################################
# (c) Two decsion variables:
welfareModel<-function(x){
 list(Profit=x$revenue-x$costs,
   Costs=-x$costs)
}
# Perform the decision analysis:
myAnalysis<-welfareDecisionAnalysis(estimate=costBenefitEstimate, 
                                    welfare=welfareModel, 
                                    numberOfModelRuns=100000,
                                    functionSyntax="data.frameNames")
# Show the analysis results:
print(summary((myAnalysis)))
}
\references{
Hubbard, Douglas W., \emph{How to Measure Anything? - Finding the Value of "Intangibles" in Business},
  John Wiley & Sons, Hoboken, New Jersey, 2014, 3rd Ed, \url{https://www.howtomeasureanything.com/}.
  
  Gravelle, Hugh and Ray Rees, \emph{Microeconomics}, Pearson Education Limited, 3rd edition, 2004.
}
\seealso{
\code{\link{mcSimulation}}, \code{\link{estimate}}, \code{\link{summary.welfareDecisionAnalysis}}
}
