#' Retrieve the list of available blueprints for a project
#'
#' This function returns the list of available blueprints
#' for a specified modeling project, as an S3 object of class
#' listOfBlueprints; see Value.
#'
#' @inheritParams DeleteProject
#' @return An S3 object of class 'listOfBlueprints', a list
#' with one element for each recommended blueprint in the
#' associated project. For more information see GetBlueprint()
#' @export
#'
ListBlueprints <- function(project) {
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "blueprints")
  blueprints <- DataRobotGET(routeString, addUrl = TRUE)
  idIndex <- which(names(blueprints) == "id")
  names(blueprints)[idIndex] <- "blueprintId"
  n <- length(blueprints$blueprintId)
  blueNames <- names(blueprints)
  m <- length(blueNames)
  blueprintList <- vector("list", n)
  element <- vector("list", m)
  for (i in 1:n) {
    for (j in 1:m) {
      element[[j]] <- blueprints[[j]][[i]]
    }
    names(element) <- blueNames
    blueprintList[[i]] <- element
  }
  blueprintList <- lapply(blueprintList, as.dataRobotBlueprint)
  class(blueprintList) <- c('listOfBlueprints', 'listSubclass')
  return(blueprintList)
}

#' Retrieve the list of available blueprints for a project
#'
#' (Deprecated in 2.3, will be removed in 3.0. Use ListBlueprints instead.)
#'
#' @inheritParams ListBlueprints
#'
#' @export

GetRecommendedBlueprints <- function(project) {
  Deprecated("GetRecommendedBlueprints (use ListBlueprints instead)", "2.3", "3.0")
  return(ListBlueprints(project))
}



as.dataRobotBlueprint <- function(inList){
  elements <- c("projectId",
                "processes",
                "blueprintId",
                "modelType",
                "blueprintCategory")
  return(ApplySchema(inList, elements))
}


#' Retrieve a blueprint
#'
#' @inheritParams DeleteProject
#' @param blueprintId Character strint Id of blueprint to retrieve
#' @return List with the following four components:
#' \describe{
#'   \item{projectId}{Character string giving the unique DataRobot project identifier}
#'   \item{processes}{List of character strings, identifying any preprocessing steps included in the
#'   blueprint}
#'   \item{blueprintId}{Character string giving the unique DataRobot blueprint identifier}
#'   \item{modelType}{Character string, specifying the type of model the blueprint builds}
#'   \item{blueprintCategory}{Character string. Describes the category of the blueprint 
#'   and the kind of model it produces.}
#' }
#' @export
#'
GetBlueprint <- function(project, blueprintId){
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "blueprints", blueprintId)
  blueprint <- DataRobotGET(routeString, addUrl = TRUE,
               simplifyDataFrame = FALSE)
  idIndex <- which(names(blueprint) == "id")
  names(blueprint)[idIndex] <- "blueprintId"
  return(as.dataRobotBlueprint(blueprint))
}

#' Retrieve a blueprint chart
#' 
#' A Blueprint chart can be used to understand data flow in blueprint.
#'
#' @inheritParams DeleteProject
#' @param blueprintId Character strint Id of blueprint to retrieve
#' @return List with the following two components:
#' \describe{
#'   \item{nodes}{list each element contains information about one node of a blueprint : id and label}
#'   \item{edges}{Two column matrix, identifying blueprint nodes connections}
#' }
#' @export
#'
GetBlueprintChart <- function(project, blueprintId){
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "blueprints", blueprintId, "blueprintChart")
  return(as.dataRobotBlueprintChart(DataRobotGET(routeString, addUrl = TRUE,
                                              simplifyDataFrame = FALSE)))
}

as.dataRobotBlueprintChart <- function(inList){
  elements <- c("nodes",
                "edges")
  return(ApplySchema(inList, elements))
}


#' Convert a blueprint chart into graphviz DOT format
#'
#' @param blueprintChart List, returned by GetBlueprintChart function
#' @return Character string representation of chart in graphviz DOT language
#' @export
#'
BlueprintChartToGraphviz <- function(blueprintChart){
  digraph <-  'digraph "Blueprint Chart" {'
  digraph <- paste(digraph, 'graph [rankdir=LR]', sep = "\n")
  for (node in blueprintChart$nodes){
    addLine <- paste(node$id,  ' [label="', node$label, '"]', sep = "")
    digraph <- paste(digraph, addLine, sep = "\n")
}
  for (edgeN in seq(nrow(blueprintChart$edges))){
    addLine <- paste(blueprintChart$edges[edgeN, 1], ' -> ',
                     blueprintChart$edges[edgeN, 2], sep = "")
    digraph <- paste(digraph, addLine, sep = "\n")
  }
  digraph <- paste(digraph, '\n}')
  return(digraph)
}

#' Get documentation for tasks used in the blueprint
#'
#' @inheritParams DeleteProject
#' @param blueprintId Character strint Id of blueprint to retrieve
#' @return list with following components
#' \describe{
#'   \item{task}{Character string name of the task described in document}
#'   \item{description}{Character string task description}
#'   \item{title}{Character string title of document}
#'   \item{parameters}{List of parameters that task can received in human-readable
#'   format with following components: name, type, description}
#'   \item{links}{List of exteranl lins used in document with following components: name, url}
#'   \item{references}{List of references used in document with following components: name, url}
#' }
#' @export
#'
GetBlueprintDocuments <- function(project, blueprintId){
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "blueprints", blueprintId, "blueprintDocs")
  docs <- DataRobotGET(routeString, addUrl = TRUE,
                       simplifyDataFrame = FALSE)
  return(as.dataRobotBlueprintDocuments(docs[[1]]))
}

as.dataRobotBlueprintDocuments <- function(inList){
  elements <- c("title",
                "task",
                "description",
                "parameters",
                "links",
                "references"
                )
  return(ApplySchema(inList, elements))
}
