adaptsim <- function(f,a,b,tol=.Machine$double.eps,trace=FALSE,...) {
#adaptsim  Numerically evaluate integral using adaptive
#   Simpson rule.
#
#   adaptsim(f,a,b) approximates the integral of
#   f(x) from A to B to machine precision. The
#   function f must return a vector of output values if
#   given a vector of input values.
#
#   adaptsim(f,a,b,tol) integrates to a relative
#   error of TOL.
#
#   adaptsim(f,a,b,tol,trace) displays the left
#   end point of the current interval, the interval
#   length, and the partial integral.
#
#   adaptsim(f,a,b,tol,trace,p1,p2,...) allows
#   coefficients p1, ... to be passed directly to the
#   function f:  G <- f(x,p1,p2,...). To use default values
#   for tol or trace, one may pass the empty matrix ([]).
#
#   See also adaptsimstp.
#
#   Walter Gander, 08/03/98
#   Reference: Gander, Computermathematik, Birkhaeuser, 1992.

  assign("termination2", FALSE, env = .GlobalEnv)
  tol <- max(tol, .Machine$double.eps)
  x <- c(a, (a+b)/2, b);
  y <- f(x, ...);
  fa <- y[1]; fm <- y[2]; fb <- y[3];
  yy <- f(a+c(0.9501, 0.2311, 0.6068, 0.4860, 0.8913)*(b-a), ...)
  is <- (b - a)/8*(sum(y)+sum(yy));
  if (is == 0) is <- b-a
  is <- is*tol/.Machine$double.eps;
  Q <- adaptsimstp(f,a,b,fa,fm,fb,is,trace,...);
  attr(Q,"termination2") <- termination2
  Q
}


adaptsimstp <- function(f,a,b,fa,fm,fb,is,trace,...) {
#adaptsimstp  Recursive function used by adaptsim.
#
#   q <- adaptsimstp('f',a,b,fa,fm,fb,is,trace) tries to
#   approximate the integral of f(x) from A to B to
#   an appropriate relative error. The argument 'f' is
#   a string containing the name of f. The remaining
#   arguments are generated by adaptsim or by recursion.
#
#   See also adaptsim.
#
#   Walter Gander, 08/03/98

  m <- (a + b)/2; h <- (b - a)/4;
  x <- c(a + h, b - h);
  y <- f(x, ...);
  fml <- y[1]; fmr <- y[2];
  i1 <- h/1.5 * (fa + 4*fm + fb);
  i2 <- h/3 * (fa + 4*(fml + fmr) + 2*fm + fb);
  i1 <- (16*i2 - i1)/15;
  if ((is + (i1-i2) == is) | (m <= a) | (b <= m)) {
    if (((m <= a) | (b <= m)) & (!eval(termination2))) {
       warning('Interval contains no more machine number. \nRequired tolerance may not be met.');
       assign(termination2, TRUE, env = .GlobalEnv)
    }
    Q <- i1;
    if (trace) cat(a, b-a, Q,"\n");
    }
  else  Q <- adaptsimstp (f,a,m,fa,fml,fm,is,trace,...) +
        adaptsimstp (f,m,b,fm,fmr,fb,is,trace,...);
  attr(Q,"termination2") <- termination2;
  Q
}


adaptlob <- function (f,a,b,tol=.Machine$double.eps,trace=FALSE,...) {

#   See also adaptlobstp.

#   Walter Gautschi, 08/03/98
#   Reference: Gander, Computermathematik, Birkhaeuser, 1992.

  assign("termination2", FALSE, env = .GlobalEnv)
  tol <- max(tol, .Machine$double.eps)
  m <- (a+b)/2; h <- (b-a)/2;
  alpha <- sqrt(2/3); beta <- 1/sqrt(5);
  x1 <- .942882415695480; x2 <- .641853342345781;
  x3 <- .236383199662150;
  x <- c(a, m-x1*h, m-alpha*h, m-x2*h, m-beta*h, m-x3*h, m,
         m+x3*h, m+beta*h, m+x2*h, m+alpha*h, m+x1*h, b);
  y <- f(x, ...);
  fa <- y[1]; fb <- y[13];
  i2 <- (h/6)*(y[1]+y[13]+5*(y[5]+y[9]));
  i1 <- (h/1470)*(77*(y[1]+y[13])+432*(y[3]+y[11])+625*(y[5]+y[9])+672*y[7]);
  is <- h*(.0158271919734802*(y[1]+y[13])+.0942738402188500
     *(y[2]+y[12])+.155071987336585*(y[3]+y[11])+
     .188821573960182*(y[4]+y[10])+.199773405226859
     *(y[5]+y[9])+.224926465333340*(y[6]+y[8])
     +.242611071901408*y[7]);
  s <- sign(is)
  if (is == 0) is <- 1
  erri1 <- abs(i1-is);
  erri2 <- abs(i2-is);
  R <- 1; if (erri2 != 0) R <- erri1/erri2
  if ((R>0 & R<1)) tol <- tol/R;
  is <- s*abs(is)*tol/.Machine$double.eps;
  if(is == 0) is <- b-a;
  Q <- adaptlobstp(f,a,b,fa,fb,is,trace,...);
  attr(Q,"termination2") <- termination2
  Q
}

adaptlobstp <- function(f,a,b,fa,fb,is,trace,...) {
#adaptlobstp  recursive function used by adaptlob.
#
#   Q  <-  adaptlobstp('f',a,b,fa,fb,is,trace) tries to
#   approximate the integral of f(x) from A to B to
#   an appropriate relative error. The remaining
#   arguments are generated by adaptlob or by recursion.
#
#   See also ADAPTLOB.

#   Walter Gautschi, 08/03/98

  h <- (b-a)/2; m <- (a+b)/2;
  alpha <- sqrt(2/3); beta <- 1/sqrt(5);
  mll <- m-alpha*h; ml <- m-beta*h; mr <- m+beta*h; mrr <- m+alpha*h;
  x <- c(mll,ml,m,mr,mrr);
  y <- f(x, ...);
  fmll <- y[1]; fml <- y[2]; fm <- y[3]; fmr <- y[4]; fmrr <- y[5];
  i2 <- (h/6)*(fa+fb+5*(fml+fmr));
  i1 <- (h/1470)*(77*(fa+fb)+432*(fmll+fmrr)+625*(fml+fmr)+672*fm);
  if ((is + (i1-i2) == is) | (m <= a) | (b <= m)) {
    if (((m <= a) | (b <= m)) & (!eval(termination2))) {
       warning('Interval contains no more machine number. \nRequired tolerance may not be met.');
       assign("termination2", TRUE, env = .GlobalEnv)
    }
    Q <- i1;
    if (trace) cat(a, b-a, Q,"\n");
    }
  else
    Q <- adaptlobstp(f,a,mll,fa,fmll,is,trace,...)+
      adaptlobstp(f,mll,ml,fmll,fml,is,trace,...)+
      adaptlobstp(f,ml,m,fml,fm,is,trace,...)+
      adaptlobstp(f,m,mr,fm,fmr,is,trace,...)+
      adaptlobstp(f,mr,mrr,fmr,fmrr,is,trace,...)+
      adaptlobstp(f,mrr,b,fmrr,fb,is,trace,...);
  attr(Q,"termination2") <- termination2;
  Q
}
