% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate.R
\name{evaluate}
\alias{evaluate}
\title{Evaluate your model's performance}
\usage{
evaluate(
  data,
  target_col,
  prediction_cols,
  type,
  id_col = NULL,
  id_method = "mean",
  apply_softmax = FALSE,
  cutoff = 0.5,
  positive = 2,
  metrics = list(),
  include_predictions = TRUE,
  parallel = FALSE,
  models = deprecated()
)
}
\arguments{
\item{data}{\code{data.frame} with predictions, targets and (optionally) an ID column.
 Can be grouped with \code{\link[dplyr]{group_by}}.

 \subsection{Multinomial}{
 When \code{`type`} is \code{"multinomial"}, the predictions can be passed in one of two formats.

 \subsection{Probabilities (Preferable)}{

 One column per class with the probability of that class.
 The columns should have the name of their class,
 as they are named in the target column. E.g.:

 \tabular{rrrrr}{
  \strong{class_1} \tab \strong{class_2} \tab
  \strong{class_3} \tab \strong{target}\cr
  0.269 \tab 0.528 \tab 0.203 \tab class_2\cr
  0.368 \tab 0.322 \tab 0.310 \tab class_3\cr
  0.375 \tab 0.371 \tab 0.254 \tab class_2\cr
  ... \tab ... \tab ... \tab ...}
 }
 \subsection{Classes}{

 A single column of type \code{character} with the predicted classes. E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  class_2 \tab class_2\cr
  class_1 \tab class_3\cr
  class_1 \tab class_2\cr
  ... \tab ...}

 }
 }
 \subsection{Binomial}{
 When \code{`type`} is \code{"binomial"}, the predictions can be passed in one of two formats.

 \subsection{Probabilities (Preferable)}{
 One column with the probability of class being
 the second class alphabetically
 (1 if classes are 0 and 1). E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  0.769 \tab 1\cr
  0.368 \tab 1\cr
  0.375 \tab 0\cr
  ... \tab ...}
 }

 \subsection{Classes}{

 A single column of type \code{character} with the predicted classes. E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  class_0 \tab class_1\cr
  class_1 \tab class_1\cr
  class_1 \tab class_0\cr
  ... \tab ...}
 }

 Note: The prediction column will be converted to the probability \code{0.0}
 for the first class alphabetically and \code{1.0} for
 the second class alphabetically.
 }
 \subsection{Gaussian}{
 When \code{`type`} is \code{"gaussian"}, the predictions should be passed as
 one column with the predicted values. E.g.:

 \tabular{rrrrr}{
  \strong{prediction} \tab \strong{target}\cr
  28.9 \tab 30.2\cr
  33.2 \tab 27.1\cr
  23.4 \tab 21.3\cr
  ... \tab ...}
 }}

\item{target_col}{Name of the column with the true classes/values in \code{`data`}.

 When \code{`type`} is \code{"multinomial"}, this column should contain the class names,
 not their indices.}

\item{prediction_cols}{Name(s) of column(s) with the predictions.

 Columns can be either numeric or character depending on which format is chosen.
 See \code{`data`} for the possible formats.}

\item{type}{Type of evaluation to perform:

 \code{"gaussian"} for regression (like linear regression).

 \code{"binomial"} for binary classification.

 \code{"multinomial"} for multiclass classification.}

\item{id_col}{Name of ID column to aggregate predictions by.

 N.B. Current methods assume that the target class/value is constant within the IDs.

 N.B. When aggregating by ID, some metrics may be disabled.}

\item{id_method}{Method to use when aggregating predictions by ID.
 Either \code{"mean"} or \code{"majority"}.

 When \code{`type`} is \code{gaussian}, only the \code{"mean"} method is available.

 \subsection{mean}{
 The average prediction (value or probability) is calculated per ID and evaluated.
 This method assumes that the target class/value is constant within the IDs.
 }
 \subsection{majority}{
 The most predicted class per ID is found and evaluated. In case of a tie,
 the winning classes share the probability (e.g. \code{P = 0.5} each when two majority classes).
 This method assumes that the target class/value is constant within the IDs.
 }}

\item{apply_softmax}{Whether to apply the softmax function to the
 prediction columns when \code{`type`} is \code{"multinomial"}.

 N.B. \strong{Multinomial models only}.}

\item{cutoff}{Threshold for predicted classes. (Numeric)

N.B. \strong{Binomial models only}.}

\item{positive}{Level from dependent variable to predict.
 Either as character (\emph{preferable}) or level index (\code{1} or \code{2} - alphabetically).

 E.g. if we have the levels \code{"cat"} and \code{"dog"} and we want \code{"dog"} to be the positive class,
 we can either provide \code{"dog"} or \code{2}, as alphabetically, \code{"dog"} comes after \code{"cat"}.

 \strong{Note:} For \emph{reproducibility}, it's preferable to \strong{specify the name directly}, as
 different \code{\link[base:locales]{locales}} may sort the levels differently.

 Used when calculating confusion matrix metrics and creating \code{ROC} curves.

 The \code{Positive Class} column in the output can be used to verify this setting.

 N.B. Only affects the evaluation metrics.

 N.B. \strong{Binomial models only}.}

\item{metrics}{\code{list} for enabling/disabling metrics.

  E.g. \code{list("RMSE" = FALSE)} would remove \code{RMSE} from the regression results,
  and \code{list("Accuracy" = TRUE)} would add the regular \code{Accuracy} metric
  to the classification results.
  Default values (\code{TRUE}/\code{FALSE}) will be used for the remaining available metrics.

  You can enable/disable all metrics at once by including
  \code{"all" = TRUE/FALSE} in the \code{list}. This is done prior to enabling/disabling
  individual metrics, why f.i. \code{list("all" = FALSE, "RMSE" = TRUE)}
  would return only the \code{RMSE} metric.

  The \code{list} can be created with
  \code{\link[cvms:gaussian_metrics]{gaussian_metrics()}},
  \code{\link[cvms:binomial_metrics]{binomial_metrics()}}, or
  \code{\link[cvms:multinomial_metrics]{multinomial_metrics()}}.

  Also accepts the string \code{"all"}.}

\item{include_predictions}{Whether to include the predictions
in the output as a nested \code{tibble}. (Logical)}

\item{parallel}{Whether to run evaluations in parallel,
when \code{`data`} is grouped with \code{\link[dplyr:group_by]{group_by}}.}

\item{models}{Deprecated.}
}
\value{
----------------------------------------------------------------

 \subsection{Gaussian Results}{

 ----------------------------------------------------------------

 \code{tibble} containing the following metrics by default:

 Average \strong{\code{RMSE}}, \strong{\code{MAE}}, \strong{\code{NRMSE(IQR)}},
 \strong{\code{RRSE}}, \strong{\code{RAE}}, \strong{\code{RMSLE}}.

 See the additional metrics (disabled by default) at
 \code{\link[cvms:gaussian_metrics]{?gaussian_metrics}}.

 Also includes:

 A nested \code{tibble} with the \strong{Predictions} and targets.

 A nested \strong{Process} information object with information
 about the evaluation.
 }

 ----------------------------------------------------------------

 \subsection{Binomial Results}{

 ----------------------------------------------------------------

 \code{tibble} with the following evaluation metrics, based on a
 \code{confusion matrix} and a \code{ROC} curve fitted to the predictions:

 \code{Confusion Matrix}:

 \strong{\code{Balanced Accuracy}},
 \strong{\code{Accuracy}},
 \strong{\code{F1}},
 \strong{\code{Sensitivity}},
 \strong{\code{Specificity}},
 \strong{\code{Positive Predictive Value}},
 \strong{\code{Negative Predictive Value}},
 \strong{\code{Kappa}},
 \strong{\code{Detection Rate}},
 \strong{\code{Detection Prevalence}},
 \strong{\code{Prevalence}}, and
 \strong{\code{MCC}} (Matthews correlation coefficient).

 \code{ROC}:

 \strong{\code{AUC}}, \strong{\code{Lower CI}}, and \strong{\code{Upper CI}}

 Note, that the \code{ROC} curve is only computed if \code{AUC} is enabled. See \code{metrics}.

 Also includes:

 A nested \code{tibble} with the \strong{predictions} and targets.

 A \code{list} of \strong{ROC} curve objects (if computed).

 A nested \code{tibble} with the \strong{confusion matrix}.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (\code{TP}), True Negative (\code{TN}),
 False Positive (\code{FP}), or False Negative (\code{FN}),
 depending on which level is the "\code{positive}" class.
 I.e. the level you wish to predict.

 A nested \strong{Process} information object with information
 about the evaluation.
 }

 ----------------------------------------------------------------

 \subsection{Multinomial Results}{

 ----------------------------------------------------------------

 For each class, a \emph{one-vs-all} binomial evaluation is performed. This creates
 a \strong{Class Level Results} \code{tibble} containing the same metrics as the binomial results
 described above (excluding \code{Accuracy}, \code{MCC}, \code{AUC}, \code{Lower CI} and \code{Upper CI}),
 along with a count of the class in the target column (\strong{\code{Support}}).
 These metrics are used to calculate the macro metrics. The nested class level results
 \code{tibble} is also included in the output \code{tibble},
 and could be reported along with the macro and overall metrics.

 The output \code{tibble} contains the macro and overall metrics.
 The metrics that share their name with the metrics in the nested
 class level results \code{tibble} are averages of those metrics
 (note: does not remove \code{NA}s before averaging).
 In addition to these, it also includes the \strong{\code{Overall Accuracy}} and
 the multiclass \strong{\code{MCC}}.

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{\code{Accuracy}},
 \emph{multiclass} \strong{\code{AUC}},
 \strong{\code{Weighted Balanced Accuracy}},
 \strong{\code{Weighted Accuracy}},
 \strong{\code{Weighted F1}},
 \strong{\code{Weighted Sensitivity}},
 \strong{\code{Weighted Sensitivity}},
 \strong{\code{Weighted Specificity}},
 \strong{\code{Weighted Pos Pred Value}},
 \strong{\code{Weighted Neg Pred Value}},
 \strong{\code{Weighted Kappa}},
 \strong{\code{Weighted Detection Rate}},
 \strong{\code{Weighted Detection Prevalence}}, and
 \strong{\code{Weighted Prevalence}}.

 Note that the "Weighted" average metrics are weighted by the \code{Support}.

 When having a large set of classes, consider keeping \code{AUC} disabled.

 Also includes:

 A nested \code{tibble} with the \strong{Predictions} and targets.

 A \code{list} of \strong{ROC} curve objects when \code{AUC} is enabled.

 A nested \code{tibble} with the multiclass \strong{Confusion Matrix}.

 A nested \strong{Process} information object with information
 about the evaluation.

 \subsection{Class Level Results}{

 Besides the binomial evaluation metrics and the \code{Support},
 the nested class level results \code{tibble} also contains a
 nested \code{tibble} with the \strong{Confusion Matrix} from the one-vs-all evaluation.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (\code{TP}), True Negative (\code{TN}),
 False Positive (\code{FP}), or False Negative (\code{FN}),
 depending on which level is the "positive" class. In our case, \code{1} is the current class
 and \code{0} represents all the other classes together.
 }
 }
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("maturing")}

 Evaluate your model's predictions
 on a set of evaluation metrics.

 Create ID-aggregated evaluations by multiple methods.

 Currently supports regression and classification
 (binary and multiclass). See \code{`type`}.
}
\details{
Packages used:

 \strong{Binomial} and \strong{Multinomial}:

 \code{ROC} and \code{AUC}:

 Binomial: \code{\link[pROC:roc]{pROC::roc}}

 Multinomial: \code{\link[pROC:multiclass.roc]{pROC::multiclass.roc}}
}
\examples{
\donttest{
# Attach packages
library(cvms)
library(dplyr)

# Load data
data <- participant.scores

# Fit models
gaussian_model <- lm(age ~ diagnosis, data = data)
binomial_model <- glm(diagnosis ~ score, data = data)

# Add predictions
data[["gaussian_predictions"]] <- predict(gaussian_model, data,
  type = "response",
  allow.new.levels = TRUE
)
data[["binomial_predictions"]] <- predict(binomial_model, data,
  allow.new.levels = TRUE
)

# Gaussian evaluation
evaluate(
  data = data, target_col = "age",
  prediction_cols = "gaussian_predictions",
  type = "gaussian"
)

# Binomial evaluation
evaluate(
  data = data, target_col = "diagnosis",
  prediction_cols = "binomial_predictions",
  type = "binomial"
)

#
# Multinomial
#

# Create a tibble with predicted probabilities and targets
data_mc <- multiclass_probability_tibble(
  num_classes = 3, num_observations = 45,
  apply_softmax = TRUE, FUN = runif,
  class_name = "class_",
  add_targets = TRUE
)

class_names <- paste0("class_", 1:3)

# Multinomial evaluation
evaluate(
  data = data_mc, target_col = "Target",
  prediction_cols = class_names,
  type = "multinomial"
)

#
# ID evaluation
#

# Gaussian ID evaluation
# Note that 'age' is the same for all observations
# of a participant
evaluate(
  data = data, target_col = "age",
  prediction_cols = "gaussian_predictions",
  id_col = "participant",
  type = "gaussian"
)

# Binomial ID evaluation
evaluate(
  data = data, target_col = "diagnosis",
  prediction_cols = "binomial_predictions",
  id_col = "participant",
  id_method = "mean", # alternatively: "majority"
  type = "binomial"
)

# Multinomial ID evaluation

# Add IDs and new targets (must be constant within IDs)
data_mc[["Target"]] <- NULL
data_mc[["ID"]] <- rep(1:9, each = 5)
id_classes <- tibble::tibble(
  "ID" = 1:9,
  "Target" = sample(x = class_names, size = 9, replace = TRUE)
)
data_mc <- data_mc \%>\%
  dplyr::left_join(id_classes, by = "ID")

# Perform ID evaluation
evaluate(
  data = data_mc, target_col = "Target",
  prediction_cols = class_names,
  id_col = "ID",
  id_method = "mean", # alternatively: "majority"
  type = "multinomial"
)

#
# Training and evaluating a multinomial model with nnet
#

# Create a data frame with some predictors and a target column
class_names <- paste0("class_", 1:4)
data_for_nnet <- multiclass_probability_tibble(
  num_classes = 3, # Here, number of predictors
  num_observations = 30,
  apply_softmax = FALSE,
  FUN = rnorm,
  class_name = "predictor_"
) \%>\%
  dplyr::mutate(Target = sample(
    class_names,
    size = 30,
    replace = TRUE
  ))

# Train multinomial model using the nnet package
mn_model <- nnet::multinom(
  "Target ~ predictor_1 + predictor_2 + predictor_3",
  data = data_for_nnet
)

# Predict the targets in the dataset
# (we would usually use a test set instead)
predictions <- predict(
  mn_model,
  data_for_nnet,
  type = "probs"
) \%>\%
  dplyr::as_tibble()

# Add the targets
predictions[["Target"]] <- data_for_nnet[["Target"]]

# Evaluate predictions
evaluate(
  data = predictions,
  target_col = "Target",
  prediction_cols = class_names,
  type = "multinomial"
)
}
}
\seealso{
Other evaluation functions: 
\code{\link{binomial_metrics}()},
\code{\link{confusion_matrix}()},
\code{\link{evaluate_residuals}()},
\code{\link{gaussian_metrics}()},
\code{\link{multinomial_metrics}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{evaluation functions}
